package cn.com.duiba.mq;

import cn.com.duiba.boot.profiler.DBTimeProfiler;
import cn.com.duiba.thirdparty.enums.DelayMsgConfigNameEnum;
import cn.com.duiba.wolf.utils.GZIPUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.ttl.threadpool.TtlExecutors;
import org.apache.commons.lang.StringUtils;
import org.apache.rocketmq.client.producer.DefaultMQProducer;
import org.apache.rocketmq.client.producer.SendResult;
import org.apache.rocketmq.client.producer.SendStatus;
import org.apache.rocketmq.common.message.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * Created by zzy on 2017/7/17.
 */
@Service
public class RocketMQMsgProducer {
    private static final Logger LOG = LoggerFactory.getLogger(RocketMQMsgProducer.class);
    @Autowired
    private DefaultMQProducer rocketMqProducer;
    private static final int MAX_RETRY_TIMES = 3;

    @Autowired
    private ExecutorService executorService;

    // 专门用来发消息和重试的线程池
    private final ExecutorService mqRetryExecutor = TtlExecutors.getTtlExecutorService(new ThreadPoolExecutor(4, 8, 10L, TimeUnit.SECONDS, new LinkedBlockingQueue<>(10000), new ThreadFactory() {
        private int index = 0;

        @Override
        public Thread newThread(Runnable r) {
            return new Thread(r, "rocketmq-producer-retry" + index++);
        }
    }));

    /**
     * 发送消息rocketmq
     *
     * @param topic   队列名单
     * @param tag     标签
     * @param key     消息key
     * @param message 消息
     * @return
     * @see <a>http://cf.dui88.com/pages/viewpage.action?pageId=5256828</a>
     */
    @DBTimeProfiler
    public void sendMsg(String topic, String tag, String key, String message, boolean useGzip, MqSentCallback callback) {
        MqResult ret = doSendMsg(topic, tag, key, message, useGzip, callback);
        // 失败
        if (!ret.success && ret.msg != null) {
            mqRetryExecutor.execute(() -> {
                retry(ret);
            });
        } else {
            if (callback == null) {
                return;
            }
            executorService.execute(() -> {
                try {
                    callback.onSuccess();
                } catch (Exception e) {
                    LOG.error("", e);
                }
            });
        }
    }

    @DBTimeProfiler
    private MqResult doSendMsg(String topic, String tag, String key, String message, boolean useGzip, MqSentCallback callback) {
        Message msg = null;
        try {
            byte[] body = useGzip ? GZIPUtils.gzip(message) : message.getBytes("utf-8");
            msg = new Message(topic, tag, body);
            //扣积分通知延时推送
            JSONObject json = JSON.parseObject(message);
            if(json.containsKey(DelayMsgConfigNameEnum.DELAY_LEVEL.getName())) {
                Integer level = json.getInteger(DelayMsgConfigNameEnum.DELAY_LEVEL.getName());
                LOG.info("[万达酒店] topic = {}, 延时消息 = [{}]", topic, message);
                msg.setDelayTimeLevel(level);
                json.remove(DelayMsgConfigNameEnum.DELAY_LEVEL.getName());
            }
            if (StringUtils.isNotEmpty(key)) {
                msg.setKeys(key);
            }
            SendResult ret = rocketMqProducer.send(msg);
            //除了SEND_OK外，另外两个状态也是成功
            boolean success = (SendStatus.SEND_OK == ret.getSendStatus()
                    || SendStatus.FLUSH_SLAVE_TIMEOUT == ret.getSendStatus()
                    || SendStatus.SLAVE_NOT_AVAILABLE == ret.getSendStatus());
            if (!success) {
                LOG.warn("MQ发送失败, sendStatus={}, topic={}, tag={}, key={}, msg={}", ret.getSendStatus(), topic, tag, key, message);
            }
            MqResult mqResult = new MqResult();
            mqResult.success = success;
            mqResult.msg = msg;
            mqResult.callback = callback;
            return mqResult;
        } catch (Exception e) {
            LOG.error("Send mq failed, topic={}, tag={}, key={}, msg={}", topic, tag, key, message, e);
            MqResult mqResult = new MqResult();
            mqResult.success = false;
            mqResult.msg = msg;
            mqResult.callback = callback;
            return mqResult;
        }
    }

    /**
     * 重试逻辑
     *
     * @param mqResult
     */
    @DBTimeProfiler
    void retry(MqResult mqResult) {
        mqResult.retryTimes++;
        try {
            // 超过重试次数，按失败处理
            if (mqResult.retryTimes > MAX_RETRY_TIMES) {
                LOG.error("MQ发送失败重试{}次，依旧失败，取消重试，topic={}, tag={}", MAX_RETRY_TIMES, mqResult.msg.getTopic(), mqResult.msg.getTags());
                if (mqResult.callback != null) {
                    mqResult.callback.onFail();
                }
                return;
            }
            // 线程休眠时间第一次重试2s之后，第二次4s，第三次8s
            TimeUnit.SECONDS.sleep(1 << mqResult.retryTimes);
            SendResult sendResult = rocketMqProducer.send(mqResult.msg);
            if (mqResult.callback != null && (SendStatus.SEND_OK == sendResult.getSendStatus()
                    || SendStatus.FLUSH_SLAVE_TIMEOUT == sendResult.getSendStatus()
                    || SendStatus.SLAVE_NOT_AVAILABLE == sendResult.getSendStatus())) {
                mqResult.callback.onSuccess();
            } else {
                retry(mqResult);
            }
        } catch (InterruptedException e) {
            LOG.warn("", e);
        } catch (Exception e) {
            LOG.warn("", e);
            retry(mqResult);
        }
    }

    private class MqResult {
        private boolean success;
        private Message msg;
        private int retryTimes = 0;
        private MqSentCallback callback;
    }

}
