package cn.com.duiba.biz.credits;

import cn.com.duiba.constant.QiaQiaConfig;
import cn.com.duiba.domain.SubCreditsMsgWrapper;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.order.center.api.dto.CreditsMessage;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.tool.AssembleTool;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

/**
 * @author XuJing
 * @since 2020/5/21 09:57 上午
 */
@Service
public class QiaQiaApi {
    private static final Logger LOGGER = LoggerFactory.getLogger(QiaQiaApi.class);

    @Autowired
    private QiaQiaConfig qiaQiaConfig;

    @Resource(name = "httpClient")
    private CloseableHttpClient httpClient;

    public Boolean isQiaQia(Long appId) {
        Set<Long> appIds = qiaQiaConfig.getAppIds();
        if (null == appIds) {
            return false;
        }
        return appIds.contains(appId);
    }

    /**
     * 构造加积分请求
     *
     * @param request
     * @return
     */
    public HttpRequestBase getAddCreditsMessage(CreditsMessageDto request) {
        // 截取请求url
        String httpUrl = request.getHttpUrl();
        // 截取url请求参数，构造新的请求参数
        String paramStr = httpUrl.substring(httpUrl.indexOf('?') + 1);
        Map<String, String> params = AssembleTool.getUrlParams(paramStr);
        return getCreditsRequest(params, true);
    }

    private HttpPut getCreditsRequest(Map<String, String> params, boolean isAdd) {
        HttpPut put = new HttpPut(String.format(qiaQiaConfig.getUrl(), qiaQiaConfig.getPointContextPath(), qiaQiaConfig.getPointVersion(), qiaQiaConfig.getPointRestPath()));
        String time = DateUtil.format(new Date(), "yyyy-MM-dd HH:mm:ss");
        addHeader(put, qiaQiaConfig.getPointContextPath(), qiaQiaConfig.getSelectVersion(), qiaQiaConfig.getPointRestPath(), time);
        put.addHeader("X-Business-Token", params.get("orderNum") + "-" + time);
        put.setEntity(new StringEntity(pointParams(isAdd, params.get("uid"), params.get("credits"), params.get("description"), params.get("orderNum")), "utf-8"));
        return put;
    }

    /**
     * 构造减积分请求 remote
     *
     * @param message
     * @return
     */
    public HttpRequestBase getSubCreditsMessage(CreditsMessage message) {
        String url = message.getHttpUrl();
        String paramsStr = url.substring(url.indexOf('?') + 1);
        Map<String, String> params = AssembleTool.getUrlParams(paramsStr);
        return getCreditsRequest(params, false);
    }

    /**
     * 构造减积分请求 mq
     *
     * @param message
     * @return
     */
    public HttpRequestBase getSubCreditsMessage(SubCreditsMsgWrapper message) {
        String url = message.getHttpUrl();
        String paramsStr = url.substring(url.indexOf('?') + 1);
        Map<String, String> params = AssembleTool.getUrlParams(paramsStr);
        return getCreditsRequest(params, false);
    }


    /**
     * 构造积分兑换请求
     *
     * @param request
     * @return
     */
    public HttpRequestBase getVirtualRequest(SupplierRequest request) {
        String url = request.getHttpUrl();
        String paramsStr = url.substring(url.indexOf('?') + 1);
        Map<String, String> params = AssembleTool.getUrlParams(paramsStr);
        String goodsNum = params.get("params");
        request.setAuthParams(params);
        if (goodsNum.startsWith(qiaQiaConfig.getVirtualCreditsPre())) {
            params.put("credits", goodsNum.replace(qiaQiaConfig.getVirtualCreditsPre(), ""));
            return getCreditsRequest(params, true);
        }
        return AssembleTool.assembleRequest(request.getHttpUrl(), request.getAuthParams());
    }

    private String pointParams(boolean isAdd, String uid, String credits, String description, String orderNum) {
        JSONObject param = new JSONObject();
        if (isAdd) {
            param.put("changeType", "SEND");
            param.put("expiredTime",DateUtil.format(DateUtil.offsetMonth(new Date(),12),"yyyy-MM-dd HH:mm:ss"));
        } else {
            param.put("changeType", "DEDUCT");
        }
        param.put("channelType", qiaQiaConfig.getChannelType());
        param.put("description", description + "-" + orderNum);
        param.put("memberId", uid);
        param.put("memberType", qiaQiaConfig.getMemberType());
        param.put("point", credits);
        return param.toJSONString();
    }

    private void addHeader(HttpRequestBase base, String contextPath, String version, String requestPath, String timestamp) {
        base.addHeader("X-Caller-Service", qiaQiaConfig.getServiceName());
        base.addHeader("X-Caller-Timestamp", timestamp);
        base.addHeader("X-Caller-Sign", generateSign(qiaQiaConfig.getServiceName(), contextPath, version, timestamp, qiaQiaConfig.getServiceSecret(), requestPath));
        base.addHeader("Content-Type", "application/json; charset=utf-8");
    }

    /**
     * 解析积分响应
     *
     * @param body
     * @param addCredits
     * @param authParams
     * @return
     */
    public String parseCreditsRsp(String body, Boolean addCredits, Map<String, String> authParams) {
        Map<String, String> map = new HashMap<>();
        try {
            if (StringUtils.isBlank(body)) {
                map.put("status", "ok");
                map.put("bizId",authParams.get("orderNum"));
            } else {
                JSONObject bodyJson = JSONObject.parseObject(body);
                map.put("status", "fail");
                String msg = bodyJson.getString("msg");
                if (StringUtils.isNotBlank(msg)) {
                    map.put("errorMessage", bodyJson.getString("msg"));
                }else {
                    map.put("errorMessage", bodyJson.getString("message"));
                }
            }
            map.put("credits", getCredits(authParams.get("uid")));
            return JSONObject.toJSONString(map);
        } catch (Exception e) {
            LOGGER.error("恰恰新增/扣减积分接口解析错误:{}--是否新增:{}", body, addCredits, e);
            map.put("status", "fail");
            map.put("errorMessage", "解析积分结果错误");
        }
        return JSONObject.toJSONString(map);
    }


    /**
     * 解析积分兑换响应
     *
     * @param request
     * @param body
     * @return
     */
    public String getVirtualResponse(SupplierRequest request, String body) {
        Map<String, String> params = request.getAuthParams();
        String goodsNum = params.get("params");
        String uid = params.get("uid");
        Map<String, String> duibaDoc = new HashMap<>();
        try {
            if (goodsNum.startsWith(qiaQiaConfig.getVirtualCreditsPre())) {
                if (StringUtils.isBlank(body)) {
                    duibaDoc.put("status", "ok");
                    duibaDoc.put("supplierBizId",params.get("orderNum"));
                    duibaDoc.put("credits", getCredits(uid));
                } else {
                    JSONObject bodyJson = JSONObject.parseObject(body);
                    duibaDoc.put("status", "fail");
                    String msg = bodyJson.getString("msg");
                    if (StringUtils.isNotBlank(msg)) {
                        duibaDoc.put("errorMessage", bodyJson.getString("msg"));
                    }else {
                        duibaDoc.put("errorMessage", bodyJson.getString("message"));
                    }
                }
                return JSON.toJSONString(duibaDoc);

            }
        } catch (Exception e) {
            LOGGER.error("恰恰积分兑换接口解析错误:{}", body, e);
            duibaDoc.put("status", "fail");
            duibaDoc.put("errorMessage", "积分兑换响应解析错误");
            return JSON.toJSONString(duibaDoc);
        }
        return body;
    }


    private String getCredits(String memberId) {
        Map<String, String> map = new HashMap<>();
        map.put("channelType", qiaQiaConfig.getChannelType());
        map.put("memberId", memberId);
        map.put("memberType", qiaQiaConfig.getMemberType());
        HttpGet get = new HttpGet(AssembleTool.assembleUrl(String.format(qiaQiaConfig.getUrl(), qiaQiaConfig.getSelectContextPath(), qiaQiaConfig.getSelectVersion(), qiaQiaConfig.getSelectRestPath()), map));
        String time = DateUtil.format(new Date(), "yyyy-MM-dd HH:mm:ss");
        addHeader(get, qiaQiaConfig.getSelectContextPath(), qiaQiaConfig.getSelectVersion(), qiaQiaConfig.getSelectRestPath(), time);
        try (CloseableHttpResponse response = httpClient.execute(get)) {
            if (response.getStatusLine().getStatusCode() == 200) {
                JSONObject jsonObject = JSON.parseObject(EntityUtils.toString(response.getEntity()));
                Double point = jsonObject.getJSONObject("pointInfo").getDouble("point");
                if (point != null) {
                    Long p = point.longValue();
                    return p.toString();
                }
            }
        } catch (Exception e) {
            LOGGER.error("恰恰积分查询错误 memberid={}", memberId, e);
        }
        return "0";
    }


    public static String generateSign(String callerService, String contextPath, String version,
                                      String timestamp, String serviceSecret, String requestPath) {
        String sign = "";
        if (StringUtils.isBlank(callerService) || StringUtils.isBlank(contextPath)||StringUtils.isBlank(timestamp)||StringUtils.isBlank(serviceSecret)) {
            return sign;
        }
        Map<String, String> map = new LinkedHashMap<>();
        map.put("callerService", callerService);
        map.put("contextPath", contextPath);
        try {
            if (requestPath != null) {
                StringBuilder sb = new StringBuilder();
                for (String part : requestPath.split("/")) {
                    sb.append("/").append(URLEncoder.encode(part, "utf-8"));
                }
                map.put("requestPath", sb.toString().substring(1));
            }
            map.put("timestamp", timestamp);
            map.put("v", version);
            sign = generateMD5Sign(serviceSecret, map);
        } catch (NoSuchAlgorithmException | UnsupportedEncodingException e) {
            LOGGER.error("签名生成失败",e);
            return "";
        }
        return sign;
    }

    private static String generateMD5Sign(String secret, Map<String, String> parameters)
            throws NoSuchAlgorithmException, UnsupportedEncodingException {
        MessageDigest md5 = MessageDigest.getInstance("MD5");
        byte[] bytes = md5.digest(generateConcatSign(secret, parameters).getBytes("utf-8"));
        return byteToHex(bytes);
    }

    private static String generateConcatSign(String secret, Map<String, String>
            parameters) {
        StringBuilder sb = new StringBuilder().append(secret);
        Set<String> keys = parameters.keySet();
        for (String key : keys) {
            sb.append(key).append(parameters.get(key));
        }
        return sb.append(secret).toString();
    }

    private static String byteToHex(byte[] bytesIn) {
        StringBuilder sb = new StringBuilder();
        for (byte byteIn : bytesIn) {
            String bt = Integer.toHexString(byteIn & 0xff);
            if (bt.length() == 1) {
                sb.append(0).append(bt);
            } else {
                sb.append(bt);
            }
        }
        return sb.toString().toUpperCase();
    }
}
