package cn.com.duiba.service.impl;

import cn.com.duiba.config.BlackHostConfig;
import cn.com.duiba.notifycenter.service.NotifyHttpClientPool;
import cn.com.duiba.service.HttpAsyncClientPool;
import cn.com.duiba.service.SlowRequestHandler;
import cn.com.duibaboot.ext.autoconfigure.httpclient.ssre.CanAccessInsideNetwork;
import com.alibaba.ttl.threadpool.TtlExecutors;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.apache.commons.lang.StringUtils;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

/**
 * Created by zzy on 2017/11/20.
 */
@Service
public class SlowRequestHandlerImpl implements SlowRequestHandler, InitializingBean, DisposableBean {
    private static final Logger log = LoggerFactory.getLogger(NotifyHttpClientPool.class);

    //屏弊unKnownHost黑名单列表
//    private static final ImmutableList<String> blackHostList = ImmutableList.of("test.lianwangshenqi.com");
    /**
     * 缓存策略1，在15s内有1次以上时延超过4s
     */
    private static volatile Cache<String, AtomicLong> urlCache1 = CacheBuilder.newBuilder().maximumSize(1000).expireAfterWrite(20, TimeUnit.SECONDS).build();
    /**
     * 缓存策略2，在30s内有3次以上时延超过2s
     */
    private static volatile Cache<String, AtomicLong> urlCache2 = CacheBuilder.newBuilder().maximumSize(1000).expireAfterWrite(30, TimeUnit.SECONDS).build();
    private static final int MAX_SIZE = 50000;
    private final BlockingQueue<NotifyHttpClientPool.CallbackProcesser> notifySlowRequestQueue = new LinkedBlockingQueue<>(MAX_SIZE);
    private final BlockingQueue<HttpAsyncClientPool.CallbackProcesser> creditsSlowRequestQueue = new LinkedBlockingQueue<>(MAX_SIZE);
    private final ExecutorService executorService = TtlExecutors.getTtlExecutorService(Executors.newFixedThreadPool(2, new ThreadFactory() {
        private int count = 1;

        @Override
        public Thread newThread(Runnable r) {
            return new Thread(r, "SlowRequest-" + count++);
        }
    }));
    private volatile boolean isWork = true;
    @Resource(name="httpAsyncClient")
    @CanAccessInsideNetwork
    private CloseableHttpAsyncClient httpClient;
    private final long logSlowReqRtVal = 1000;

    @Autowired
    private BlackHostConfig blackHostConfig;

    /**
     * 是否是慢请求host
     *
     * @param url
     * @return
     */
    @Override
    public boolean isSlow(String url) {
        AtomicLong val1 = urlCache1.getIfPresent(url);
        if (val1 != null && val1.get() >= 1) {
            return true;
        }
        AtomicLong val2 = urlCache2.getIfPresent(url);
        return val2 != null && val2.get() >= 3;
    }

    /**
     * 标记慢请求
     *
     * @param url
     * @param delay 延迟的时间(s)
     */
    public synchronized void addSlow(String url, int delay) {
        if (delay >= 4) {
            AtomicLong val1 = urlCache1.getIfPresent(url);
            if (val1 != null) {
                val1.incrementAndGet();
            } else {
                urlCache1.put(url, new AtomicLong(1));
            }
        } else {
            AtomicLong val2 = urlCache2.getIfPresent(url);
            if (val2 != null) {
                val2.incrementAndGet();
            } else {
                urlCache2.put(url, new AtomicLong(1));
            }
        }
    }

    public boolean addQueue(NotifyHttpClientPool.CallbackProcesser callbackProcesser) {
        if(notifySlowRequestQueue.size() > MAX_SIZE){
            log.warn("notify slow queue size exceeds {}", MAX_SIZE);
        }
        return notifySlowRequestQueue.add(callbackProcesser);
    }

    @Override
    public boolean addQueue(HttpAsyncClientPool.CallbackProcesser callbackProcesser) {
        if(creditsSlowRequestQueue.size() > MAX_SIZE){
            log.warn("credits slow queue size exceeds {}", MAX_SIZE);
        }
        return creditsSlowRequestQueue.add(callbackProcesser);
    }

    public void sendNotifyMsg() {
        NotifyHttpClientPool.CallbackProcesser processer = null;
        try {
            processer = notifySlowRequestQueue.take();
        } catch (InterruptedException e) {
            log.warn("", e);
        }
        if (processer == null) {
            return;
        }
        long s = System.currentTimeMillis();
        httpClient.execute(processer.getRequest(), processer);
        long cost = System.currentTimeMillis() - s;
        if (cost > logSlowReqRtVal) {
            log.warn("slow queue execute cost {}ms, host={}, appId={}", cost, processer.getRequest().getURI().getHost(), processer.getAppId());
        }
    }

    public void sendCreditsMsg() {
        HttpAsyncClientPool.CallbackProcesser processer = null;
        try {
            processer = creditsSlowRequestQueue.take();
        } catch (InterruptedException e) {
            log.warn("", e);
        }
        if (processer == null) {
            return;
        }
        long s = System.currentTimeMillis();
        httpClient.execute(processer.getRequest(), processer);
        long cost = System.currentTimeMillis() - s;
        if (cost > logSlowReqRtVal) {
            log.warn("slow queue execute cost {}ms, host={}, key={}", cost, processer.getRequest().getURI().getHost(), processer.getQueueKey());
        }
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                while (isWork) {
                    try {
                        sendNotifyMsg();
                    } catch (Exception e) {
                        log.warn("sendNotifyMsg failed", e);
                    }
                }
            }
        });
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                while (isWork) {
                    try {
                        sendCreditsMsg();
                    } catch (Exception e) {
                        log.warn("sendCreditsMsg failed", e);
                    }
                }
            }
        });
    }

    @Override
    public void destroy() throws Exception {
        isWork = false;
        executorService.shutdown();
    }

    @Override
    public boolean isBlackHost(String urlPath) {
        String blackHostCfg = blackHostConfig.getBlackHostConfig();
        if(StringUtils.isBlank(blackHostCfg)){
            return false;
        }

        List<String> blackHostList = Arrays.asList(blackHostCfg.trim().split(","));
        for(String blackHost : blackHostList){
            if(urlPath.startsWith(blackHost)){
                return true;
            }
        }
        return false;
    }
}