package cn.com.duiba.biz.credits;

import cn.com.duiba.dao.HttpMessageDAO;
import cn.com.duiba.domain.HttpMessageDO;
import cn.com.duiba.job.AbstractDuibaSimpleElasticJob;
import cn.com.duiba.service.HttpAsyncClientPool;
import cn.com.duiba.thirdparty.dto.HttpRequestMessageDto;
import cn.com.duiba.thirdparty.dto.VirtualCurrencyMessage;
import cn.com.duiba.tool.JsonTool;
import com.alibaba.dubbo.common.utils.CollectionUtils;
import com.alibaba.dubbo.common.utils.StringUtils;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import io.elasticjob.autoconfigure.annotation.ElasticJob;
import io.elasticjob.lite.api.ShardingContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

/**
 * 扣虚拟货币异常重试的定时任务
 */
@ElasticJob(name = "subVirtualCurrencyRetryJob", cron = "0/30 * * * * ?", shardingTotalCount = 1, overwrite = true)
public class SubVirtualCurrencyRetryJob extends AbstractDuibaSimpleElasticJob {

	private Logger log = LoggerFactory.getLogger(SubVirtualCurrencyRetryJob.class);

	Cache<Long, AtomicLong> cache = CacheBuilder.newBuilder().expireAfterWrite(3, TimeUnit.MINUTES).build();

	@Autowired
	private HttpMessageDAO httpMessageDAO;

	@Autowired
	private SubVirtualCurrencyToDeveloper subVirtualCurrencyToDeveloper;

	@Autowired
	private HttpAsyncClientPool httpAsyncClientPool;

	@Override
	public void doProcess(ShardingContext shardingContext) {
		long s = System.currentTimeMillis();
		List<HttpMessageDO> dos = httpMessageDAO.findNeedRetry(HttpRequestMessageDto.RETRY_SUB_VIRTUAL_CURRENCY, 2000);
		if (CollectionUtils.isEmpty(dos)) {
			return;
		}
		for (HttpMessageDO db : dos) {
			if (!checkParams(db)) {
				continue;
			}
			if (httpAsyncClientPool.canSubmitToAppPool(db.getAppId().toString())) {
				VirtualCurrencyMessage virtualCurrencyMessage = JsonTool.jsonToObject(db.getBizParams(), VirtualCurrencyMessage.class);
				String msgTopic = virtualCurrencyMessage.getMsgTopic();
				String msgTag = virtualCurrencyMessage.getMsgTag();
				String msgKey = virtualCurrencyMessage.getMsgKey();
				Long messageId = db.getId();
				Integer messageNumber = db.getNumber();
				subVirtualCurrencyToDeveloper.submit(virtualCurrencyMessage, msgTopic, msgTag, msgKey, messageId, messageNumber);
			} else {
				AtomicLong addCache = cache.getIfPresent(db.getAppId());
				if(addCache == null){
					cache.put(db.getAppId(), new AtomicLong(1));
				}else{
					addCache.incrementAndGet();
				}
			}
		}

		if(cache.size() > 0){
			for(Map.Entry<Long, AtomicLong> entry : cache.asMap().entrySet()){
				log.warn("scan cannot be submit to queue, appId:{}, times:{}", entry.getKey(), entry.getValue().get());
			}
		}

		long e = System.currentTimeMillis();
		log.info("subVirtualCurrency scan size:{} time:{} ms", dos.size(), (e - s));
	}

	private boolean checkParams(HttpMessageDO db) {
		if(db == null || db.getAppId() == null || StringUtils.isBlank(db.getBizParams())){
			return false;
		}
		VirtualCurrencyMessage virtualCurrencyMessage = JsonTool.jsonToObject(db.getBizParams(), VirtualCurrencyMessage.class);
		return virtualCurrencyMessage != null;
	}

}
