package cn.com.duiba.biz.credits;

import cn.com.duiba.biz.Exception.ThirdpatyException;
import cn.com.duiba.constant.UnionPayConstants;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.enums.union.UnionGoodsUserSendType;
import cn.com.duiba.tool.DES3Tool;
import cn.com.duiba.tool.UnionPayBankUtil;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.lang.Pair;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * @Description 银联优惠券Api
 * @Date 2022/3/11
 * @Author WeiChaoChao
 */
@Service
public class UnionCouponApi {
    private static final Logger LOGGER = LoggerFactory.getLogger(UnionCouponApi.class);

    public static final String LOG_NAME = "银联通用优惠券虚拟商品";

    @Autowired
    private UnionPayConstants unionPayConstants;

    @Autowired
    private UnionCommonApi unionCommonApi;

    /**
     * 是否是区分发放身份的银联优惠券虚拟商品
     *
     * @param goodsNum
     * @return
     */
    public Boolean boolUnionNewCouponVirtual(String goodsNum) {
        if (goodsNum.startsWith(UnionGoodsUserSendType.COUPON_MOBILE.getGoodsPrefix())
                || goodsNum.startsWith(UnionGoodsUserSendType.COUPON_OPEN_ID.getGoodsPrefix())) {
            return true;
        }
        return false;
    }

    /**
     * 区分用户发放身份类型的银联优惠券赠送请求
     *
     * @param goodsNum
     * @param authParamMap
     * @param request
     * @return
     */
    public HttpRequestBase buildCouponSendRequest(String goodsNum, Map<String, String> authParamMap, SupplierRequest request) {
        /**
         *  商品编码前缀 UnionCouponMobile 以手机号作为用户身份发放
         *  商品编码前缀 UnionCouponOpenId 以小程序openId作为用户身份去发放
         *  商品编码格式为 {商品编码前缀}_{小程序appId}_{券Id}
         */
        String[] s = goodsNum.split("_");
        String goodsPrefix = s[0];
        String unionAppId = s[1];
        String couponId = s[2];
        // 从楼层系统中拿到小程序对应的openId和手机号
        Pair<String, String> userOpenIdAndMobile = unionCommonApi.getUserOpenIdAndMobile(Long.valueOf(request.getConsumerId()), unionAppId);
        String openId = userOpenIdAndMobile.getKey();
        String mobile = userOpenIdAndMobile.getValue();

        // 组装请求参数
        Map<String, String> requestParams = new HashMap<>();
        requestParams.put("appId", unionAppId);
        requestParams.put("transSeqId", authParamMap.get("orderNum"));
        requestParams.put("transTs", DateUtil.format(new Date(), "yyyyMMdd"));
        requestParams.put("couponId", couponId);
        // 区分用户身份类型赠送
        if (UnionGoodsUserSendType.COUPON_OPEN_ID.getGoodsPrefix().equals(goodsPrefix)) {
            requestParams.put("openId", openId);
            requestParams.put("acctEntityTp", UnionGoodsUserSendType.COUPON_OPEN_ID.getAcctEntityTp());
        } else if (UnionGoodsUserSendType.COUPON_MOBILE.getGoodsPrefix().equals(goodsPrefix)) {
            requestParams.put("mobile", mobile);
            requestParams.put("acctEntityTp", UnionGoodsUserSendType.COUPON_MOBILE.getAcctEntityTp());
        }
        requestParams.put("couponNum", "1");
        requestParams.put("nonceStr", UnionPayBankUtil.createNonceStr());
        requestParams.put("timestamp", Long.toString(System.currentTimeMillis() / 1000));
        requestParams.put("signature", UnionPayBankUtil.sign(requestParams, unionPayConstants.getSignPrivateKey(unionAppId)));
        LOGGER.info("{}, orderId={}, 请求参数={}", LOG_NAME, request.getOrderId(), JSON.toJSONString(requestParams));
        // 先验签再加密
        if (goodsPrefix.equals(UnionGoodsUserSendType.COUPON_MOBILE.getGoodsPrefix())) {
            try {
                requestParams.put("mobile", DES3Tool.getEncryptedValue(mobile, unionPayConstants.getSymmetricKey(unionAppId)));
            } catch (Exception e) {
                LOGGER.warn("{}, 虚拟商品参数加密失败, orderId={}", LOG_NAME, request.getOrderId(), e);
                throw new ThirdpatyException(LOG_NAME + ", 加密失败");
            }
        }
        HttpPost post = new HttpPost(unionPayConstants.getVirtualUrl());
        post.setEntity(new StringEntity(JSON.toJSONString(requestParams), ContentType.APPLICATION_JSON));
        return post;
    }

    /**
     * 区分用户发放身份的赠送优惠券结果解析
     *
     * @param request
     * @param body
     * @return
     */
    public String buildUnionCouponResponse(SupplierRequest request, String body) {
        if (StringUtils.isBlank(body)) {
            throw new ThirdpatyException("银联优惠券，虚拟商品定制，返回结果为空");
        }
        LOGGER.info("{},响应参数, orderId={}, body={}", LOG_NAME, request.getOrderId(), body);
        Map<String, String> map = new HashMap<>();
        try {
            JSONObject jsonObject = JSON.parseObject(body);
            map.put("status", Objects.equals("00", jsonObject.getString("resp")) ? "success" : "fail");
            map.put("errorMessage", StringEscapeUtils.unescapeHtml4(jsonObject.getString("msg")));
        } catch (Exception e) {
            LOGGER.error("{},响应参数，结果解析错误:{}", LOG_NAME, body, e);
            map.put("status", "fail");
            map.put("errorMessage", "虚拟商品充值接口响应解析错误");
        }
        return JSON.toJSONString(map);
    }

}
