package cn.com.duiba.biz.credits;

import cn.com.duiba.biz.Exception.ThirdpatyException;
import cn.com.duiba.constant.UnionJiDianConstant;
import cn.com.duiba.constant.UnionPayConstants;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.tool.DES3Tool;
import cn.com.duiba.tool.UnionPayBankUtil;
import cn.com.duiba.vo.union.UnionJiDianConfig;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.lang.Pair;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @Description 银联积分虚拟商品api
 * @Date 2022/3/2
 * @Author WeiChaoChao
 */
@Service
public class UnionJiDianApi {
    private static final Logger LOGGER = LoggerFactory.getLogger(UnionJiDianApi.class);

    public static final String LOG_NAME = "银联积点虚拟商品";

    @Autowired
    private UnionJiDianConstant unionJiDianConstant;

    @Autowired
    private UnionPayConstants unionPayConstants;

    @Autowired
    private UnionCommonApi unionCommonApi;

    /**
     * 构建积点虚拟商品请求request
     *
     * @param authParams
     * @return
     */
    public HttpRequestBase buildUnionJiDianRequest(Map<String, String> authParams, SupplierRequest request, UnionJiDianConfig unionJiDianConfig) {
        String duibaAppId = request.getAppId();
        String unionAppId = unionJiDianConfig.getMiniAppId();
        // 银联小程序免登用的是unionId openId作为extra参数
        Pair<String, String> userOpenIdAndMobile = unionCommonApi.getUserOpenIdAndMobile(Long.valueOf(request.getConsumerId()), unionAppId);
        String openId = userOpenIdAndMobile.getKey();
        String mobile = userOpenIdAndMobile.getValue();
        if (StringUtils.isBlank(openId) && StringUtils.isBlank(mobile)) {
            LOGGER.error("{}, appId={}, consumerId={}, 用户参数缺失", LOG_NAME, duibaAppId, request.getConsumerId());
            throw new RuntimeException("用户信息参数缺失");
        }

        // 解析商家编码的值
        Pair<String, String> params1 = getBusinessTypeAndPoint(authParams.get("params"));
        String jidianNum = params1.getValue();
        // 是否以openId的形式发放 1: 以小程序openId发放 2：以手机号发放
        boolean sendByOpenId = StringUtils.isBlank(unionJiDianConfig.getSendUserIdType()) || "openId".equals(unionJiDianConfig.getSendUserIdType());

        // 组装请求参数
        Map<String, String> params = new HashMap<>();
        params.put("appId", unionAppId);
        params.put("sysId", unionJiDianConfig.getSysId());
        // 默认发放方式以openId的形式发放
        if (sendByOpenId) {
            params.put("openId", openId);
        } else {
            params.put("mobile", mobile);
        }
        params.put("getSource", unionJiDianConfig.getGetSource());
        params.put("transSeqId", authParams.get("orderNum"));
        params.put("transTs", DateUtil.format(new Date(), "yyyyMMddHHmmss"));
        params.put("pointAt", jidianNum);
        params.put("transDigest", unionJiDianConfig.getTransDigest());
        params.put("descCode", unionJiDianConfig.getDescCode());
        params.put("inMode", unionJiDianConfig.getInMode());
        params.put("timestamp", Long.toString(System.currentTimeMillis() / 1000));
        params.put("nonceStr", UnionPayBankUtil.createNonceStr());
        params.put("signature", UnionPayBankUtil.sign(params, unionPayConstants.getSignPrivateKey(unionAppId)));
        LOGGER.info("{},请求参数, orderId={}, params={}", LOG_NAME, request.getOrderId(), JSON.toJSONString(params));

        // 以下字段 先明文签名后再加密传输
        try {
            String symmetricKey = unionPayConstants.getSymmetricKey(unionAppId);
            params.put("pointAt", DES3Tool.getEncryptedValue(jidianNum, symmetricKey));
            if (!sendByOpenId) {
                params.put("mobile", DES3Tool.getEncryptedValue(mobile, symmetricKey));
            }
        } catch (Exception e) {
            LOGGER.warn("{}, 银联积点虚拟商品加密失败, orderId={}", LOG_NAME, request.getOrderId(), e);
            throw new ThirdpatyException(LOG_NAME + ", 加密失败");
        }

        HttpPost post = new HttpPost(unionJiDianConstant.getJiDianSendUrl());
        post.setEntity(new StringEntity(JSON.toJSONString(params), ContentType.APPLICATION_JSON));
        return post;
    }

    /**
     * 解析虚拟商品充值响应结果
     *
     * @param request
     * @param body
     */
    public String buildUnionJiDianResponse(SupplierRequest request, String body) {
        if (StringUtils.isBlank(body)) {
            throw new ThirdpatyException("银联积点,虚拟商品定制,返回结果为空");
        }
        LOGGER.info("{}, 银联积点响应参数, orderId={}, body={}", LOG_NAME, request.getOrderId(), body);
        Map<String, String> map = new HashMap<>();
        try {
            JSONObject jsonObject = JSON.parseObject(body);
            map.put("status", Objects.equals("00", jsonObject.getString("resp")) ? "success" : "fail");
            map.put("errorMessage", StringEscapeUtils.unescapeHtml4(jsonObject.getString("msg")));
        } catch (Exception e) {
            LOGGER.error("{}，虚拟商品定制，结果解析错误:{}", LOG_NAME, body, e);
            map.put("status", "fail");
            map.put("errorMessage", "虚拟商品充值接口响应解析错误");
        }
        return JSON.toJSONString(map);
    }


    /**
     * 解析商家编码字段
     *
     * @param params
     * @return
     */
    private Pair<String, String> getBusinessTypeAndPoint(String params) {
        /**
         * 标准积点商品编码参数
         * 积点商品前缀    业务类型标识    赠送积点数值
         * UnionJiDian_${businessType}_${num}
         */
        String[] goodsNum = params.split("_");
        if (goodsNum.length < 3) {
            throw new ThirdpatyException("银联积点商品编码错误");
        }
        return Pair.of(goodsNum[1], goodsNum[2]);
    }


    /**
     * 判断是否是银联积点商品
     *
     * @param goodsParams
     * @param duibaAppId
     * @return
     */
    public UnionJiDianConfig getUnionJiDianGoodsConfig(String goodsParams, String duibaAppId) {
        // 根据开发者appId判断此应用是否有配置银联积点虚拟商品配置
        List<UnionJiDianConfig> unionJiDianConfigs = JSON.parseArray(unionJiDianConstant.getMiniAppConfigs(), UnionJiDianConfig.class);
        List<UnionJiDianConfig> unionJiDianConfig = unionJiDianConfigs.stream()
                .filter(item -> duibaAppId.equals(item.getAppId()))
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(unionJiDianConfig)) {
            return null;
        }

        // 解析商家编码值找到对应小程序的配置
        Pair<String, String> businessTypeAndPoint = getBusinessTypeAndPoint(goodsParams);
        String businessType = businessTypeAndPoint.getKey();
        UnionJiDianConfig unionBusConfig = unionJiDianConfig.stream()
                .filter(item -> businessType.equals(item.getBusinessType()))
                .findFirst()
                .orElse(null);
        if (unionBusConfig == null) {
            return null;
        }
        return unionBusConfig;
    }
}
