package cn.com.duiba.biz.credits;

import cn.com.duiba.biz.Exception.ThirdpatyException;
import cn.com.duiba.constant.UnionMemberConstant;
import cn.com.duiba.constant.UnionPayConstants;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.tool.DES3Tool;
import cn.com.duiba.tool.UnionPayBankUtil;
import cn.com.duiba.vo.union.UnionMemberConfig;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.lang.Pair;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @Description 银联会员虚拟商品api
 * @Date 2022/3/2
 * @Author WeiChaoChao
 */
@Service
public class UnionMemberApi {
    private static final Logger LOGGER = LoggerFactory.getLogger(UnionMemberApi.class);

    public static final String LOG_NAME = "银联会员虚拟商品";

    @Autowired
    private UnionMemberConstant unionMemberConstant;

    @Autowired
    private UnionPayConstants unionPayConstants;

    @Autowired
    private UnionCommonApi unionCommonApi;

    /**
     * 构建积点虚拟商品请求request
     *
     * @param authParams
     * @return
     */
    public HttpRequestBase buildUnionMemberRequest(Map<String, String> authParams, SupplierRequest request, UnionMemberConfig unionMemberConfig) {
        String duibaAppId = request.getAppId();
        String unionAppId = unionMemberConfig.getMiniAppId();
        // 银联小程序免登用的是unionId openId作为extra参数
        Pair<String, String> userOpenIdAndMobile = unionCommonApi.getUserOpenIdAndMobile(Long.valueOf(request.getConsumerId()), unionAppId);
        String openId = userOpenIdAndMobile.getKey();
        String mobile = userOpenIdAndMobile.getValue();
        if (StringUtils.isBlank(openId) && StringUtils.isBlank(mobile)) {
            LOGGER.error("{}, appId={}, consumerId={}, 用户参数缺失", LOG_NAME, duibaAppId, request.getConsumerId());
            throw new RuntimeException("用户信息参数缺失");
        }

        // 解析商家编码的值
        Pair<String, String> params1 = getBusinessTypeAndMemberType(authParams.get("params"));
        String memberType = params1.getValue();
        // 是否以openId的形式发放 1: 以小程序openId发放 2：以手机号发放
        boolean sendByOpenId = StringUtils.isBlank(unionMemberConfig.getSendUserIdType()) || "openId".equals(unionMemberConfig.getSendUserIdType());

        // 组装请求参数
        Map<String, String> params = new HashMap<>();
        params.put("appId", unionAppId);
        params.put("sysId", unionMemberConfig.getSysId());
        // 默认发放方式以openId的形式发放
        if (sendByOpenId) {
            params.put("openId", openId);
        } else {
            params.put("mobile", mobile);
        }
        params.put("isLimit", unionMemberConfig.getLimit());
        if (StringUtils.isNotBlank(unionMemberConfig.getRemark())) {
            params.put("remark", unionMemberConfig.getRemark());
        }
        params.put("memberType", memberType);
        params.put("transSeqId", authParams.get("orderNum"));
        params.put("transTs", DateUtil.format(new Date(), "yyyyMMddHHmmss"));
        params.put("timestamp", Long.toString(System.currentTimeMillis() / 1000));
        params.put("nonceStr", UnionPayBankUtil.createNonceStr());
        params.put("signature", UnionPayBankUtil.sign(params, unionPayConstants.getSignPrivateKey(unionAppId)));
        LOGGER.info("{},请求参数, orderId={}, params={}", LOG_NAME, request.getOrderId(), JSON.toJSONString(params));

        // 以下字段 先明文签名后再加密传输
        try {
            String symmetricKey = unionPayConstants.getSymmetricKey(unionAppId);
            if (!sendByOpenId) {
                params.put("mobile", DES3Tool.getEncryptedValue(mobile, symmetricKey));
            }
        } catch (Exception e) {
            LOGGER.warn("{}, 银联会员虚拟商品加密失败, orderId={}", LOG_NAME, request.getOrderId(), e);
            throw new ThirdpatyException(LOG_NAME + ", 加密失败");
        }

        HttpPost post = new HttpPost(unionMemberConstant.getMemberSendUrl());
        post.setEntity(new StringEntity(JSON.toJSONString(params), ContentType.APPLICATION_JSON));
        return post;
    }

    /**
     * 解析虚拟商品充值响应结果
     *
     * @param request
     * @param body
     */
    public String buildUnionMemberResponse(SupplierRequest request, String body) {
        if (StringUtils.isBlank(body)) {
            throw new ThirdpatyException("银联会员,虚拟商品定制,返回结果为空");
        }
        LOGGER.info("{}, 银联会员响应参数, orderId={}, body={}", LOG_NAME, request.getOrderId(), body);
        Map<String, String> map = new HashMap<>();
        try {
            JSONObject jsonObject = JSON.parseObject(body);
            map.put("status", Objects.equals("00", jsonObject.getString("resp")) ? "success" : "fail");
            map.put("errorMessage", StringEscapeUtils.unescapeHtml4(jsonObject.getString("msg")));
        } catch (Exception e) {
            LOGGER.error("{}，虚拟商品定制，结果解析错误:{}", LOG_NAME, body, e);
            map.put("status", "fail");
            map.put("errorMessage", "虚拟商品充值接口响应解析错误");
        }
        return JSON.toJSONString(map);
    }


    /**
     * 解析商家编码字段
     *
     * @param params
     * @return
     */
    private Pair<String, String> getBusinessTypeAndMemberType(String params) {
        /**
         * 标准会员商品编码参数
         * 积点商品前缀    业务类型标识     会员类型 (01月卡 02季卡 03 年卡 05食神卡 08 svip)
         * UnionMember_${businessType}_${memberType}
         */
        String[] goodsNum = params.split("_");
        if (goodsNum.length < 3) {
            throw new ThirdpatyException("银联会员商品编码错误");
        }
        return Pair.of(goodsNum[1], goodsNum[2]);
    }


    /**
     * 判断是否是银联会员商品
     *
     * @param goodsParams
     * @param duibaAppId
     * @return
     */
    public UnionMemberConfig getUnionMemberGoodsConfig(String goodsParams, String duibaAppId) {
        // 根据开发者appId判断此应用是否有配置银联积点虚拟商品配置
        List<UnionMemberConfig> unionMemberConfigs = JSON.parseArray(unionMemberConstant.getMiniAppConfigs(), UnionMemberConfig.class);
        List<UnionMemberConfig> unionMemberConfig = unionMemberConfigs.stream()
                .filter(item -> duibaAppId.equals(item.getAppId()))
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(unionMemberConfig)) {
            return null;
        }

        // 解析商家编码值找到对应小程序的配置
        Pair<String, String> businessTypeAndPoint = getBusinessTypeAndMemberType(goodsParams);
        String businessType = businessTypeAndPoint.getKey();
        UnionMemberConfig unionBusConfig = unionMemberConfig.stream()
                .filter(item -> businessType.equals(item.getBusinessType()))
                .findFirst()
                .orElse(null);
        if (unionBusConfig == null) {
            return null;
        }
        return unionBusConfig;
    }
}
