package cn.com.duiba.biz.credits;

import cn.com.duiba.biz.Exception.ThirdpatyException;
import cn.com.duiba.constant.UnionPayConstants;
import cn.com.duiba.constant.UnionRedConstant;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.tool.DES3Tool;
import cn.com.duiba.tool.UnionPayBankUtil;
import cn.com.duiba.vo.union.UnionRedConfig;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.lang.Pair;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @Description 银联发红包api
 * @Date 2022/3/11
 * @Author WeiChaoChao
 */
@Service
public class UnionRedApi {

    private static final String LOG_NAME = "银联通用红包虚拟商品";

    private static final Logger LOGGER = LoggerFactory.getLogger(UnionRedApi.class);

    @Autowired
    private UnionRedConstant unionRedConstant;

    @Autowired
    private UnionPayConstants unionPayConstants;

    @Autowired
    private UnionCommonApi unionCommonApi;


    /**
     * 获取银联红包相关配置
     *
     * @param goodsParams
     * @param duibaAppId
     * @return
     */
    public UnionRedConfig getUnionRedConfig(String goodsParams, String duibaAppId) {
        // 根据开发者appId判断此应用是否有配置银联积点虚拟商品配置
        List<UnionRedConfig> unionRedConfigs = JSON.parseArray(unionRedConstant.getMiniAppConfigs(), UnionRedConfig.class);
        List<UnionRedConfig> unionRedMiniConfig = unionRedConfigs.stream()
                .filter(item -> duibaAppId.equals(item.getAppId()))
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(unionRedMiniConfig)) {
            return null;
        }

        // 解析商家编码值找到对应小程序的配置
        Pair<String, String> businessTypeAndPoint = getBusinessTypeAndPoint(goodsParams);
        String businessType = businessTypeAndPoint.getKey();
        UnionRedConfig unionBusConfig = unionRedMiniConfig.stream()
                .filter(item -> businessType.equals(item.getBusinessType()))
                .findFirst()
                .orElse(null);
        if (unionBusConfig == null) {
            return null;
        }
        return unionBusConfig;
    }

    /**
     * 解析商家编码字段
     *
     * @param params
     * @return
     */
    private Pair<String, String> getBusinessTypeAndPoint(String params) {
        /**
         * 标准红包商品编码参数
         * 红包商品前缀  业务类型标识  赠送红包数值(单位分)
         * UnionRed_${businessType}_${num}
         */
        String[] goodsNum = params.split("_");
        if (goodsNum.length < 3) {
            throw new ThirdpatyException("银联红包商品编码错误");
        }
        return Pair.of(goodsNum[1], goodsNum[2]);
    }

    /**
     * 构建红包银联红包请求
     *
     * @param authParamMap
     * @param request
     * @param unionRedConfig
     * @return
     */
    public HttpRequestBase buildUnionRedRequest(Map<String, String> authParamMap, SupplierRequest request, UnionRedConfig unionRedConfig) {
        // 银联小程序AppId
        String miniAppId = unionRedConfig.getMiniAppId();

        // 从楼层系统中拿到小程序对应的openId和手机号
        Pair<String, String> userOpenIdAndMobile = unionCommonApi.getUserOpenIdAndMobile(Long.valueOf(request.getConsumerId()), miniAppId);
        String openId = userOpenIdAndMobile.getKey();
        String mobile = userOpenIdAndMobile.getValue();
        if (StringUtils.isBlank(openId) && StringUtils.isBlank(mobile)) {
            LOGGER.error("{}, appId={}, consumerId={}, 用户参数缺失", LOG_NAME, request.getAppId(), request.getConsumerId());
            throw new RuntimeException("用户信息参数缺失");
        }

        // 是否以openId的形式发放 1: 以小程序openId发放 2：以手机号发放
        boolean sendByOpenId = StringUtils.isBlank(unionRedConfig.getSendUserIdType()) || "openId".equals(unionRedConfig.getSendUserIdType());

        // 商品编码
        String goodsNum = authParamMap.get("params");
        // 赠送红包数值
        Pair<String, String> businessTypeAndPoint = getBusinessTypeAndPoint(goodsNum);
        String pointAt = businessTypeAndPoint.getValue();

        // 组装请求参数
        Map<String, String> requestParams = new HashMap<>();
        requestParams.put("appId", miniAppId);
        requestParams.put("transSeqId", authParamMap.get("orderNum"));
        requestParams.put("transTs", DateUtil.format(new Date(), "yyyyMMddHHmmss"));
        requestParams.put("nonceStr", UnionPayBankUtil.createNonceStr());
        requestParams.put("insAcctId", unionRedConfig.getInsAcctId());
        requestParams.put("pointId", unionRedConfig.getPointId());
        requestParams.put("pointAt", pointAt);
        requestParams.put("transDigest", unionRedConfig.getTransDigest());
        requestParams.put("timestamp", Long.toString(System.currentTimeMillis() / 1000));

        // 区分是使用openId发放还是手机号发放
        if (sendByOpenId) {
            requestParams.put("openId", openId);
            requestParams.put("acctEntityTp", "03");
        } else {
            requestParams.put("mobile", mobile);
            requestParams.put("acctEntityTp", "01");
        }

        // 业务信息
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("campaignId", unionRedConfig.getCampaignId());
        jsonObject.put("campaignName", unionRedConfig.getCampaignName());
        requestParams.put("busiInfo", jsonObject.toJSONString());

        // 备注信息 配置了则传  没配置则不传
        if (StringUtils.isNotBlank(unionRedConfig.getRemark())) {
            requestParams.put("remark", unionRedConfig.getRemark());
        }

        // 有效开始时间和有效截至时间 没配置则传
        if (StringUtils.isNotBlank(unionRedConfig.getValidBeginTs()) && StringUtils.isNotBlank(unionRedConfig.getValidEndTs())) {
            requestParams.put("validBeginTs", unionRedConfig.getValidBeginTs());
            requestParams.put("validEndTs", unionRedConfig.getValidEndTs());
        }

        //先签再加密
        requestParams.put("signature", UnionPayBankUtil.sign(requestParams, unionPayConstants.getSignPrivateKey(miniAppId)));
        LOGGER.info("{}, orderId={}, 请求参数={}", LOG_NAME, request.getOrderId(), JSON.toJSONString(requestParams));
        try {
            if (!sendByOpenId) {
                requestParams.put("mobile", DES3Tool.getEncryptedValue(mobile, unionPayConstants.getSymmetricKey(miniAppId)));
            }
        } catch (Exception e) {
            LOGGER.warn("{}, 虚拟商品参数加密失败, orderId={}", LOG_NAME, request.getOrderId(), e);
            throw new ThirdpatyException(LOG_NAME + ", 加密失败");
        }
        HttpPost post = new HttpPost(unionRedConstant.getRedSendUrl());
        post.setEntity(new StringEntity(JSON.toJSONString(requestParams), ContentType.APPLICATION_JSON));
        return post;
    }

    /**
     * 解析虚拟商品充值响应结果
     *
     * @param request
     * @param body
     */
    public String buildUnionRedResponse(SupplierRequest request, String body) {
        if (StringUtils.isBlank(body)) {
            throw new ThirdpatyException("银联红包，虚拟商品定制，返回结果为空");
        }
        LOGGER.info("{}, 银联红包响应参数, orderId={}, body={}", LOG_NAME, request.getOrderId(), body);
        Map<String, String> map = new HashMap<>();
        try {
            JSONObject jsonObject = JSON.parseObject(body);
            map.put("status", Objects.equals("00", jsonObject.getString("resp")) ? "success" : "fail");
            map.put("errorMessage", StringEscapeUtils.unescapeHtml4(jsonObject.getString("msg")));
        } catch (Exception e) {
            LOGGER.error("{}，虚拟商品定制，结果解析错误:{}", LOG_NAME, body, e);
            map.put("status", "fail");
            map.put("errorMessage", "虚拟商品充值接口响应解析错误");
        }
        return JSON.toJSONString(map);
    }
}
