package cn.com.duiba.biz.credits;

import cn.com.duiba.api.bo.subcredits.SubCreditsMsgDto;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.constant.XieChengConfig;
import cn.com.duiba.domain.SubCreditsMsgWrapper;
import cn.com.duiba.order.center.api.dto.CreditsMessage;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.tool.JsonTool;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.codec.digest.HmacUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;

/**
 * 携程接口服务 api
 *
 * @author cxw
 */
@Service
public class XieChengApi {

    private static final Logger LOGGER = LoggerFactory.getLogger(XieChengApi.class);

    private static final String CHARACTER_ENCODE = "UTF-8";
    private static final String APPLICATION_JSON = "application/json";
    /**
     * transfer 参数分割符  channelCode_duiba_accessToken
     */
    private static final String TRANSFER_SPLIT = "_duiba_";

    @Autowired
    private XieChengConfig xieChengConfig;

    public boolean isXieChengAppid(Long appId) {
        String xiechengAppId = xieChengConfig.getAppId();
        return StringUtils.isNotBlank(xiechengAppId) && xiechengAppId.equals(String.valueOf(appId));
    }

    /**
     * 组装扣积分对象  remote形式发起的扣积分请求
     *
     * @param request
     * @return
     */
    public CreditsMessage getSubCreditsMessage(CreditsMessage request) throws BizException {
        Map<String, String> originAuthParams = request.getAuthParams();
        // 订单号作为请求流水号
        String requestNo = originAuthParams.get("orderNum");
        // 构建请求参数
        Map<String, String> authParams = this.buildAuthParams(requestNo, originAuthParams);
        request.setAuthParams(authParams);
        request.setHttpType(CreditsMessage.HTTP_POST);
        return request;
    }

    /**
     * 构造减积分请求 remote
     *
     * @param subCreditsMsgWrapper
     * @return
     */
    public HttpRequestBase getSubCreditsHttpRequestBase(SubCreditsMsgWrapper subCreditsMsgWrapper) {
        Map<String, String> params = subCreditsMsgWrapper.getSubCreditsMsg().getAuthParams();
        String creditsConsumeRequestUrl = subCreditsMsgWrapper.getSubCreditsMsg().getCreditsConsumeRequestUrl();
        String paramsJsonStr = JSON.toJSONString(params);
        HttpPost httpPost = new HttpPost(creditsConsumeRequestUrl);
        StringEntity stringEntity = new StringEntity(paramsJsonStr, CHARACTER_ENCODE);
        stringEntity.setContentEncoding(CHARACTER_ENCODE);
        stringEntity.setContentType(APPLICATION_JSON);
        httpPost.setEntity(stringEntity);
        subCreditsMsgWrapper.setHttpUrl(creditsConsumeRequestUrl);
        LOGGER.info("携程定制扣积分请求，requestUrl：{}，params：{}", creditsConsumeRequestUrl, paramsJsonStr);
        return httpPost;
    }


    /**
     * 组装扣积分对象  mq形式发起的扣积分请求
     *
     * @param subCreditsMsgWrapper
     * @return
     */
    public SubCreditsMsgWrapper getSubCreditsMessage(SubCreditsMsgWrapper subCreditsMsgWrapper) throws BizException {
        SubCreditsMsgDto msg = subCreditsMsgWrapper.getSubCreditsMsg();
        Map<String, String> originAuthParams = msg.getAuthParams();
        // 订单号作为请求流水号
        String requestNo = originAuthParams.get("orderNum");
        // 构建请求参数
        Map<String, String> authParams = this.buildAuthParams(requestNo, originAuthParams);
        msg.setAuthParams(authParams);
        msg.setHttpType(SubCreditsMsgDto.HTTP_POST);
        return subCreditsMsgWrapper;
    }


    /**
     * 组装请求params
     *
     * @param originAuthParams 原始参数
     * @return
     */
    private Map<String, String> buildAuthParams(String requestNo, Map<String, String> originAuthParams) {
        // transfer包括两个参数  channelCode  accessToken
        String[] transfers = originAuthParams.get("transfer").split(TRANSFER_SPLIT);
        // accessToken 获取（开发者传来的 放在dcustomer里）
        String accessToken = transfers[1];
        // openId 获取（用户id）
        String openId = originAuthParams.get("uid");
        // pointNum 获取积分数量
        String pointNum = originAuthParams.get("credits");
        // 携程channelCode配置
        String channelCode = transfers[0];
        // 当前时间戳
        String timestamp = String.valueOf(System.currentTimeMillis());
        //signVal String 是 参数签名（使用 hmacSha1 进行签名。加密串：accessToken+channelCode+openId+pointNum+requestNo+timestamp）
        String signVal = accessToken + channelCode + openId + pointNum + requestNo + timestamp;

        // 请求参数
        Map<String, String> authParams = new HashMap<>(7);
        authParams.put("openId", openId);
        authParams.put("accessToken", accessToken);
        authParams.put("pointNum", pointNum);
        authParams.put("requestNo", requestNo);
        authParams.put("channelCode", channelCode);
        authParams.put("timestamp", timestamp);

        // 携程scene配置
        if(transfers.length >= 3){
            authParams.put("scene", transfers[2]);
            LOGGER.info("携程定制加积分请求，场景值：{}，params：{}", transfers[2], JSON.toJSONString(authParams));
        }

        try {
            authParams.put("sign", HmacUtils.hmacSha1Hex(xieChengConfig.getSignKey(), signVal));
        } catch (Exception e) {
            LOGGER.warn("携程 sign 签名失败 params = {}", JSON.toJSONString(authParams), e);
        }
        return authParams;
    }

    /**
     * 携程增加积分请求
     *
     * @param request
     * @return
     */
    public CreditsMessageDto getAddCreditsMessage(CreditsMessageDto request) {
        Map<String, String> params = request.getAuthParams();
        // bizId --> requestNo
        String bizId = request.getBizId();
        Map<String, String> authParams = this.buildAuthParams(bizId, params);
        request.setAuthParams(authParams);
        request.setHttpType(CreditsMessage.HTTP_POST);
        return request;
    }

    /**
     * 构造加积分请求 remote
     *
     * @param
     * @return
     */
    public HttpRequestBase getAddCreditsHttpRequestBase(CreditsMessageDto message) {
        Map<String, String> params = message.getAuthParams();
        String httpUrl = message.getHttpUrl();
        String host = httpUrl.substring(0, httpUrl.indexOf('?'));
        String paramsJsonStr = JSONObject.toJSONString(params);
        HttpPost httpPost = new HttpPost(host);
        StringEntity stringEntity = new StringEntity(paramsJsonStr, CHARACTER_ENCODE);
        stringEntity.setContentEncoding(CHARACTER_ENCODE);
        stringEntity.setContentType(APPLICATION_JSON);
        httpPost.setEntity(stringEntity);
        message.setHttpUrl(host);
        LOGGER.info("携程定制加积分请求，requestUrl：{}，params：{}", host, paramsJsonStr);
        return httpPost;
    }

    /**
     * 积分接口响应
     *
     * @param body
     * @param authParams
     * @return
     */
    public String getResponseCredits(String body, Map<String, String> authParams) {
        JSONObject jsonBody;
        try {
            jsonBody = JSON.parseObject(body);
            Map<String, String> duibaDoc = new HashMap<>();
            String msg = jsonBody.getString("msg");
            if ("SUCCESS".equals(msg)) {
                duibaDoc.put("status", "ok");
            } else {
                LOGGER.warn("携程定制积分加减响应异常 body={}，请求参数={}", body, JSON.toJSONString(authParams));
                duibaDoc.put("status", "fail");
            }
            duibaDoc.put("errorMessage", msg);
            duibaDoc.put("bizId", authParams.get("requestNo"));
            //获取用户积分
            duibaDoc.put("credits", authParams.get("pointNum"));
            return JsonTool.objectToJson(duibaDoc);
        } catch (Exception e) {
            LOGGER.error("携程定制积分加减 响应异常 body={},error={}", body, e.getMessage());
            return body;
        }
    }
}
