package cn.com.duiba.remoteimpl.boc;

import cn.com.duiba.config.FuJianBocConfig;
import cn.com.duiba.constant.boc.FjBocConstant;
import cn.com.duiba.thirdparty.api.boc.RemoteFjBocService;
import cn.com.duiba.thirdparty.dto.boc.IdentityDataDto;
import cn.com.duiba.thirdparty.dto.boc.LabelDataReqParam;
import cn.com.duiba.thirdparty.dto.boc.PublicDataReqParam;
import cn.com.duiba.thirdparty.dto.boc.UnionNoReqParam;
import cn.com.duiba.thirdparty.dto.boc.UserJoinActivityDataDto;
import cn.com.duiba.thirdparty.dto.boc.UserPvAndUvDataDto;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.boc.ConverTool;
import cn.com.duiba.tool.boc.SM3;
import cn.hutool.core.util.HexUtil;
import cn.hutool.crypto.BCUtil;
import cn.hutool.crypto.SecureUtil;
import cn.hutool.crypto.asymmetric.KeyType;
import cn.hutool.crypto.asymmetric.SM2;
import cn.hutool.crypto.digest.DigestAlgorithm;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.bouncycastle.crypto.engines.SM2Engine;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpMethod;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author chenzhenxing
 * @date 2021/11/10 5:15 下午
 */
@RestController
public class RemoteFjBocServiceImpl implements RemoteFjBocService {

    private static Logger log = LoggerFactory.getLogger(RemoteFjBocServiceImpl.class);

    private static final String DUIBA_DATA_UPLOAD_PATH = "/unlogin/fjtj/duibaDataUpload";

    private static final String WX_DATA_URL_PATH = "/unlogin/fjtj/wxdata";

    private static final String LABEL_DATA_URL_PATH = "/unlogin/fjtj/labeldata";

    private static final String GET_UNION_ON_URL_PATH = "/unlogin/fjtj/getunionno";

    private static final String USER_LOG_URL_PATH = "/activitylogapi/userlog";

    private static final String ACTIVITY_LOG_URL_PATH = "/activitylogapi/activitylog";

    private static final int FIVE_SECONDS = 5*1000;

    private static final int TEN_SECONDS = 10*1000;

    private static final RequestConfig requestConfig;

    static {
        //http请求超时配置
        requestConfig = RequestConfig.custom().setConnectTimeout(FIVE_SECONDS).setSocketTimeout(TEN_SECONDS).setConnectionRequestTimeout(500).build();
    }

    @Resource
    private FuJianBocConfig fuJianBocConfig;

    @Resource(name = "httpClient")
    private CloseableHttpClient httpClient;

    @Override
    public JSONObject duibaDataUpload(List<IdentityDataDto> dataDtoList) throws IOException {
        List<JSONObject> jsonObjectList = dataDtoList.stream().map(identityDataDto -> {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("openid", identityDataDto.getOpenId());
            jsonObject.put("mobile", identityDataDto.getMobile());
            jsonObject.put("union_no", identityDataDto.getUnionNo());
            return jsonObject;
        }).collect(Collectors.toList());
        JSONObject paramJsonObject = new JSONObject();
        paramJsonObject.put("datalist",jsonObjectList);
        paramJsonObject.put("appid",fuJianBocConfig.getAppId());
        paramJsonObject.put("appsecret",fuJianBocConfig.getAppSecret());
        String url = fuJianBocConfig.getUrl() + DUIBA_DATA_UPLOAD_PATH;
        String jsonStr = JSONObject.toJSONString(paramJsonObject);
        log.info("福建中行实名数据批量上送,data:{}", jsonStr);
        JSONObject jsonObject = sendRequest(jsonStr,url,HttpMethod.POST);
        if (null == jsonObject){
            return new JSONObject();
        }
        return assembleResult(jsonObject);
    }

    @Override
    public JSONObject wxDataExport(PublicDataReqParam publicDataReqParam) throws IOException {
        publicDataReqParam.setAppid(fuJianBocConfig.getAppId());
        publicDataReqParam.setAppsecret(fuJianBocConfig.getAppSecret());
        String jsonStr = JSON.toJSONString(publicDataReqParam);
        String url = fuJianBocConfig.getUrl() + WX_DATA_URL_PATH;
        JSONObject jsonObject = sendRequest(jsonStr,url,HttpMethod.POST);
        if (null == jsonObject){
            return new JSONObject();
        }
        log.info("公众号数据上送请求入参：{}",JSONObject.toJSONString(publicDataReqParam));
        return assembleResult(jsonObject);
    }

    @Override
    public JSONObject labelDataExport(LabelDataReqParam dataRerParam) throws IOException {
        dataRerParam.setAppid(fuJianBocConfig.getAppId());
        dataRerParam.setAppsecret(fuJianBocConfig.getAppSecret());
        String jsonStr = JSON.toJSONString(dataRerParam);
        String url = fuJianBocConfig.getUrl() + LABEL_DATA_URL_PATH;
        log.info("外部第三方标签数据上送请求入参");
        JSONObject jsonObject = sendRequest(jsonStr,url,HttpMethod.POST);
        if (null == jsonObject){
            log.info("外部第三方标签数据上送请求入参返回为空");
            return new JSONObject();
        }
        log.info("外部第三方标签数据上送请求入参成功：{}",jsonObject.getString("msgcode"));
        return assembleResult(jsonObject);
    }

    @Override
    public JSONObject getUnionNo(UnionNoReqParam reqParam) throws IOException {
        String jsonStr = JSON.toJSONString(reqParam);
        String url = fuJianBocConfig.getUrl() + GET_UNION_ON_URL_PATH;
        JSONObject jsonObject = sendRequest(jsonStr,url,HttpMethod.POST);
        if (null == jsonObject){
            return new JSONObject();
        }
        return assembleResult(jsonObject);
    }

    @Override
    public String verifyUser(String openId) {
        String s = Base64.getEncoder().encodeToString(openId.getBytes());
        return fuJianBocConfig.getVerifyUserUrl() + "?openid=" + s;
    }

    @Override
    public JSONObject sendUserJoinActivityDailyData(List<UserJoinActivityDataDto> userJoinActivityDataList) {
        Map paramMap = Maps.newHashMap();
        paramMap.put("appkey",fuJianBocConfig.getWeChatAppKey());
        paramMap.put("appsecret",fuJianBocConfig.getWeChatAppSecret());
        paramMap.put("datalist",JSONObject.toJSONString(userJoinActivityDataList));
        paramMap.put("timestamp",System.currentTimeMillis());
        String sign = SecureUtil.signParams(DigestAlgorithm.MD5, paramMap, "&", "=", true);
        paramMap.put("sign",sign);
        paramMap.remove("appsecret");
        String urlPath = fuJianBocConfig.getWeChatUrl() + USER_LOG_URL_PATH;
        HttpPost httpPost = new HttpPost(urlPath);
        httpPost.setConfig(requestConfig);
        StringEntity se = new StringEntity(JSONObject.toJSONString(paramMap), "utf-8");
        httpPost.setHeader("Content-Type", "application/json; charset=UTF-8");
        log.info("福建中行每日用户数据上送请求，requestUrl：{}，params：{}", urlPath, JSONObject.toJSONString(paramMap));
        httpPost.setEntity(se);
        String res = fjBocRequest(httpPost);
        return JSONObject.parseObject(res);
    }

    @Override
    public JSONObject sendUserPvAndUvDailyData(List<UserPvAndUvDataDto> userPvAndUvDataList) {
        Map paramMap = Maps.newHashMap();
        paramMap.put("appkey",fuJianBocConfig.getWeChatAppKey());
        paramMap.put("appsecret",fuJianBocConfig.getWeChatAppSecret());
        paramMap.put("datalist",JSONObject.toJSONString(userPvAndUvDataList));
        paramMap.put("timestamp",System.currentTimeMillis());
        String sign = SecureUtil.signParams(DigestAlgorithm.MD5, paramMap, "&", "=", true);
        paramMap.put("sign",sign);
        paramMap.remove("appsecret");
        String urlPath = fuJianBocConfig.getWeChatUrl() + ACTIVITY_LOG_URL_PATH;
        HttpPost httpPost = new HttpPost(urlPath);
        httpPost.setConfig(requestConfig);
        StringEntity se = new StringEntity(JSONObject.toJSONString(paramMap), "utf-8");
        httpPost.setHeader("Content-Type", "application/json; charset=UTF-8");
        log.info("福建中行活动PV/UV数据上送请求，requestUrl：{}，params：{}", urlPath, JSONObject.toJSONString(paramMap));
        httpPost.setEntity(se);
        String res = fjBocRequest(httpPost);
        return JSONObject.parseObject(res);
    }

    private JSONObject sendRequest(String jsonStr, String urlPath, HttpMethod httpMethod) throws IOException {
        byte[] dataBytes = jsonStr.getBytes("UTF-8");
        String encResultData = getEncResultData(dataBytes);
        String signatureSign = getSignatureSign(dataBytes);
        JSONObject jsonObject = new JSONObject();
        jsonObject.put(FjBocConstant.ENCORE_RESULT, encResultData);
        jsonObject.put(FjBocConstant.SIGN_DATA, signatureSign);
        Map<String, String> params = JSONObject.parseObject(jsonObject.toJSONString(), new TypeReference<Map<String, String>>(){});
        String res = "";
        if (httpMethod == HttpMethod.POST){
            HttpPost httpPost = new HttpPost(urlPath);
            httpPost.setConfig(requestConfig);
            StringEntity se = new StringEntity(JSONObject.toJSONString(params), "utf-8");
            httpPost.setHeader("Content-Type", "application/json; charset=UTF-8");
            log.info("福建中行请求，requestUrl：{},jsonStr:{}", urlPath, jsonStr);
            httpPost.setEntity(se);
            res = fjBocRequest(httpPost);
        }else {
            String assembleUrl = AssembleTool.assembleUrl(urlPath,params);
            log.info("福建中行-请求url，url:{},jsonstr:{}", assembleUrl, jsonStr);
            HttpGet httpGet = new HttpGet(assembleUrl);
            httpGet.setConfig(requestConfig);
            res = fjBocRequest(httpGet);
        }
        log.info("福建中行请求结果，res：{}", res);
        return JSONObject.parseObject(res);
    }

    private String fjBocRequest(HttpUriRequest httpRequest) {
        try(final CloseableHttpResponse response = httpClient.execute(httpRequest)) {
            // 响应参数
            String responseStr = EntityUtils.toString(response.getEntity());
            return responseStr;
        } catch (Exception e) {
            log.warn("福建中行响应异常，url:{}",httpRequest.getURI().getPath(), e);
        }
        return null;
    }

    /**
     * 对请求参数进行公钥加密
     *
     * @param dataBytes
     * @return
     */
    private String getEncResultData(byte[] dataBytes) {
        String shPublicKey = fuJianBocConfig.getSHPublicKey();
        //加密
        if (shPublicKey.length() == 130) {
            shPublicKey = shPublicKey.substring(2);
        }
        String shxHex = shPublicKey.substring(0, 64);
        String shhyHex = shPublicKey.substring(64, 128);
        ECPublicKeyParameters ecPublicKeyParameters = BCUtil.toSm2Params(shxHex, shhyHex);
        SM2 sm2 = new SM2(null, ecPublicKeyParameters);
        sm2.usePlainEncoding();
        sm2.setMode(SM2Engine.Mode.C1C2C3);
        byte[] encData = sm2.encrypt(dataBytes, KeyType.PublicKey);
        return ConverTool.parseByte2HexStr(encData);
    }

    /**
     * 对请求参数进行私钥加签
     *
     * @param dataBytes
     * @return
     * @throws IOException
     */
    private String getSignatureSign(byte[] dataBytes) throws IOException {
        //加签
        //sm3摘要(解密后的body体)+平台签名信息,组成签名数据，signinfo串需要行方提供
        String sm2SignTextData = SM3.byteArrayToHexString(SM3.hash(dataBytes)) + fuJianBocConfig.getAppId();
        byte[] sm2SignDataBytesData = sm2SignTextData.getBytes("UTF-8");
        ECPrivateKeyParameters privateKeyParameters = BCUtil.toSm2Params(fuJianBocConfig.getPTPrivateKeyHex());
        //创建sm2 对象
        SM2 sm22 = new SM2(privateKeyParameters, null);
        //这里需要手动设置，sm2 对象的默认值与我们期望的不一致 , 使用明文编码
        sm22.usePlainEncoding();
        sm22.setMode(SM2Engine.Mode.C1C2C3);
        byte[] sign = sm22.sign(sm2SignDataBytesData, null);
        //签名
        return HexUtil.encodeHexStr(sign);
    }

    /**
     * 对响应参数进行平台私钥解密
     *
     * @param encResultData
     * @return
     * @throws UnsupportedEncodingException
     */
    private String decodeResult(String encResultData) throws UnsupportedEncodingException {
        ECPrivateKeyParameters ecPrivateKeyParameters = BCUtil.toSm2Params(fuJianBocConfig.getPTPrivateKeyHex());
        SM2 sm2pri = new SM2(ecPrivateKeyParameters, null);
        sm2pri.usePlainEncoding();
        sm2pri.setMode(SM2Engine.Mode.C1C2C3);
        byte[] decdata = sm2pri.decrypt(ConverTool.parseHexStr2Byte(encResultData), KeyType.PrivateKey);
        return new String(decdata, "UTF-8");
    }

    /**
     * 对响应参数进行商户公钥验签
     *
     * @param decodeBody
     * @param signatureSign
     * @return
     * @throws IOException
     */
    private boolean verifyResult(String decodeBody, String signatureSign) throws IOException {
        //这里需要根据公钥的长度进行加工
        String shPublicKey = fuJianBocConfig.getSHPublicKey();
        if (shPublicKey.length() == 130) {
            //这里需要去掉开始第一个字节 第一个字节表示标记
            shPublicKey = shPublicKey.substring(2);
        }
        String xhex = shPublicKey.substring(0, 64);
        String yhex = shPublicKey.substring(64, 128);
        ECPublicKeyParameters params = BCUtil.toSm2Params(xhex, yhex);

        //创建sm2 对象
        SM2 sm2b = new SM2(null, params);
        //这里需要手动设置，sm2 对象的默认值与我们期望的不一致 , 使用明文编码
        sm2b.usePlainEncoding();
        sm2b.setMode(SM2Engine.Mode.C1C2C3);

        //sm3摘要(解密后的body体)+appid信息,组成签名数据,验证商户签名数据是否正确
        String sm2signtext = SM3.byteArrayToHexString(SM3.hash(decodeBody.getBytes("UTF-8"))) + fuJianBocConfig.getSignInfo();

        byte[] sm2signdataBytes = sm2signtext.getBytes("UTF-8");

        return sm2b.verify(sm2signdataBytes, HexUtil.decodeHex(signatureSign));
    }

    /**
     * 组装返回参数
     * @param jsonObject
     * @return
     * @throws IOException
     */
    private JSONObject assembleResult(JSONObject jsonObject) throws IOException {
        JSONObject result = new JSONObject();
        String decodeBody = "";
        if (!FjBocConstant.SUCCESS_RESPONSE_CODE.equals(jsonObject.getString(FjBocConstant.MESSAGE_CODE))) {
            return jsonObject;
        }
        if (!jsonObject.containsKey(FjBocConstant.SIGN_DATA)) {
            return jsonObject;
        }
        if (jsonObject.containsKey(FjBocConstant.ENCORE_RESULT)) {
            String encresult = jsonObject.getString(FjBocConstant.ENCORE_RESULT);
            decodeBody = decodeResult(encresult);
            result = JSON.parseObject(decodeBody);
            result.put(FjBocConstant.MESSAGE_CODE,jsonObject.getString(FjBocConstant.MESSAGE_CODE));
        }
        if (StringUtils.isBlank(decodeBody) || !verifyResult(decodeBody, jsonObject.getString(FjBocConstant.SIGN_DATA))) {
            log.error("验签失败");
            jsonObject.put("msgcode", "E0S");
            jsonObject.put("msg", "验签失败,签名数据错误");
            return jsonObject;
        } else {
            return result;
        }
    }



}
