package cn.com.duiba.service.virtualsupplierprocess.customHttpFactory;

import cn.com.duiba.boot.exception.BizException;
import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.impl.client.DefaultConnectionKeepAliveStrategy;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.impl.nio.client.HttpAsyncClients;
import org.apache.http.impl.nio.conn.PoolingNHttpClientConnectionManager;
import org.apache.http.impl.nio.reactor.DefaultConnectingIOReactor;
import org.apache.http.nio.conn.NoopIOSessionStrategy;
import org.apache.http.nio.conn.SchemeIOSessionStrategy;
import org.apache.http.nio.conn.ssl.SSLIOSessionStrategy;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import java.security.cert.X509Certificate;

@Configuration
public class XiangShangHttpClientFactory {

    private static final Logger LOGGER = LoggerFactory.getLogger(XiangShangHttpClientFactory.class);

    /**
     * 连接超时，连接目标url最大超时
     */
    private static final int CONNECT_TIMEOUT = 10 * 1000;

    /**
     * 从连接池中获取连接的超时时间
     */
    private static final int DEFAULT_REQUEST_TIMEOUT = 3 * 1000;

    /**
     * 长链接空闲时间
     */
    private static final int KEEPALIVE_TIMEOUT = 5 * 1000;

    /**
     * 处理超时，等待响应（读数据）最大超时
     */
    private static final int SOCKET_TIMEOUT = 10 * 1000;

    /**
     * 连接池中的最大连接数
     */
    private static final int MAX_CONNECT = 100;

    /**
     * 每个路由最大连接{并发}值，连接同一个route最大的并发数
     */
    private static final int MAX_ROUTE_CONNECT = 20;

    /**
     * 广发中心异步连接池
     * @return
     */
    @Bean(initMethod="start", destroyMethod = "close", name="xiangShangHttpAsyncClient")
    public CloseableHttpAsyncClient cgbHttpAsyncClient(){
        RequestConfig config = RequestConfig.custom()
                .setConnectTimeout(CONNECT_TIMEOUT)
                .setConnectionRequestTimeout(DEFAULT_REQUEST_TIMEOUT)
                .setSocketTimeout(CONNECT_TIMEOUT)
                .build();
        CloseableHttpAsyncClient httpClient = HttpAsyncClients.custom()
                .setConnectionManager(getConnectionManager())
                .setDefaultRequestConfig(config)
                .setMaxConnTotal(MAX_CONNECT)
                .setMaxConnPerRoute(MAX_ROUTE_CONNECT)
                .setKeepAliveStrategy(getKeepAliveStrategy())
                .build();
        return httpClient;
    }

    /**
     * 获取ConnectionManager
     * @return
     */
    private PoolingNHttpClientConnectionManager getConnectionManager() {
        try {
            SSLContext sslContext = getSSLContext();
            if(sslContext == null) {
                throw new BizException("getConnectionManager failed, sslContext is null!!!");
            }
            return new PoolingNHttpClientConnectionManager(
                    new DefaultConnectingIOReactor()
                    , RegistryBuilder.<SchemeIOSessionStrategy>create()
                    .register("http", NoopIOSessionStrategy.INSTANCE)
                    .register("https", new SSLIOSessionStrategy(sslContext, NoopHostnameVerifier.INSTANCE))
                    .build()
            );
        } catch (Exception e) {
            LOGGER.warn("", e);
            return null;
        }
    }

    /**
     * 获取SSLContext
     * @return
     */
    private SSLContext getSSLContext() {
        SSLContext sslContext = null;
        try {
            TrustManager[] trustAllCerts = new TrustManager[]{
                    new X509TrustManager() {
                        @Override
                        public X509Certificate[] getAcceptedIssuers() {
                            return null;
                        }

                        @Override
                        public void checkClientTrusted(X509Certificate[] certs, String authType) {
                            // don't check
                        }

                        @Override
                        public void checkServerTrusted(X509Certificate[] certs, String authType) {
                            // don't check
                        }
                    }
            };
            sslContext = SSLContext.getInstance("TLS");
            sslContext.init(null, trustAllCerts, null);
        } catch (Exception e) {
            LOGGER.error("获取SSLContext异常", e);
        }
        return sslContext;
    }

    /**
     * 长连接策略
     * @return
     */
    private static DefaultConnectionKeepAliveStrategy getKeepAliveStrategy() {
        return new DefaultConnectionKeepAliveStrategy() {
            @Override
            public long getKeepAliveDuration(HttpResponse response, HttpContext context) {
                long duration = super.getKeepAliveDuration(response, context);
                if (duration == -1) {
                    return KEEPALIVE_TIMEOUT;
                }
                return duration;
            }
        };
    }

}
