package cn.com.duiba.biz.credits;

import cn.com.duiba.api.bo.subcredits.SubCreditsMsgDto;
import cn.com.duiba.biz.Exception.ThirdpatyException;
import cn.com.duiba.dto.*;

import cn.com.duiba.domain.SubCreditsMsgWrapper;
import cn.com.duiba.notifycenter.domain.NotifyQueueDO;
import cn.com.duiba.order.center.api.dto.CreditsMessage;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.tool.AssembleTool;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.dataformat.xml.XmlMapper;
import com.fasterxml.jackson.dataformat.xml.ser.ToXmlGenerator;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.Map;
import java.util.Objects;

/**
 * Created by fangdong on 2019/07/22
 * 苏州农业商业银行
 */
@Service
public class WjrcbApi {
    private static final Logger log = LoggerFactory.getLogger(WjrcbApi.class);
    private static final String SUCCESS_CODE = "000000";

    private XmlMapper xmlMapper = new XmlMapper();

    @PostConstruct
    public void initXmlMapper() {
        // 添加xml头
        xmlMapper.configure(ToXmlGenerator.Feature.WRITE_XML_DECLARATION, true);
        // 忽略未知属性
        xmlMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    public boolean isWjrcb(Long appId) {
        return Objects.equals(62363L, appId);
    }

    /**
     * rpc减积分
     */
    public HttpRequestBase getSubCreditsHttpRequest(CreditsMessage message) {
        String url = message.getHttpUrl();
        String newUrl = this.getNewUrl(url);
        String paramsStr = url.substring(url.indexOf('?') + 1);
        Map<String, String> params = AssembleTool.getUrlParams(paramsStr);

        message.setHttpType(CreditsMessageDto.HTTP_POST);
        message.setHttpUrl(newUrl);
        message.setAuthParams(params);

        return this.getSubCreditsHttpPost(message.getHttpUrl(), params);
    }

    /**
     * mq减积分
     */
    public HttpRequestBase getMqSubCreditsHttpRequest(SubCreditsMsgWrapper message) {
        String url = message.getHttpUrl();
        String newUrl = url.substring(0, url.indexOf('?'));
        String paramsStr = url.substring(url.indexOf('?') + 1);
        Map<String, String> params = AssembleTool.getUrlParams(paramsStr);

        message.getSubCreditsMsg().setHttpType(SubCreditsMsgDto.HTTP_POST);
        message.setHttpUrl(newUrl);
        message.getSubCreditsMsg().setAuthParams(params);

        return this.getSubCreditsHttpPost(message.getHttpUrl(), params);
    }

    /**
     * 解析积分响应
     * @param body 积分响应
     * @return 积分响应
     */
    public String parseCreditsRsp(String body) {
        log.info("苏州农商扣积分response, body={}", body);
        WjrcbResponseMessage<WjrcbCreditsResultDto> message;
        try {
            message = xmlMapper.readValue(body, new TypeReference<WjrcbResponseMessage<WjrcbCreditsResultDto>>() {});
        } catch (Exception e) {
            log.error("苏州农商xml反序列化失败, body={}", body, e);
            return body;
        }

        JSONObject json = new JSONObject();
        json.put("errorMessage", message.getHead().getRejMsg());
        if (SUCCESS_CODE.equals(message.getHead().getRejCode()) && message.getBody() != null) {
            json.put("status", Objects.equals("S", message.getBody().getStatus()) ? "ok" : "fail");
            json.put("bizId", message.getBody().getTransSeq());
            json.put("credits", message.getBody().getCredits());
        } else {
            json.put("status", "fail");
        }

        return json.toString();
    }

    /**
     * 结果通知
     * @param notifyUrl 通知地址
     * @param record 通知记录
     * @return 通知请求
     */
    public HttpRequestBase getRequestNotify(String notifyUrl, NotifyQueueDO record) {
        String transfer = record.getTransfer();
        Map<String,String> transferMap = this.getTransferMap(transfer);

        WjrcbRequestMessage<WjrcbRequestNotifyDto> message = new WjrcbRequestMessage<>();
        // head
        WjrcbRequestMessageHead head = new WjrcbRequestMessageHead(new Date());
        head.setTransactionId("ResultNotify");
        if (MapUtils.isNotEmpty(transferMap)) {
            head.setIdType(transferMap.get("IdType"));
            head.setIdNo(transferMap.get("IdNo"));
        }

        message.setHead(head);
        // body
        WjrcbRequestNotifyDto body = new WjrcbRequestNotifyDto();
        body.setBizId(record.getDeveloperBizId());
        body.setStatus(Boolean.TRUE.equals(record.getResult()) ? "S" : "F");
        body.setUserId(record.getPartnerUserId());
        body.setOrderNum(record.getDuibaOrderNum());
        message.setBody(body);

        String content;
        try {
            content = xmlMapper.writeValueAsString(message);
        } catch (Exception e) {
            log.error("苏州农商xml序列化异常, content={}", JSON.toJSONString(message), e);
            throw new ThirdpatyException("xml序列化异常");
        }

        HttpPost request = new HttpPost(notifyUrl);
        request.addHeader("accept", "text/b2exml; charset=GBK");
        request.setEntity(new StringEntity(content, ContentType.create("application/stream")));

        return request;
    }

    public String getResponseNotify(String body) {
        log.info("苏州农商通知response, body={}", body);
        WjrcbResponseMessage<Object> message;
        try {
            message = xmlMapper.readValue(body, new TypeReference<WjrcbResponseMessage<Object>>() {});
        } catch (Exception e) {
            log.error("苏州农商xml反序列化失败, body={}", body, e);
            return body;
        }

        if (SUCCESS_CODE.equals(message.getHead().getRejCode())) {
            return "ok";
        }

        return body;
    }

    private String getNewUrl(String url) {
        return url.substring(0, url.indexOf('?'));
    }

    private HttpRequestBase getSubCreditsHttpPost(String httpUrl, Map<String, String> params) {
        String content = this.getSubCreditsMessage(params);

        HttpPost request = new HttpPost(httpUrl);
        request.addHeader("accept", "text/b2exml; charset=GBK");
        request.setEntity(new StringEntity(content, ContentType.create("application/stream")));

        return request;
    }

    private String getSubCreditsMessage(Map<String, String> params) {
        String transfer = params.get("transfer");
        Map<String,String> transferMap = this.getTransferMap(transfer);

        WjrcbRequestMessage<WjrcbCreditsRequestDto> request = new WjrcbRequestMessage<>();
        // head
        WjrcbRequestMessageHead head = new WjrcbRequestMessageHead(new Date());
        head.setTransactionId("PointAdjust");
        if (MapUtils.isNotEmpty(transferMap)) {
            head.setIdType(transferMap.get("IdType"));
            head.setIdNo(transferMap.get("IdNo"));
        }

        request.setHead(head);

        // body
        WjrcbCreditsRequestDto body = new WjrcbCreditsRequestDto();
        body.setOrderNum(params.get("orderNum"));
        body.setUserId(params.get("uid"));
        body.setCredits(params.get("credits"));
        body.setDescription(params.get("description"));
        body.setOrderType(params.get("type"));
        body.setOrderAmount(params.get("facePrice"));
        body.setActualPrice(params.get("actualPrice"));
        body.setParams(params.get("params"));
        body.setEvent("ExchangeEvent");

        request.setBody(body);

        try {
            return xmlMapper.writeValueAsString(request);
        } catch (Exception e) {
            log.error("苏州农商扣积分参数xml序列化异常, content={}", JSON.toJSONString(request), e);
            throw new ThirdpatyException("苏州农商扣积分参数xml序列化异常");
        }
    }

    /**
     * 获取transfer参数map
     * @param encodedTransfer 编码过的transfer
     * @return transfer参数map
     */
    private Map<String, String> getTransferMap(String encodedTransfer) {
        if (StringUtils.isEmpty(encodedTransfer)) {
            return null;
        }

        String transfer;
        try {
            transfer = URLDecoder.decode(encodedTransfer, StandardCharsets.UTF_8.toString());
        } catch (Exception e) {
            log.error("苏州农商transfer解码异常, transfer={}", encodedTransfer, e);
            return null;
        }

        return AssembleTool.getUrlParams(transfer);
    }
}
