package cn.com.duiba.biz.credits.strategy.Impl.kouweiwang;

import cn.com.duiba.biz.credits.strategy.ApiStrategy;
import cn.com.duiba.domain.SubCreditsMsgWrapper;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.dto.kouweiwang.KwwCreditsRespBody;
import cn.com.duiba.dto.kouweiwang.KwwCreditsRespData;
import cn.com.duiba.dto.kouweiwang.KwwNotifyResp;
import cn.com.duiba.notifycenter.domain.NotifyQueueDO;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.kouweiwang.KwwTool;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Splitter;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * 口味王接口订制
 *
 * @author fja
 */
@Service
public class KwwApiStrategy implements ApiStrategy {

    private final Logger log = LoggerFactory.getLogger(this.getClass());

    @Resource
    private KwwTool kwwTool;

    private static final int THREE_SECONDS = 3 * 1000;


    private final RequestConfig config;

    {
        //http请求超时配置
        config = RequestConfig.custom().setConnectTimeout(THREE_SECONDS).setSocketTimeout(THREE_SECONDS).setConnectionRequestTimeout(500).build();
    }

    /**
     * 目前扣积分只有 积分商城普兑（卡券）
     *
     * @return http对象
     */
    @Override
    public HttpRequestBase getMqSubCreditsHttpRequest(SubCreditsMsgWrapper message) {
        //请求链接 & 解析
        String url = kwwTool.getHostName(message.getHttpUrl());
        Map<String, String> originData = AssembleTool.getUrlParams(kwwTool.getParamUrl(message.getHttpUrl()));


        Long consumerId = message.getSubCreditsMsg().getConsumerId();
        log.info("[kww]减积分 cid = {} originData = {}", consumerId, JSON.toJSONString(originData));

        //拼接请求参数
        Map<String, Object> params = buildSubCreditsParams(originData);

        try {
            //加密
            String cipherText = kwwTool.aesEncrypt(JSON.toJSONString(params));
            log.info("[kww]减积分加密参数 = {}", cipherText);


            HttpPost httpPost = new HttpPost(url);
            httpPost.setEntity(new StringEntity(cipherText, ContentType.APPLICATION_JSON));
            httpPost.setConfig(config);


            message.setHttpUrl(url);
            message.getSubCreditsMsg().setAuthParams(params.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> String.valueOf(entry.getValue()))));

            return httpPost;
        } catch (Exception e) {
            log.error(String.format("[kww] cid = 【%s】 生成扣积分请求失败，originData = 【%s】", consumerId, JSON.toJSONString(originData)), e);
            throw new IllegalStateException(e);
        }
    }

    @Override
    public String parseCreditsRsp(String body, Boolean addCredits, Map<String, String> authParams) {
        JSONObject result = new JSONObject();

        //解密
        KwwCreditsRespBody kwwCreditsRespBody;
        try {
            String plainText = kwwTool.aesDecrypt(body);
            log.info("[kww]积分接口响应 = {}", plainText);

            kwwCreditsRespBody = JSON.parseObject(plainText, KwwCreditsRespBody.class);
        } catch (Exception e) {
            log.error(String.format("[kww] 解析积分响应 解密异常 authParams = %s", JSON.toJSONString(authParams)), e);
            throw new IllegalStateException(e);
        }


        if (KwwCreditsRespBody.SUCCESS.equals(kwwCreditsRespBody.getFlag())) {
            result.put("status", "ok");

            KwwCreditsRespData respData = kwwCreditsRespBody.getKwwCreditsRespData();
            if (Objects.nonNull(respData) && Objects.nonNull(respData.getAmount())) {
                //返回用户最新积分
                result.put("credits", respData.getAmount());
                // 开发者不返回订单号，自定义
                result.put("bizId", System.currentTimeMillis() + RandomStringUtils.randomNumeric(6));
            } else {
                log.warn(String.format("[kww] authParams = 【%s】 无积分返回 respData = 【%s】", JSON.toJSONString(authParams), JSON.toJSONString(respData)));
            }
        } else {
            result.put("status", "fail");
            result.put("errorMessage", kwwCreditsRespBody.getMsg());
        }


        return result.toJSONString();
    }

    @Override
    public HttpRequestBase getAddCreditsMessageRequest(CreditsMessageDto message) {
        //请求链接 & 解析
        String url = kwwTool.getHostName(message.getHttpUrl());
        Map<String, String> originData = AssembleTool.getUrlParams(kwwTool.getParamUrl(message.getHttpUrl()));


        String consumerId = message.getConsumerId();
        log.info("[kww]加积分 cid = {} originData = {}", consumerId, JSON.toJSONString(originData));


        //拼接请求参数
        boolean isProjectx = "projectX".equals(message.getRelationType());
        Map<String, Object> params = buildAddCreditsParams(originData, isProjectx);

        try {
            //加密
            String cipherText = kwwTool.aesEncrypt(JSON.toJSONString(params));
            log.info("[kww]加积分 cid = {} 加密参数 = {}", consumerId, cipherText);


            HttpPost httpPost = new HttpPost(url);
            httpPost.setEntity(new StringEntity(cipherText, ContentType.APPLICATION_JSON));
            httpPost.setConfig(config);


            message.setHttpUrl(url);
            message.setAuthParams(params.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> String.valueOf(entry.getValue()))));

            return httpPost;
        } catch (Exception e) {
            log.error(String.format("[kww]加积分 cid = 【%s】 生成加积分请求失败，originData = 【%s】", consumerId, JSON.toJSONString(originData)), e);
            throw new IllegalStateException(e);
        }
    }

    @Override
    public HttpRequestBase getVirtualRequest(SupplierRequest request) {
        //口味王虚拟商品兑换走加积分
        //请求链接 & 解析
        String url = kwwTool.getHostName(request.getHttpUrl());
        Map<String, String> originData = AssembleTool.getUrlParams(kwwTool.getParamUrl(request.getHttpUrl()));
        log.info("[kww]虚拟商品兑换originData = {}", JSON.toJSONString(originData));

        String merchantCode = Optional.ofNullable(originData.get("params")).orElse("");
        if (!merchantCode.startsWith("JF")) {
            throw new IllegalStateException(String.format("虚拟商品商家编码异常，originData = %s", JSON.toJSONString(originData)));
        }


        Map<String, Object> params = buildVirtualExchangeParams(originData, merchantCode);

        try {
            String cipherText = kwwTool.aesEncrypt(JSON.toJSONString(params));
            log.info("[kww]虚拟商品兑换 cid = {} 加密参数 = {}", request.getConsumerId(), cipherText);

            HttpPost httpPost = new HttpPost(url);
            httpPost.setEntity(new StringEntity(cipherText, ContentType.APPLICATION_JSON));
            httpPost.setConfig(config);

            request.setHttpUrl(url);
            request.setAuthParams(params.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> String.valueOf(entry.getValue()))));
            return httpPost;
        } catch (Exception e) {
            log.error(String.format("[kww]虚拟商品兑换 cid = 【%s】 生成虚拟商品兑换请求失败，originData = 【%s】", request.getConsumerId(), JSON.toJSONString(originData)), e);
            throw new IllegalStateException(e);
        }
    }


    @Override
    public String getVirtualResponse(SupplierRequest request, String body) {
        JSONObject result = new JSONObject();

        //解密
        KwwCreditsRespBody kwwCreditsRespBody;
        try {
            String plainText = kwwTool.aesDecrypt(body);
            log.info("[kww]虚拟商品兑换接口响应 = {}", plainText);

            kwwCreditsRespBody = JSON.parseObject(plainText, KwwCreditsRespBody.class);
        } catch (Exception e) {
            log.error(String.format("[kww] 解析虚拟商品响应 解密异常 authParams = %s  body = %s", JSON.toJSONString(request.getAuthParams()), body), e);
            throw new IllegalStateException(e);
        }


        if (KwwCreditsRespBody.SUCCESS.equals(kwwCreditsRespBody.getFlag())) {
            result.put("status", "success");

            KwwCreditsRespData respData = kwwCreditsRespBody.getKwwCreditsRespData();
            if (Objects.nonNull(respData) && Objects.nonNull(respData.getAmount())) {
                //返回用户最新积分
                result.put("credits", respData.getAmount());
            }
        } else {
            result.put("status", "fail");
            result.put("errorMessage", kwwCreditsRespBody.getMsg());
        }


        return result.toJSONString();
    }

    @Override
    public HttpRequestBase getRequestNotify(String notifyUrl, NotifyQueueDO record) {
        HttpPost httpPost = new HttpPost(notifyUrl);
        String partnerUserId = record.getPartnerUserId();

        try {
            Map<String, String> params = buildNotifyParams(record);
            log.info("[kww]通知请求 = {}", JSON.toJSONString(params));

            String cipherText = kwwTool.aesEncrypt(JSON.toJSONString(params));
            //生成HttpPost
            httpPost.setEntity(new StringEntity(cipherText, ContentType.APPLICATION_JSON));
            httpPost.setConfig(config);


            return httpPost;
        } catch (Exception e) {
            log.error(String.format("[kww]通知 uid =【%s】 生成notify请求失败，NotifyQueueDO = 【%s】", partnerUserId, JSON.toJSONString(record)), e);
            throw new IllegalStateException(e);
        }
    }

    @Override
    public String getResponseNotify(String body) {
        //解密
        KwwNotifyResp kwwNotifyResp;
        try {
            String plainText = kwwTool.aesDecrypt(body);
            log.info("[kww]通知接口响应 = {}", plainText);

            kwwNotifyResp = JSON.parseObject(plainText, KwwNotifyResp.class);
        } catch (Exception e) {
            log.error(String.format("[kww] 解析通知响应 解密异常 body = %s", body), e);
            throw new IllegalStateException(e);
        }

        if (KwwNotifyResp.SUCCESS.equals(kwwNotifyResp.getFlag())) {
            return "ok";
        }

        //TODO
        return body;
    }


    /**
     * 生成口味王的扣积分请求参数
     *
     * @param originData 数据
     * @return 扣积分请求参数
     */
    private Map<String, Object> buildSubCreditsParams(Map<String, String> originData) {
        Map<String, Object> params = Maps.newHashMap();


        params.put("uid", originData.get("uid"));
        params.put("amount", Integer.parseInt(originData.get("credits")));
        params.put("actionDesc", originData.get("description"));
        params.put("busNo", originData.get("orderNum"));
        params.put("type", originData.get("type"));


        return params;
    }


    /**
     * 生成口味王的加积分请求参数
     *
     * @param originData 数据
     * @return 加积分请求参数
     */
    private Map<String, Object> buildAddCreditsParams(Map<String, String> originData, boolean isProjectx) {
        Map<String, Object> params = Maps.newHashMap();


        params.put("uid", originData.get("uid"));
        params.put("amount", Integer.parseInt(originData.get("credits")));
        params.put("busNo", originData.get("orderNum"));

        //固定传task
        params.put("type", "task");

        //星速台的加积分描述 已 @ 分割description@subType
        if (isProjectx) {
            String desc = originData.get("description");
            List<String> splitToList = Splitter.on("@").omitEmptyStrings().splitToList(desc);

            params.put("actionDesc", splitToList.get(0));
            params.put("subType", splitToList.get(1));
        } else {
            params.put("actionDesc", originData.get("description"));
        }

        return params;
    }


    /**
     * 生成口味王的虚拟商品兑换请求参数
     *
     * @param originData   数据
     * @param merchantCode 商家变吗
     * @return 虚拟商品兑换请求参数
     */
    private Map<String, Object> buildVirtualExchangeParams(Map<String, String> originData, String merchantCode) {
        Map<String, Object> params = Maps.newHashMap();


        params.put("uid", originData.get("uid"));
        String amountStr = merchantCode.replace("JF", "");
        params.put("amount", Integer.parseInt(amountStr));
        params.put("busNo", originData.get("orderNum"));
        //固定传recharge
        params.put("type", "recharge");
        params.put("actionDesc", originData.get("description"));


        return params;
    }

    /**
     * 生成通知businessObject对象
     *
     * @param record 通知DO
     * @return bizObject对象
     */
    private Map<String, String> buildNotifyParams(NotifyQueueDO record) {
        Map<String, String> bizParams = Maps.newHashMap();

        bizParams.put("uid", record.getPartnerUserId());
        bizParams.put("status", isExchangeSuccess(record));
        bizParams.put("orderNo", record.getDuibaOrderNum());
        bizParams.put("msg", record.getError4developer());

        return bizParams;
    }


    /**
     * 订单是否成功
     *
     * @param record 记录
     * @return 00: 兑换失败 10: 兑换成功
     */
    private String isExchangeSuccess(NotifyQueueDO record) {
        return BooleanUtils.isTrue(record.getResult()) ? "10" : "00";
    }
}
