package cn.com.duiba.service.virtual.impl;

import cn.com.duiba.biz.Exception.ThirdpatyException;
import cn.com.duiba.constant.BiliBiliConfig;
import cn.com.duiba.service.impl.AbstractDuibaVirtualSupplier;
import cn.com.duiba.service.virtual.impl.dto.BiliBiliResponse;
import cn.com.duiba.thirdparty.dto.SupplierRequestDto;
import cn.com.duiba.thirdparty.enums.virtual.VirtualItemChannelEnum;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import javax.crypto.Cipher;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.DESKeySpec;
import java.security.SecureRandom;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * @Auther: dingshitai
 * @Date: 2021/7/19 15:24
 * @Description:
 */
@Service
public class BiliBiliApiStrategy extends AbstractDuibaVirtualSupplier {
    /**
     * 日志
     */
    private static final Logger log = LoggerFactory.getLogger(BiliBiliApiStrategy.class);

    private static final String LOGGER_PREFIX = "bilibili虚拟商品对接";

    private static final String SUCCESS = "0000";

    private static final String FAIL = "0001";

    /**
     * 相关配置
     */
    @Autowired
    private BiliBiliConfig biliBiliConfig;


    private static final String Error4ConsumerMessage = "出了点小问题，请重新下单";

    /**
     * 构造请求
     *
     * @param request
     * @return
     */
    @Override
    public HttpRequestBase getVirtualRequest(SupplierRequestDto request) {
        // 解析出url请求中的参数
        Map<String, String> originData = request.getParams();
        // 根据请求参数组装成密文
        String dataDes = createParam(originData);
        if(StringUtils.isEmpty(dataDes)){
            throw new ThirdpatyException(LOGGER_PREFIX+"加密密文失败");
        }
        // 构建请求的入参
        Map<String, String> params = Maps.newHashMap();
        params.put("mid",biliBiliConfig.getMid());
        params.put("data",dataDes);
        HttpPost httpPost = new HttpPost(biliBiliConfig.getSubmitOrderUrl()+"?mid="+biliBiliConfig.getMid()+"&data="+dataDes);
        httpPost.setEntity(new StringEntity(JSON.toJSONString(params), ContentType.APPLICATION_FORM_URLENCODED));
        return httpPost;
    }

    /**
     * 调用bilibili订单接口返回结果判断解析
     *
     * @param request
     * @param body
     * @return
     */
    @Override
    public String getVirtualResponse(SupplierRequestDto request, String body) {
        Map<String,String> duibaDoc = Maps.newHashMap();
        if(StringUtils.isEmpty(body)){
            log.error(LOGGER_PREFIX+"请求下单失败！请求订单Id:{},返回的response解析为空",request.getOrderId());
            duibaDoc.put("status", "fail");
            return JSONObject.toJSONString(duibaDoc);
        }
        BiliBiliResponse biliBiliResponse = JSON.parseObject(body, BiliBiliResponse.class);
        if(SUCCESS.equals(biliBiliResponse.getResult())){
            log.info(LOGGER_PREFIX+"请求下单成功！请求订单Id:{},response={}",request.getOrderId(),JSONObject.toJSONString(biliBiliResponse));
            duibaDoc.put("status", "process");
            return JSONObject.toJSONString(duibaDoc);
        }else if(FAIL.equals(biliBiliResponse.getResult())){
            log.error(LOGGER_PREFIX+"请求下单失败！请求订单Id:{},失败原因是:{}",request.getOrderId(),biliBiliResponse.getMsg());
            duibaDoc.put("status", "fail");
            return JSONObject.toJSONString(duibaDoc);
        }else{
            log.error(LOGGER_PREFIX+"请求下单失败！出现未知回复,请求订单Id:{},response={}",request.getOrderId(),JSONObject.toJSONString(biliBiliResponse));
            duibaDoc.put("status", "fail");
            return JSONObject.toJSONString(duibaDoc);
        }
    }

    /**
     * 根据数据和秘钥进行请求入参的组装
     * @param originData
     * @return
     */
    private String createParam(Map<String, String> originData){
        String dataDes = null;
        JSONObject js = new JSONObject();
        String secretKey = biliBiliConfig.getSecretKey();// 密钥
        String mid = biliBiliConfig.getMid();// 商户ID
        // 下游回调地址
        String notifyUrl = biliBiliConfig.getNotifyUrl();
        try{
            // 校验参数
            checkParam(originData);
            SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmss");
            String date = sdf.format(new Date());// 日期
            String phone = originData.get("account");// 手机号
            String size = originData.get("bizParams");// 产品编码
            String snum = originData.get("orderNum");// 下游订单号
            js.put("mid", mid);
            js.put("date", date);
            js.put("phone", phone);
            js.put("size", size);
            js.put("snum", snum);
            js.put("notifyUrl", notifyUrl);
            log.info(LOGGER_PREFIX+js.toJSONString());
            dataDes = BiliBiliUtils.encrypt(js.toJSONString(),secretKey);
        } catch (Exception e) {
            log.error(LOGGER_PREFIX+"加密失败！秘钥:{},加密内容:{}",secretKey,js.toJSONString());
            return null;
        }
        return dataDes;
    }

    @Override
    public String getVirtualTypeCode() {
        return VirtualItemChannelEnum.BILIBILI.getCode();
    }

    /**
     * 校验参数
     * @param originData
     */
    private void checkParam(Map<String, String> originData){
        String phone = originData.get("account");// 手机号
        if(StringUtils.isEmpty(phone)){
            log.error(LOGGER_PREFIX+"手机号不能为空");
            throw new ThirdpatyException(LOGGER_PREFIX+"手机号不能为空");
        }
        String size = originData.get("bizParams");// 产品编码
        if(StringUtils.isEmpty(size)){
            log.error(LOGGER_PREFIX+"下单的产品编码不能为空");
            throw new ThirdpatyException(LOGGER_PREFIX+"下单的产品编码不能为空");
        }
        String snum = originData.get("orderNum");// 下游订单号
        if(StringUtils.isEmpty(snum)){
            log.error(LOGGER_PREFIX+"订单号不能为空");
            throw new ThirdpatyException(LOGGER_PREFIX+"订单号不能为空");
        }
    }

    /**
     * bilibili提供的加密方法
     */
    static class BiliBiliUtils {
        public static final String ALGORITHM = "DES";
        /**
         * 加密方法
         * @param data
         * @param pwd
         * @return
         * @throws Exception
         */
        public static String encrypt(String data, String pwd) throws Exception {
            return byte2hex(encrypt(data.getBytes(), pwd.getBytes()));
        }

        /**
         * 加密
         * @param data
         * @param key
         * @return
         * @throws Exception
         */
        private static byte[] encrypt(byte[] data, byte[] key) throws Exception {
            // DES算法要求有一个可信任的随机数源
            SecureRandom sr = new SecureRandom();
            // 从原始密匙数据创建DESKeySpec对象
            DESKeySpec dks = new DESKeySpec(key);
            // 创建一个密匙工厂，然后用它把DESKeySpec转换成
            // 一个SecretKey对象
            SecretKeyFactory keyFactory = SecretKeyFactory.getInstance(ALGORITHM);
            SecretKey secureKey = keyFactory.generateSecret(dks);
            // Cipher对象实际完成加密操作
            Cipher cipher = Cipher.getInstance(ALGORITHM);
            // 用密匙初始化Cipher对象
            cipher.init(Cipher.ENCRYPT_MODE, secureKey, sr);
            // 现在，获取数据并加密
            // 正式执行加密操作
            return cipher.doFinal(data);
        }

        /**
         * 转码方法
         * @param b
         * @return
         */
        public static String byte2hex(byte[] b) {
            String hs = "";
            String stmp = "";
            for (int n = 0; n < b.length; n++) {
                stmp = (java.lang.Integer.toHexString(b[n] & 0XFF));
                if (stmp.length() == 1)
                    hs = hs + "0" + stmp;
                else
                    hs = hs + stmp;
            }
            return hs.toUpperCase();
        }
    }

    @Override
    public String getDefaultError4ConsumerMessage() {
        return Error4ConsumerMessage;
    }

}
