package cn.com.duiba.tool.cgb;

import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1Encoding;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1Integer;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.gm.GMNamedCurves;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.sec.ECPrivateKey;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.asn1.x9.X9ECParameters;
import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.CryptoException;
import org.bouncycastle.crypto.InvalidCipherTextException;
import org.bouncycastle.crypto.KeyGenerationParameters;
import org.bouncycastle.crypto.engines.SM2Engine;
import org.bouncycastle.crypto.generators.ECKeyPairGenerator;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.crypto.params.ECKeyGenerationParameters;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.crypto.params.ParametersWithID;
import org.bouncycastle.crypto.params.ParametersWithRandom;
import org.bouncycastle.crypto.signers.SM2Signer;
import org.bouncycastle.math.ec.ECConstants;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.security.SecureRandom;

/**
 * sm2�㷨�����࣬�ṩ�ӽ��ܡ�ǩ����ǩ����
 * <p>��Ϊ���ݼӽ��ܶ��Ƕ��ֽ����ݼӽ��ܣ������Ҫע�����ǰ�ͽ��ܺ���ַ�������һ��
 * <p>��������˵�����ӿڽ��յĶ���ԭʼ�Ķ��������ݣ���hex����base64��������ݣ���ؽ���֮���ٴ�����
 * @author liangruxing-yfzx
 *
 */
public class SM2Util {

	
	private static final X9ECParameters sm2p256v1 = GMNamedCurves.getByName("sm2p256v1");
	private static final int SM3_DIGEST_LENGTH_32 = 32;
	private static final byte[] defaultUserID = "1234567812345678".getBytes();
	
	
	/**
	 * sm2����
	 * @param publicKey ��Կ�����������ݣ���������ģ�������ٴ��룬�类hex���룬��hex������ٴ�����
	 * @param data �����ܵ�����
	 * @return ����der�������������
	 * @throws InvalidCipherTextException
	 * @throws IOException
	 */
	public static byte[] encrypt(byte[] publicKey, byte[] data) throws InvalidCipherTextException, IOException
    {
		if(publicKey.length == 64) {//��λ���0x04����ʶδ��ѹ��
			byte tmp[] = new byte[65];
			System.arraycopy(publicKey, 0, tmp, 1, publicKey.length);
			tmp[0] = 0x04;
			publicKey = tmp;
		}
	    ECDomainParameters parameters = new ECDomainParameters(sm2p256v1.getCurve(), sm2p256v1.getG(), sm2p256v1.getN());
	    ECPublicKeyParameters pubKeyParameters = new ECPublicKeyParameters(sm2p256v1.getCurve().decodePoint(publicKey), parameters);
	    SM2Engine engine = new SM2Engine();
	    ParametersWithRandom pwr = new ParametersWithRandom(pubKeyParameters, new SecureRandom());
	    engine.init(true, pwr);
	    byte[] cipher = engine.processBlock(data, 0, data.length);
	    return encodeSM2CipherToDER(cipher);
    }

	/**
	 * sm2����
	 * @param privateKey ˽Կ�����������ݣ���������ģ�������ٴ��룬�类hex���룬��hex������ٴ�����
	 * @param encryptedData ���ģ�����������
	 * @return ���ؽ��ܺ������
	 * @throws InvalidCipherTextException
	 * @throws IOException
	 */
	public static byte[] decrypt(byte[] privateKey, byte[] encryptedData) throws InvalidCipherTextException, IOException
	{
	    ECDomainParameters parameters = new ECDomainParameters(sm2p256v1.getCurve(), sm2p256v1.getG(), sm2p256v1.getN());
	
	    ECPrivateKeyParameters priKeyParameters = new ECPrivateKeyParameters(new BigInteger(1,privateKey),parameters);
	    SM2Engine engine = new SM2Engine();
	    engine.init(false, priKeyParameters);
	    byte[] cipher = decodeDERSM2Cipher(encryptedData);
	    return engine.processBlock(cipher, 0, cipher.length);
	}
	
	/**
	 * sm2ǩ��
	 * <p>userIdʹ��Ĭ�ϣ�1234567812345678
	 * @param privateKey ˽Կ������������
	 * @param sourceData ��ǩ������
	 * @return ����der�����ǩ��ֵ
	 * @throws CryptoException
	 */
	public static byte[] sign(byte[] privateKey, byte[] sourceData) throws CryptoException {
		return sign(defaultUserID, privateKey, sourceData);
	}
	
	/**
	 * sm2ǩ��
	 * @param userId IDֵ������Լ����ʹ��Ĭ�ϣ�1234567812345678
	 * @param privateKey ˽Կ������������
	 * @param sourceData ��ǩ������
	 * @return ����der�����ǩ��ֵ
	 * @throws CryptoException
	 */
	public static byte[] sign(byte[] userId, byte[] privateKey, byte[] sourceData) throws CryptoException {
		
	    ECDomainParameters parameters = new ECDomainParameters(sm2p256v1.getCurve(), sm2p256v1.getG(), sm2p256v1.getN());
	    ECPrivateKeyParameters priKeyParameters = new ECPrivateKeyParameters(new BigInteger(1,privateKey),parameters);
		SM2Signer signer = new SM2Signer();
        CipherParameters param = null;
        ParametersWithRandom pwr = new ParametersWithRandom(priKeyParameters, new SecureRandom());
        if (userId != null) {
            param = new ParametersWithID(pwr, userId);
        } else {
            param = pwr;
        }
        signer.init(true, param);
        signer.update(sourceData, 0, sourceData.length);
        return signer.generateSignature();
	}
	
	/**
	 * sm2��ǩ
	 * <p>userIdʹ��Ĭ�ϣ�1234567812345678
	 * @param publicKey ��Կ������������
	 * @param sourceData ����ǩ����
	 * @param signData ǩ��ֵ
	 * @return �����Ƿ�ɹ�
	 */
	public static boolean verifySign(byte[] publicKey, byte[] sourceData, byte[] signData){
		return verifySign(defaultUserID, publicKey, sourceData, signData);
	}
	
	/**
	 * sm2��ǩ
	 * @param userId IDֵ������Լ����ʹ��Ĭ�ϣ�1234567812345678
	 * @param publicKey ��Կ������������
	 * @param sourceData ����ǩ����
	 * @param signData ǩ��ֵ
	 * @return �����Ƿ�ɹ�
	 */
	public static boolean verifySign(byte[] userId, byte[] publicKey, byte[] sourceData, byte[] signData) {
		
		if(publicKey.length == 64) {
			byte tmp[] = new byte[65];
			System.arraycopy(publicKey, 0, tmp, 1, publicKey.length);
			tmp[0] = 0x04;
			publicKey = tmp;
		}
		
		ECDomainParameters parameters = new ECDomainParameters(sm2p256v1.getCurve(), sm2p256v1.getG(), sm2p256v1.getN());
	    ECPublicKeyParameters pubKeyParameters = new ECPublicKeyParameters(sm2p256v1.getCurve().decodePoint(publicKey), parameters);
		SM2Signer signer = new SM2Signer();
        CipherParameters param;
        if (userId != null) {
            param = new ParametersWithID(pubKeyParameters, userId);
        } else {
            param = pubKeyParameters;
        }
        signer.init(false, param);
        signer.update(sourceData, 0, sourceData.length);
        return signer.verifySignature(signData);
	}
	
	/**
	 * ��ȡder�����µĹ�Կ
	 * @param derData der����Ĺ�Կ������������
	 * @return ���ع�Կֵ
	 */
	public static byte[] getPublicKey(byte[] derData)
    {

        SubjectPublicKeyInfo info = SubjectPublicKeyInfo.getInstance(derData);
        return info.getPublicKeyData().getBytes();
    }

	/**
	 * ��ȡder�����µ�˽Կ
	 * @param derData der�����˽Կ������������
	 * @return ����˽Կֵ
	 * @throws IOException
	 */
    public static byte[] getPrivateKey(byte[] derData) throws IOException
    {

        PrivateKeyInfo pinfo = PrivateKeyInfo.getInstance(derData);
        ECPrivateKey cpk = ECPrivateKey.getInstance(pinfo.parsePrivateKey());

        int length = 32;
        byte[] bytes = cpk.getKey().toByteArray();
        if (bytes.length == length)
        {
            return bytes;
        }

        int start = bytes[0] == 0 ? 1 : 0;
        int count = bytes.length - start;

        if (count > length)
        {
            return null;
        }

        byte[] tmp = new byte[length];
        System.arraycopy(bytes, start, tmp, tmp.length - count, count);
        return tmp;
    }
	
    /**
     * ����sm2��˽Կ��
     * @return
     */
	public static SM2KeyPair generateKeyPair(){
		ECDomainParameters parameters = new ECDomainParameters(sm2p256v1.getCurve(), sm2p256v1.getG(), sm2p256v1.getN());
	    KeyGenerationParameters kgp = new ECKeyGenerationParameters(parameters, new SecureRandom());
	    ECKeyPairGenerator ecKeyPairGenerator = new ECKeyPairGenerator();
		ecKeyPairGenerator.init(kgp);
		
		ECPrivateKeyParameters ecpriv = null;
		ECPublicKeyParameters ecpub = null;
//		int count = 0;
		do {
			AsymmetricCipherKeyPair keypair = ecKeyPairGenerator.generateKeyPair();
			ecpriv = (ECPrivateKeyParameters) keypair.getPrivate();
			ecpub = (ECPublicKeyParameters) keypair.getPublic();
		}while(ecpriv == null || ecpriv.getD().equals(ECConstants.ZERO)
				|| ecpriv.getD().compareTo(sm2p256v1.getN()) >= 0 || ecpriv.getD().signum() <= 0);

        byte[] privKey = formartBigNum(ecpriv.getD(),32);
        byte[] pubxKey = formartBigNum(ecpub.getQ().getAffineXCoord().toBigInteger(), 32);
        byte[] pubyKey = formartBigNum(ecpub.getQ().getAffineYCoord().toBigInteger(), 32);
        byte[] pubKey = new byte[64];
        System.arraycopy(pubxKey, 0, pubKey, 0, pubxKey.length);
        System.arraycopy(pubyKey, 0, pubKey, pubxKey.length, pubyKey.length)
        ;
        return new SM2KeyPair(privKey, pubKey);
    }
	
	/**
	 * ��c1c2c3����ת��der����
	 * @param cipher c1c2c3����
	 * @return der�����sm2����
	 * @throws IOException
	 */
    public static byte[] encodeSM2CipherToDER(byte[] cipher)
        throws IOException {
        int startPos = 1;
        int curveLength = (sm2p256v1.getCurve().getFieldSize() + 7) / 8;
        int digestLength = SM3_DIGEST_LENGTH_32;
        
        byte[] c1x = new byte[curveLength];
        System.arraycopy(cipher, startPos, c1x, 0, c1x.length);
        startPos += c1x.length;

        byte[] c1y = new byte[curveLength];
        System.arraycopy(cipher, startPos, c1y, 0, c1y.length);
        startPos += c1y.length;

        byte[] c2 = new byte[cipher.length - c1x.length - c1y.length - 1 - digestLength];
        System.arraycopy(cipher, startPos, c2, 0, c2.length);
        startPos += c2.length;

        byte[] c3 = new byte[digestLength];
        System.arraycopy(cipher, startPos, c3, 0, c3.length);

        ASN1Encodable[] arr = new ASN1Encodable[4];
        arr[0] = new ASN1Integer(new BigInteger(1, c1x));
        if(new BigInteger(1, c1x).toByteArray().length<32){
        	System.out.println("");
        }
        arr[1] = new ASN1Integer(new BigInteger(1,c1y));
        arr[2] = new DEROctetString(c3);
        arr[3] = new DEROctetString(c2);
        DERSequence ds = new DERSequence(arr);
        return ds.getEncoded(ASN1Encoding.DER);
    }
    
    /**
         * ��DER�������ģ����ݡ�SM2�����㷨ʹ�ù淶�� GM/T 0009-2012��
     *
     * @param derCipher ��der�����sm2����ת��c1c2c3��ʽ
     * @return ����c1c2c3��ʽ����
     * @throws IOException 
     */
    public static byte[] decodeDERSM2Cipher(byte[] derCipher) throws IOException {
    	
    	ByteArrayInputStream bis = new ByteArrayInputStream(derCipher);
    	ASN1InputStream dis = new ASN1InputStream(bis);
//        ASN1Sequence as = DERSequence.getInstance(derCipher);
    	ASN1Sequence as = (ASN1Sequence) dis.readObject();
        byte[] c1x = ((ASN1Integer) as.getObjectAt(0)).getValue().toByteArray();
        byte[] c1y = ((ASN1Integer) as.getObjectAt(1)).getValue().toByteArray();
        byte[] c3 = ((DEROctetString) as.getObjectAt(2)).getOctets();
        byte[] c2 = ((DEROctetString) as.getObjectAt(3)).getOctets();
        dis.close();

        int pos = 0;
        int curveLength = (sm2p256v1.getCurve().getFieldSize() + 7) / 8;
        byte[] cipherText = new byte[1 + curveLength*2 + c2.length + c3.length];

        final byte uncompressedFlag = 0x04;
        cipherText[0] = uncompressedFlag;
        pos += 1;

        if (c1x.length >= curveLength) {
        	System.arraycopy(c1x, c1x.length-curveLength, cipherText, pos, curveLength);
        } else {
        	System.arraycopy(c1x, 0, cipherText, pos+curveLength-c1x.length, c1x.length);
        }
        pos += curveLength;
        
        if (c1y.length >= curveLength) {
        	System.arraycopy(c1y, c1y.length-curveLength, cipherText, pos, curveLength);
        } else {
        	System.arraycopy(c1y, 0, cipherText, pos+curveLength-c1y.length, c1y.length);
        }
        pos += curveLength;

        System.arraycopy(c2, 0, cipherText, pos, c2.length);
        pos += c2.length;

        System.arraycopy(c3, 0, cipherText, pos, c3.length);

        return cipherText;
    }
	
    /**
     * ��ʽ��BigInteger��bg.toByteArray()��ȡ�����ֽ����ݳ��Ȳ��̶��������Ҫ��ʽ��Ϊ�̶�����
     * @param bg ����
     * @param needLength ����Ҫ�ĳ���
     * @return
     */
	private static byte[] formartBigNum(BigInteger bg,int needLength) {
		
		byte[] tmp = new byte[needLength];
		byte[] bgByte = bg.toByteArray();
		if(bgByte == null) {
			return null;
		}
		
		if(bgByte.length > needLength) {
			System.arraycopy(bgByte, bgByte.length - needLength, tmp, 0, needLength);
		}else if(bgByte.length == needLength) {
			tmp = bgByte;
		}else {
			System.arraycopy(bgByte, 0, tmp, needLength - bgByte.length, bgByte.length);
		}
		
		return tmp;
	}
	
}
