package cn.com.duiba.order.center.biz.handle;

import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import org.slf4j.LoggerFactory;
import cn.com.duiba.idmaker.service.api.enums.IDMakerTypeEnums;
import cn.com.duiba.idmaker.service.api.remoteservice.RemoteIDMakerService;
import cn.com.duiba.order.center.api.dto.AmbOrderFastDto;
import cn.com.duiba.order.center.biz.bo.AmbOrderFastBo;
import cn.com.duiba.order.center.biz.dao.BaseOrderTool;
import cn.com.duiba.order.center.biz.dao.orderconsumer.OrdersSimpleDao;
import cn.com.duiba.order.center.biz.dao.orderhouse.OrderHouseSimpleDao;
import cn.com.duiba.order.center.biz.entity.AmbSubOrdersEntity;
import cn.com.duiba.order.center.biz.entity.OrdersEntity;
import cn.com.duiba.order.center.biz.service.credits.AmbSubOrdersService;
import cn.com.duiba.order.center.biz.service.mainorder.OrderSyncService;
import cn.com.duiba.order.center.biz.service.mainorder.orderconsumer.OrdersTextChangeService;
import cn.com.duiba.order.center.biz.tool.IdGenerator;
import cn.com.duiba.order.center.biz.tool.TimeProfile;
import cn.com.duiba.order.center.biz.tool.TradeCenterException;
import cn.com.duiba.paycenter.client.PayCenterServiceClient;
import cn.com.duiba.paycenter.client.RpcResult;
import cn.com.duiba.service.exception.BusinessException;
import cn.com.duiba.service.exception.StatusException;
import cn.com.duiba.wolf.dubbo.DubboResult;

/**
 * Created by yansen on 16/9/22.
 */
@Component
public class MainOrderSimpleHandle {

	private static final Logger log = LoggerFactory.getLogger(MainOrderSimpleHandle.class);

	@Autowired
	private OrdersSimpleDao ordersSimpleDao;

	@Autowired
	private OrderHouseSimpleDao orderHouseSimpleDao;

	@Autowired
	private PayCenterServiceClient payCenterServiceClient;

	@Autowired
	private RemoteIDMakerService remoteIDMakerService;

	@Autowired
	private OrderSyncService orderSyncService;
	@Autowired
	private AmbOrderFastBo ambOrderFastBo;
	@Autowired
	private OrdersTextChangeService ordersTextChangeService;
	@Autowired
	private AmbSubOrdersService ambSubOrdersService;

	/**
	 *
	 * @param id
	 * @param consumerId
	 * @return
	 */
	public OrdersEntity findById(Long id, Long consumerId) {
		String tableName = BaseOrderTool.getTableNameStr(consumerId);
		return ordersSimpleDao.findById(id, tableName);
	}
	
	public List<OrdersEntity> findByIds(List<Long> ids, Long consumerId){
		if(consumerId == null){
			return null;
		}
		String tableName = BaseOrderTool.getTableNameStr(consumerId);
		return ordersSimpleDao.findByIds(ids, tableName);
	}
	
	public List<OrdersEntity> findByIdsInSequence(List<Long> ids, String sequence, Long consumerId){
		if(consumerId == null){
			return null;
		}
		String tableName = BaseOrderTool.getTableNameStr(consumerId);
		return ordersSimpleDao.findByIdsInSequence(ids, tableName, sequence);
	}

	/**
	 *
	 * @param orderNum
	 * @return
	 */
	public OrdersEntity findByOrderNum(String orderNum) {
		if (!orderNum.contains(IdGenerator.CHASH)) {
			throw new TradeCenterException("此订单号不能查询此库");
		}
		int cindex = orderNum.indexOf(IdGenerator.CHASH);
		String tableSuffix = orderNum.substring(cindex + 1, cindex + 5);
		String tableName = BaseOrderTool.getTableNameBySuffix(tableSuffix);
		return ordersSimpleDao.findByOrderNum(orderNum, tableName);
	}

	/**
	 *
	 * @param orderId
	 * @param consumerId
	 * @return
	 */
	public OrdersEntity select4updatelock(Long orderId, Long consumerId) {
		String tableName = BaseOrderTool.getTableNameStr(consumerId);
		return ordersSimpleDao.select4updatelock(orderId, consumerId, tableName);
	}

	/**
	 *
	 * @param order
	 * @throws Exception
	 */
	public void insert(OrdersEntity order) throws BusinessException {
		try {
			TimeProfile.enter("insert");
			if (StringUtils.isBlank(order.getPayStatus()) || StringUtils.isBlank(order.getDuibaPayStatus())
					|| StringUtils.isBlank(order.getConsumerPayStatus())) {
				throw new BusinessException("状态不正确");
			}
			long sequence = getSequenceId();
			long id = Long.parseLong(sequence + "" + IdGenerator.get1024Suffix(order.getConsumerId()));
			String orderNum = IdGenerator.generate(sequence, order.getConsumerId());
			order.setId(id);
			order.setOrderNum(orderNum);
			order.setTableName(BaseOrderTool.getTableNameStr(order.getConsumerId()));
			int flag = ordersSimpleDao.insert(order);
			if (flag != 1) {
				throw new BusinessException("订单没有创建");
			}
		} finally {
			TimeProfile.release();
		}
		orderSyncService.sync(order.getId(), order.getConsumerId());
	}

	/**
	 *
	 * @param order
	 * @throws Exception
	 */
	public void takeOrderFree(OrdersEntity order) throws BusinessException {
		insert(order);
	}

	/**
	 *
	 * @param order
	 * @throws Exception
	 */
	public void takeOrderDuibaTakeMOney(OrdersEntity order) throws BusinessException {
		// 校验帐户金额够不够--改造为web验证
		insert(order);
	}

	/**
	 *
	 * @param order
	 * @throws Exception
	 */
	public void takeOrderTakeMoney(OrdersEntity order) throws Exception {
		/*** 对接支付中心，加入新代码 begin ***/
		checkAccountMoneyByDevelopId(order);

		// 初始定单支付状态弄成unKnow_pay状态。
		insert(order);
	}

	/**
	 * 加钱购
	 * 
	 * @param order
	 * @throws Exception
	 */
	public void takeOrderConsumerPay(OrdersEntity order, AmbSubOrdersEntity ambSubOrdersEntity)
			throws BusinessException {
		// 生成主定单
		insert(order);
		// 生成子定单
		ambSubOrdersEntity.setOrdersId(order.getId());
		AmbSubOrdersEntity sub = ambSubOrdersService.createSubOrders(ambSubOrdersEntity);

		ordersTextChangeService.updateSubOrderIdById(order.getId(), order.getConsumerId(), sub.getId());
		// 生成加速定单
		AmbOrderFastDto ret = ambOrderFastBo.createOrderFastMaxTime(order.getId(), order.getConsumerId());
		if (ret == null) {
			throw new BusinessException("加速  定单已经存在");
		}
	}

	private Long getSequenceId() {
		try {
			TimeProfile.enter("getSequenceId");
			DubboResult<Long> id = remoteIDMakerService.getNextID(IDMakerTypeEnums.ORDER.getType());
			if (id.isSuccess()) {
				return id.getResult();
			} else {
				log.error("ID生成失败原因=" + id.getMsg());
				throw new TradeCenterException("ID生成失败");
			}
		} finally {
			TimeProfile.release();
		}
	}

	private void checkAccountMoneyByDevelopId(OrdersEntity order) throws Exception {
		RpcResult<Long> rpcResult = payCenterServiceClient.getBalance(order.getDeveloperId());

		// 调用失败
		if (!rpcResult.isOpSuccess()) {
			throw new BusinessException("从支付中心获取金额 失败 ", rpcResult.getRpcException());
		}
		if (rpcResult.getResult() < order.getActualPrice()) {
			throw new StatusException(StatusException.CodeBalanceNotEnough);
		}
	}

	/**
	 * 改用发号器的订单使用查询分库,改造前的查询house库
	 * 
	 * @param id
	 * @return
	 */
	public OrdersEntity findById(Long id) {
		String tableSuffix = BaseOrderTool.getTableSuffixByOrderId(id);
		if (StringUtils.isEmpty(tableSuffix)) {
			return orderHouseSimpleDao.find(id);
		}
		return ordersSimpleDao.findById(id, BaseOrderTool.getTableNameBySuffix(tableSuffix));
	}

}
