/**
 * Project Name:activity-center-api<br>
 * File Name:RemotePluginService.java<br>
 * Package Name:cn.com.duiba.tuia.activity.center.api.remoteservice<br>
 * Date:2017年5月9日上午10:20:07<br>
 * Copyright (c) 2017, duiba.com.cn All Rights Reserved.<br>
 */

package cn.com.duiba.tuia.activity.center.api.remoteservice;

import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.tuia.activity.center.api.common.PageDto;
import cn.com.duiba.tuia.activity.center.api.dto.ActivityOptionDto;
import cn.com.duiba.tuia.activity.center.api.dto.ActivityPluginDto;
import cn.com.duiba.tuia.activity.center.api.dto.PluginOptionDto;
import cn.com.duiba.tuia.activity.center.api.dto.PluginSkinDto;
import cn.com.duiba.wolf.dubbo.DubboResult;
import org.springframework.web.bind.annotation.RequestMapping;

import java.util.List;

/**
 * ClassName: RemotePluginService <br/>
 * 活动插件工具Dubbo服务类 date: 2017年5月9日 上午10:20:07 <br/>
 * .
 *
 * @author youhaijun
 * @since JDK 1.7
 */
@AdvancedFeignClient
public interface RemotePluginService {

    /**
     * 保存皮肤数据，根据ID是否存在区分是新增还是更新.
     *
     * @param dto the dto
     * @return the dubbo result< long>
     */
    DubboResult<Long> saveSkin(PluginSkinDto dto);

    /**
     * 根据皮肤ID查询皮肤详情. 调用方需要校验是否删除
     * 
     * @param id the id
     * @return the dubbo result< plugin skin dto>
     */
    DubboResult<PluginSkinDto> selectSkinById(Long id);

    /**
     * 分页查询有效皮肤列表，返回只包括id，名称.
     * 
     * @param offset the offset
     * @param max the max
     * @return the dubbo result< page dto< plugin skin dto>>
     */
    DubboResult<PageDto<PluginSkinDto>> selectSkinByPage(Integer offset, Integer max);

    /**
     * 根据插件id查询插件配置的奖项列表). <br/>
     *
     * @param pluginId the plugin id
     * @param includeStock 是否包含库存值(库存值需要调库存中心查询，不需要的话就不要查询)
     * @return the dubbo result< list< PluginOptionDto>>
     * @author youhaijun
     * @since JDK 1.7
     * @deprecated 奖项已经被移动到tb_activity_option中
     */
    @Deprecated
    DubboResult<List<PluginOptionDto>> selectOptionByPluginId(Long pluginId, Boolean includeStock);

    /**
     * 根据插件id查询插件配置的奖项列表.从tb_activity_option中查询 <br/>
     *
     * @param pluginId the plugin id
     * @param includeStock 是否包含库存值(库存值需要调库存中心查询，不需要的话就不要查询)
     * @return List<ActivityOptionDto>
     * @author guopengfei
     */
    DubboResult<List<ActivityOptionDto>> selectActivityOptionByPluginId(Long pluginId, Boolean includeStock);

    /**
     * 根据id查询奖项. <br/>
     *
     * @param optionId 奖项id
     * @return the dubbo result< activity option dto>
     * @author youhaijun
     * @since JDK 1.7
     * @deprecated 奖项已经被移动到tb_activity_option中
     */
    @Deprecated
    DubboResult<PluginOptionDto> selectOption(Long optionId);

    /**
     * 根据id查询奖项.从tb_activity_option中查询 <br/>
     *
     * @param optionId 奖项id
     * @return ActivityOptionDto
     * @author guopengfei
     */
    DubboResult<ActivityOptionDto> selectActivityOption(Long optionId);

    /**
     * saveActivityOption:(保存奖项). <br/>
     * 有id update <br/>
     * 无id insert <br/>
     *
     * @param options 批量保存奖项
     * @return 保存的数量
     * @author youhaijun
     * @since JDK 1.7
     * @deprecated 奖项已经被移动到tb_activity_option中
     */
    @Deprecated
    DubboResult<Integer> saveOption(List<PluginOptionDto> options);

    /**
     * saveActivityOptions:(保存奖项).tb_activity_option <br/>
     * 有id update <br/>
     * 无id insert <br/>
     *
     * @param options 批量保存奖项
     * @return 保存的数量
     * @author guopengfei
     */
    DubboResult<Integer> saveActivityOptions(List<ActivityOptionDto> options);

    /**
     * 批量删除奖项. <br/>
     *
     * @param optionIds 奖项ids
     * @return 删除数量
     * @author youhaijun
     * @since JDK 1.7
     * @deprecated 奖项已经被移动到tb_activity_option中
     */
    @Deprecated
    DubboResult<Integer> deleteBatchOptions(List<Long> optionIds);

    /**
     * 批量删除奖项.tb_activity_option <br/>
     *
     * @param optionIds 奖项ids
     * @return 删除数量
     * @author guopengfei
     */
    DubboResult<Integer> deleteBatchActivityOptions(List<Long> optionIds);

    /**
     * 保存活动插件工具，根据id区分新增和更新.
     *
     * @param dto the dto
     * @return the dubbo result< long>
     */
    DubboResult<Long> saveActivtiyPlugin(ActivityPluginDto dto);
    
    /**
     * 根据ID查询活动插件.
     *
     * @param id the id
     * @return the dubbo result< activity plugin dto>
     */
    DubboResult<ActivityPluginDto> selectActivityPluginById(Long id);

    /**
     * 根据id list 查询插件活动
     * 会过滤关闭，删除的插件活动
     * @param ids id list
     * @return dto list
     */
    DubboResult<List<ActivityPluginDto>> selectAvailablePluginByIds(List<Long> ids);
    
    /**
     * 分页查询活动插件列表，只返回id，状态和名称.
     *
     * @param offset the offset
     * @param max the max
     * @param id the id
     * @param title the title
     * @return the dubbo result< page dto< activity plugin dto>>
     */
    DubboResult<PageDto<ActivityPluginDto>> selectActivtiyPluginByPage(Integer offset, Integer max, Long id, String title);

    /**
     * 查询所有插件
     * @param type 插件类型
     * @return list dto
     */
    List<ActivityPluginDto> selectAllPluginByType(Integer type);


    /**
     * 分页查询活动插件列表，只返回id，状态和名称与插件类型.
     *
     * @param offset the offset
     * @param max the max
     * @param id the id
     * @param title the title
     * @return the dubbo result< page dto< activity plugin dto>>
     */
    @RequestMapping("selectActivtiyPluginByPageType")
    DubboResult<PageDto<ActivityPluginDto>> selectActivtiyPluginByPage(Integer offset, Integer max, Long id, String title,Integer type);
    
    /**
     * 更新插件开启关闭状态.
     *
     * @param record the record
     * @return the dubbo result< integer>
     */
    DubboResult<Integer> updateOpenStatus(ActivityPluginDto record);
    
    /**
     * 更新插件开启关闭状态.
     *
     * @param record the record
     * @return the dubbo result< integer>
     */
    DubboResult<Integer> updateDirectApp(ActivityPluginDto record);


    /**
     * 更新插件定向广告位.
     *
     * @param record the record
     * @return the dubbo result< integer>
     */
    DubboResult<Integer> updateDirectSlot(ActivityPluginDto record);

    /**
     * 根据ID删除活动插件.
     *
     * @param id the id
     * @return the dubbo result< boolean>
     */
    DubboResult<Integer> deletePluginById(Long id);
    
    /**
     * 删除插件皮肤.
     *
     * @param id the id
     * @return the dubbo result< integer>
     */
    DubboResult<Integer> deleteSkinById(Long id);
    
    /**
     * 查询可以投放的活动插件.
     * 筛选条件包括：是否开启，是否定向app，是否在触发时段中且没有自动关闭，当前页面是否在触发场景中，当前页面参与次数是否满足触发事件
     *
     * @return the dubbo result< activity plugin dto> 返回可以用来投放的一个插件工具，没有可用的时候返回空list,请校验
     */
    DubboResult<List<ActivityPluginDto>> selectAvailableActivityPlugin();

    /**
     * 查询可以投放的嵌入式插件 <br/>
     * 筛选条件：是否开启，是否定向app <br/>
     * 按优先级顺序返回。  <br/>
     * @param pageId 引导页id
     * @param blockName 区块名称
     * @return 按顺序返回
     */
    DubboResult<List<ActivityPluginDto>> selectAvailableEmbeddedPlugin(Long pageId,String blockName);

    /**
     * 根据插件ID 查询嵌入式插件
     * @param pluginId
     * @return
     */
    DubboResult<List<ActivityPluginDto>> selectAvailableEmbeddedPluginById(Long pluginId);

    /**
     * 根据皮肤ID查询此皮肤绑定的可用插件id，如有多个，返回最新的.
     *
     * @param skinId the skin id
     * @return the dubbo result< long>
     */
    DubboResult<Long> selectAvilPluginBySkinId(Long skinId);

    /***
         * @Description: selectAvailableResponsivePlugin 查询可用响应式插件列表
         *
         * @Param: [pageId, blockName]
         * @Return: cn.com.duiba.wolf.dubbo.DubboResult<java.util.List<cn.com.duiba.tuia.activity.center.api.dto.ActivityPluginDto>>
         * @throws:
         * @author: youhaijun
         * @Date:   2017/9/13
         */
    DubboResult<List<ActivityPluginDto>> selectAvailableResponsivePlugin();

    /**
     * 根据时间点查询可用插件
     * @return dto list
     */
    DubboResult<List<ActivityPluginDto>> selectAvailablePluginByMinute();


    /**
     * 更新定时插件rides
     */
    DubboResult<Boolean> updateTimingRides(ActivityPluginDto activityPluginDto,Integer updateType);

    String getRedis();

    boolean submitAudit(Long managerId,Long pluginId);

    /**
     * 查询 插件 根据id或title 模糊匹配
     * @param activityPluginDto
     * @return list dto
     */
    List<ActivityPluginDto> selectPluginLikeIdOrName(ActivityPluginDto activityPluginDto);
}
