package cn.com.duiba.tuia.activity.center.api.util;

import cn.com.duiba.tuia.activity.center.api.annotation.ExcelRowIndex;
import cn.com.duiba.tuia.activity.center.api.constant.ErrorCode;
import cn.com.duiba.tuia.activity.center.api.exception.ActivityCenterException;
import cn.com.duiba.wolf.utils.DateUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFDateUtil;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.NumberToTextConverter;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;
import org.springframework.web.multipart.commons.CommonsMultipartResolver;

import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.Field;
import java.util.*;

/**
 * Excel 解析工具类
 *
 * @author lixin
 * @date 2019/6/6 9:57
 **/
public class ExcelUtil {

    /** 日志 */
    private static final Logger LOGGER = LoggerFactory.getLogger(ExcelUtil.class);

    /** 请求中的文件对应的参数名称 */
    public static final String REQUEST_PARAM_NAME = "file";

    /** 文件后缀：xls */
    public static final String FILE_SUFFIX_XLS = "xls";

    /** 文件后缀：xlsx */
    public static final String FILE_SUFFIX_XLSX = "xlsx";

    /** 字段类型关系，key为类型类的名字，value为类型类对象 */
    private static Map<String, Class<?>> classMap = new HashMap<>();

    private static final String METHOD_NAME = "valueOf";

    /**
     * private constructor
     */
    private ExcelUtil() {
    }

    /**
     * 初始化字段类型关系集合
     */
    static {
        classMap.put(String.class.getSimpleName(), String.class);
        classMap.put(Integer.class.getSimpleName(), Integer.class);
        classMap.put(int.class.getSimpleName(), Integer.class);
        classMap.put(Double.class.getSimpleName(), Double.class);
        classMap.put(double.class.getSimpleName(), Double.class);
        classMap.put(Long.class.getSimpleName(), Long.class);
        classMap.put(long.class.getSimpleName(), Long.class);
        classMap.put(Boolean.class.getSimpleName(), Boolean.class);
        classMap.put(boolean.class.getSimpleName(), Boolean.class);
        classMap.put(Date.class.getSimpleName(), Date.class);
    }

    /**
     * 解析Excel，返回给定的对象类型的集合
     *
     * @param request        request
     * @param clzz           clzz
     * @param ignoreFirstRow 解析时是否跳过第一行（true：跳过；false：不跳过）,default: false<br/>
     *                       true: 即认为excel有标题行，从第二行开始解析数据；
     *                       false: 即excel没有标题行，直接从第一行解析数据
     * @param <T>
     * @return list
     * @throws ActivityCenterException
     */
    public static <T> List<T> parseExcel(HttpServletRequest request, Class<T> clzz, Boolean ignoreFirstRow) throws ActivityCenterException {
        // 解析excel，返回workbook
        Workbook wb = parseExcelToWorkbook(request);

        // 目前只支持解析第一个sheet页的数据
        Sheet sheet = wb.getSheetAt(0);

        if (Objects.isNull(ignoreFirstRow)) {
            ignoreFirstRow = false;
        }
        return (List<T>) parseExcelSheet(sheet, clzz, ignoreFirstRow);
    }

    /**
     * 解析sheet页
     *
     * @param sheet          sheet
     * @param clzz           clzz
     * @param ignoreFirstRow
     * @return list
     * @throws ActivityCenterException
     */
    private static List<Object> parseExcelSheet(Sheet sheet, Class<?> clzz, Boolean ignoreFirstRow) throws ActivityCenterException {
        int firstRowIndex = sheet.getFirstRowNum();
        int lastRowIndex = sheet.getLastRowNum();
        List<Object> resList = new ArrayList<>();
        for (int index = ignoreFirstRow ? firstRowIndex + 1 : firstRowIndex; index <= lastRowIndex; index++) {
            Object obj = parseRowData(sheet.getRow(index), clzz);
            // 当obj为空时则说明当前行为空，未解析出数据，即为空行
            if (Objects.isNull(obj)) {
                continue;
            }
            resList.add(obj);
        }
        return resList;
    }

    /**
     * 解析row数据
     *
     * @param row  row
     * @param clzz clzz
     * @return obj
     * @throws ActivityCenterException
     */
    private static Object parseRowData(Row row, Class<?> clzz) throws ActivityCenterException {
        try {
            if (null == row) {
                LOGGER.warn("当前行数据为空！");
                return null;
            }

            Object obj = clzz.newInstance();
            Field[] fields = clzz.getDeclaredFields();
            ExcelRowIndex excelRowIndex;
            int rowIndex;
            Cell cell;
            for (Field field : fields) {
                excelRowIndex = field.getAnnotation(ExcelRowIndex.class);
                if (null == excelRowIndex) {
                    continue;
                }

                rowIndex = excelRowIndex.value();
                cell = row.getCell(rowIndex);
                parseCellValueToField(cell, field, obj);
            }
            return obj;
        } catch (Exception e) {
            throw new ActivityCenterException(ErrorCode.E9999999.getErrorCode(), e.getMessage());
        }
    }

    /**
     * 解析cell，并映射到给定的对象中去
     *
     * @param cell  cell
     * @param field field
     * @param obj   obj
     * @throws ActivityCenterException
     */
    private static void parseCellValueToField(Cell cell, Field field, Object obj) throws ActivityCenterException {
        String cellValue;
        if (cell == null) {
            return;
        }

        // 判断数据的类型
        switch (cell.getCellTypeEnum()) {
            case NUMERIC: // 数字
                if (HSSFDateUtil.isCellDateFormatted(cell)) {// 处理日期格式、时间格式
                    cellValue = DateUtils.getSecondStr(cell.getDateCellValue());
                } else {
                    //处理数值格式
                    cellValue = NumberToTextConverter.toText(cell.getNumericCellValue());
                }
                break;
            case BOOLEAN: // Boolean
                cellValue = String.valueOf(cell.getBooleanCellValue());
                break;
            case FORMULA: // 公式
                cellValue = String.valueOf(cell.getCellFormula());
                break;
            default:
                // 默认以字符串处理
                cellValue = String.valueOf(cell.getStringCellValue());
                break;
        }

        Class<?> fieldType = field.getType();
        field.setAccessible(true);
        setValueByFieldType(fieldType, cellValue, field, obj);
    }

    /**
     * 根据字段类型将单元格数据转换成对应格式的值赋值到字段中
     *
     * @param fieldType fieldType
     * @param cellValue cellValue
     * @param field     field
     * @param obj       obj
     * @param <T>
     * @throws ActivityCenterException
     */
    private static <T> void setValueByFieldType(Class<T> fieldType, String cellValue, Field field, Object obj) throws ActivityCenterException {
        String fieldTypeName = fieldType.getSimpleName();
        Class<?> cls = classMap.get(fieldTypeName);
        if (Objects.isNull(cls)) {
            throw new ActivityCenterException(String.format("字段类型[%s]无法映射cell值", fieldTypeName));
        }

        try {
            if (cls == String.class) {
                field.set(obj, cellValue);
            } else {
                field.set(obj, cls.getDeclaredMethod(METHOD_NAME, String.class).invoke(null, cellValue));
            }
        } catch (Exception e) {
            throw new ActivityCenterException(String.format("字段[%s]映射cell值[%s]异常", field.getName(), cellValue));
        }
    }

    /**
     * 解析excel，返回workbook
     *
     * @param request request
     * @return Workbook
     * @throws ActivityCenterException
     */
    private static Workbook parseExcelToWorkbook(HttpServletRequest request) throws ActivityCenterException {
        try {
            // 解析器解析request的上下文
            CommonsMultipartResolver multipartResolver = new CommonsMultipartResolver(
                    request.getSession().getServletContext());
            // 先判断request中是否包涵multipart类型的数据
            if (!multipartResolver.isMultipart(request)) {
                throw new ActivityCenterException(ErrorCode.E9999999);
            }

            MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
            MultipartFile mf = multipartRequest.getFile(REQUEST_PARAM_NAME);
            if (mf == null) {
                throw new ActivityCenterException("未解析到文件！请重新上传");
            }
            String fileName = mf.getOriginalFilename();
            String[] split = fileName.split("\\.");
            Workbook wb;
            //根据文件后缀（xls/xlsx）进行判断
            if (StringUtils.equals(split[1], FILE_SUFFIX_XLS)) {
                wb = new HSSFWorkbook(mf.getInputStream());
            } else if (StringUtils.equals(split[1], FILE_SUFFIX_XLSX)) {
                wb = new XSSFWorkbook(mf.getInputStream());
            } else {
                throw new ActivityCenterException("excel格式有误，请按标准格式上传!");
            }

            return wb;
        } catch (Exception e) {
            throw new ActivityCenterException(ErrorCode.E9999999.getErrorCode(), e.getMessage());
        }
    }
}
