package cn.com.duiba.tuia.activity.center.api.common;

import lombok.Data;

import java.io.Serializable;
import java.util.Collections;
import java.util.List;

/**
 * Basic Java Bean implementation of {@code Pageable}.
 *
 * @author ikongyi
 */
@Data
public class PageRequest implements Pageable, Serializable {

    private static final long serialVersionUID = -3990989391042856586L;

    public static final Integer DEFAULT_OFFSET = 0;
    public static final Integer DEFAULT_PAGE_SIZE = 200;

    private Integer offset;
    private Integer pageSize;
    private List<OrderBy> orderByList;

    /**
     * Creates a new unsorted {@link PageRequest}.
     *
     * @param offset zero-based page index, must not be negative.
     * @param size the size of the page to be returned, must be greater than 0.
     */
    public static PageRequest of(int offset, int size) {
        return of(offset, size, Collections.emptyList());
    }

    /**
     * Creates a new {@link PageRequest} with sort parameters applied.
     *
     * @param offset zero-based page index.
     * @param size the size of the page to be returned.
     * @param orderBy must not be {@literal null}
     */
    public static PageRequest of(int offset, int size, OrderBy orderBy) {
        return newPageRequest(offset, size, Collections.singletonList(orderBy));
    }

    /**
     * Creates a new {@link PageRequest} with sort parameters applied.
     *
     * @param offset zero-based page index.
     * @param size the size of the page to be returned.
     * @param orderByList must not be {@literal null}
     */
    public static PageRequest of(int offset, int size, List<OrderBy> orderByList) {
        return newPageRequest(offset, size, orderByList);
    }

    /**
     * Creates a new {@link PageRequest} with sort direction and properties applied.
     *
     * @param offset        zero-based page index, must not be negative.
     * @param size        the size of the page to be returned, must be greater than 0.
     * @param direction   must not be {@literal null}.
     * @param columnNames must not be {@literal null}.
     */
    public static PageRequest of(int offset, int size, OrderBy.Direction direction, String... columnNames) {
        return of(offset, size, OrderBy.by(direction, columnNames));
    }

    static PageRequest newPageRequest(int offset, int size, List<OrderBy> orderByList) {
        PageRequest pageRequest = new PageRequest();
        pageRequest.setOffset(offset);
        pageRequest.setPageSize(size);
        pageRequest.setOrderByList(orderByList);
        return pageRequest;
    }

    /*
     * (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return String.format("Page request [offset %d, size: %s]", getOffset(), getPageSize());
    }

    @Override
    public List<OrderBy> getSort() {
        return orderByList;
    }
}

