package cn.com.duiba.tuia.activity.center.api.util;


import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.*;

import static java.time.temporal.ChronoField.DAY_OF_MONTH;
import static java.time.temporal.ChronoField.DAY_OF_YEAR;
import static java.time.temporal.ChronoUnit.MONTHS;
import static java.time.temporal.ChronoUnit.YEARS;
import static java.time.temporal.TemporalAdjusters.*;

/**
 * 基于 JDK 8 time包的时间工具类
 * author:
 */
public class DateUtils {

    private static final Logger LOG = LoggerFactory.getLogger(DateUtils.class);

    private static final Map<String, DateTimeFormatter> FORMATTER_CACHE = new HashMap<>();

    private static final String YYYY_FORMAT = "yyyy";

    private static final String MM_DD_FORMAT = "MM-dd";

    private static final String MINUTE_FORMAT = "yyyyMMdd-HHmm";

    private static final String MONTH_DAY_YEAR_FORMAT = "MM/dd/yyyy";

    private static final String SECOND_ONLY_FORMAT = "HH:mm:ss";

    private static final String SECOND_FORMAT = "yyyy-MM-dd HH:mm:ss";

    private static final String MINUTE_NORMAL_FORMAT = "yyyy-MM-dd HH:mm";

    private static final String MINUTE_ONLY_FORMAT = "HH:mm";

    private static final String DAY_FORMAT = "yyyy-MM-dd";

    private static final String DAY_NUMBER_FORMAT = "yyyyMMdd";

    private static final String YEAR_DAY_NUMBER_FORMAT = "yyMMdd";

    private static final String MILLI_SECOND_FORMAT = "yyyyMMddHHmmssSSS";

    private static final String SECOND_FORMAT_NO_SYMBOL = "yyyyMMddHHmmss";

    /**
     * 获取默认时间格式: yyyy-MM-dd HH:mm:ss
     */
    private static final DateTimeFormatter DEFAULT_DATETIME_FORMATTER = TimeFormat.LONG_DATE_PATTERN_LINE.formatter;


    private DateUtils() {
    }




    /**
     * method : 1、根据Date类型获取"yyyy"字符串
     */
    public static String getYearStr(Date date) {
        return formatDate(date, YYYY_FORMAT);
    }

    /**
     * method : 2、根据时间戳获取"yyyy"字符串
     */
    public static String getYearStr(final long date) {
        return formatDate(date, YYYY_FORMAT);
    }

    /**
     * method : 3、根据Date类型获取"yyyyMMdd-HHmm"字符串
     */
    public static String getMinuteDbStr(final Date date) {
        return formatDateTime(date, MINUTE_FORMAT);
    }

    /**
     * method : 4、根据"yyyyMMdd-HHmm"字符串 获取 Date类型
     */
    public static Date getMinuteDbDate(final String str) {
        return getDateTime(str,MINUTE_FORMAT);
    }
    /**
     * method : 4、根据"MM/dd/yyyy"字符串 获取 Date类型
     */
    public static Date getMonDayYearDate(final String str) {
        return getDate(str,MONTH_DAY_YEAR_FORMAT);
    }

    /**
     * method : 5、根据Date类型 获取"HH:mm:ss"字符串
     */
    public static String getSecondOnlyStr(final Date date) {
        return formatDateTime(date,SECOND_ONLY_FORMAT);
    }

    /**
     * method : 6、根据时间戳获取"HH:mm:ss"字符串
     */
    public static String getSecondOnlyStr(final long timestamp) {
        return formatDateTime(timestamp,SECOND_ONLY_FORMAT);
    }

    /**
     * method : 7、根据时间戳获取"MM-dd"字符串
     */
    public static String getOnlyDayStr(final long timestamp) {
        return formatDateTime(timestamp,MM_DD_FORMAT);
    }

    /**
     * method : 8、根据时间戳获取"yyyy-MM-dd HH:mm:ss"字符串
     */
    public static String getSecondStr(final long timestamp) {
        return formatDateTime(timestamp,SECOND_FORMAT);
    }

    /**
     * method : 9、根据Date获取"yyyy-MM-dd HH:mm:ss"字符串
     */
    public static String getSecondStr(final Date date) {
        return formatDateTime(date,SECOND_FORMAT);
    }

    /**
     * method : 10、根据Date获取"yyyy-MM-dd HH:mm"字符串
     */
    public static String getMinuteStr(final Date date) {
        return formatDateTime(date,MINUTE_NORMAL_FORMAT);
    }

    /**
     * method : 11、根据时间戳获取"yyyy-MM-dd HH:mm"字符串
     */
    public static String getMinuteStr(final long timestamp) {
        return formatDateTime(timestamp,MINUTE_NORMAL_FORMAT);
    }

    /**
     * method : 12、根据Date获取"HH:mm"字符串
     */
    public static String getMinuteOnlyStr(final Date date) {
        return formatDateTime(date,MINUTE_ONLY_FORMAT);
    }

    /**
     * method : 13、根据Date获取"yyyy-MM-dd"字符串
     */
    public static String getDayStr(final Date date) {
        return formatDate(date,DAY_FORMAT);
    }

    /**
     * method : 14、根据时间戳获取"yyyy-MM-dd"字符串
     */
    public static String getDayStr(final long timestamp) {
        return formatDate(timestamp,DAY_FORMAT);
    }

    /**
     * method : 15、根据Date获取"yyyyMMdd"形式的数字
     */
    public static int getDayNumber(final Date date) {
        if (date == null) {
            return 0;
        }
        return Integer.valueOf(formatDate(date, DAY_NUMBER_FORMAT));
    }

    /**
     * method : 16、根据Date获取"yyMMdd"形式的数字
     */
    public static int getYYDayNumber(final Date date) {
        if (date == null) {
            return 0;
        }
        return Integer.valueOf(formatDate(date, YEAR_DAY_NUMBER_FORMAT));
    }

    /**
     * method : 17、根据Date获取"yyyy-MM-dd 00:00:00"形式的Date
     */
    public static Date getDayDate(final Date date) {
        return getDayDate(getDayStr(date));
    }

    /**
     * method : 18、根据"yyyy-MM-dd HH:mm:ss"形式的字符串 获取Date
     */
    public static Date getSecondDate(final String dateStr) {
        return getDateTime(dateStr, SECOND_FORMAT);
    }

    /**
     * method : 19、根据"yyyy-MM-dd"形式的字符串 获取Date
     */
    public static Date getDayDate(final String dateStr) {
        return getDate(dateStr, DAY_FORMAT);
    }

    /**
     * method : 20、根据"HH:mm"形式的字符串 获取Date 获取的为1970-01-01 HH:mm:00形式的Date，一般不常用
     */
    public static Date getMinuteOnlyDate(final String dateStr) {
        final String newDateStr = LocalDate.ofEpochDay(0).toString() +" "+ dateStr;
        return getDateTime(newDateStr, MINUTE_NORMAL_FORMAT);
    }

    /**
     * method : 21、根据"yyyy-MM-dd HH:mm"形式的字符串 获取Date
     */
    public static Date getMinuteDate(final String dateStr) {
        return getDateTime(dateStr, MINUTE_NORMAL_FORMAT);
    }

    /**
     * method : 22、根据时间戳 获取"yyyy-MM-dd HH:mm"形式的Date
     */
    public static Date getMinuteDate(long time) {
        return getMinuteDate(getMinuteStr(time));
    }

    /**
     * method :
     * 23、根据（Date获取"yyyy-MM-dd"形式字符串）和（"HH:mm"的字符串）拼接成"yyyy-MM-dd HH:mm"
     * 形式的字符串，然后返回时间戳
     */
    public static long getMiniteDate(Date date, String str) {
        if (str == null)
            return 0;
        Date date1 = getMinuteDate(getDayStr(date) + " " + str);
        if(date1 == null){
            return 0;
        }
        return date1.getTime();
    }

    /**
     * 获取 Date类型
     * @param dateStr 字符串时间
     * @param pattern 格式参数
     * @return
     */
    private static Date getDate(final String dateStr,final String pattern){
        final LocalDate localDate = getLocalDate(dateStr,pattern);
        return Date.from(localDate.atStartOfDay(ZoneId.systemDefault()).toInstant());
    }

    /**
     * 返回 LocalDate  精确到 日期
     * @param dateStr
     * @param pattern
     * @return
     */
    private static LocalDate getLocalDate(final String dateStr,final String pattern){
        final DateTimeFormatter formatter = getOrCreateFormatter(pattern);
        return  LocalDate.parse(dateStr, formatter);
    }


    /**
     * 获取 Date类型
     * @param dateStr 字符串时间
     * @param pattern 格式参数
     * @return
     */
    private static Date getDateTime(final String dateStr,final String pattern){
        final LocalDateTime localDateTime = getLocalDateTime(dateStr,pattern);
        return Date.from(localDateTime.atZone(ZoneId.systemDefault()).toInstant());
    }

    /**
     * 返回 LocalDate  精确到 日期
     * @param dateStr
     * @param pattern
     * @return
     */
    private static LocalDateTime getLocalDateTime(final String dateStr,final String pattern){
        final DateTimeFormatter formatter = getOrCreateFormatter(pattern);
        return  LocalDateTime.parse(dateStr, formatter);
    }


    /**
     * date类型 转成 字符串 相应的格式 精确到 日期
     * @param date 时间参数
     * @param pattern 格式参数
     * @return
     */
    private static String formatDate(final Date date,final String pattern) {
        //date先转换成LocalDate
        final LocalDate localDate = date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
        return localDate.format(getOrCreateFormatter(pattern));
    }

    /**
     * date类型 转成 字符串 相应的格式 精确到日期和时间的组合
     * @param date 时间参数
     * @param pattern 格式参数
     * @return
     */
    private static String formatDateTime(final Date date,final String pattern) {
        //date先转换成LocalDate
        final LocalDateTime localDate = date.toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
        return localDate.format(getOrCreateFormatter(pattern));
    }

    /**
     * timestamp类型 转成 字符串 相应的格式 只精确到日期
     * @param timestamp 时间搓
     * @param pattern 格式参数
     * @return
     */
    private static String formatDate(final long timestamp,final String pattern) {
        //date先转换成LocalDate
        final LocalDate localDate = Instant.ofEpochMilli(timestamp).atZone(ZoneId.systemDefault()).toLocalDate();
        return localDate.format(getOrCreateFormatter(pattern));
    }

    /**
     * timestamp类型 转成 字符串 相应的格式 如果精确到日期加时间
     * @param timestamp 时间搓
     * @param pattern 格式参数
     * @return
     */
    private static String formatDateTime(final long timestamp,final String pattern) {
        //date先转换成LocalDateTime
        final LocalDateTime localDateTime = Instant.ofEpochMilli(timestamp).atZone(ZoneId.systemDefault()).toLocalDateTime();
        return localDateTime.format(getOrCreateFormatter(pattern));
    }

    private static DateTimeFormatter getOrCreateFormatter(String pattern) {
        DateTimeFormatter formatter;
        if (FORMATTER_CACHE.containsKey(pattern)) {
            formatter = FORMATTER_CACHE.get(pattern);
        } else {
            formatter = DateTimeFormatter.ofPattern(pattern);
            FORMATTER_CACHE.put(pattern, formatter);
        }
        return formatter;
    }


    /**
     * method : 把date的月数增加offset,比如1月2号加1后变为2月2号。
     * <br/>这里要注意1月30号，+1后会变成3月2号（因为2月没有30号，溢出了）。
     */
    public static Date monthsAddOrSub(Date date, int offset) {
        return addOrSub(date, Calendar.MONTH, offset);
    }

    /**
     * method : 24、把date的天数增加offset
     */
    public static Date daysAddOrSub(Date date, int offset) {
        return addOrSub(date, Calendar.DAY_OF_MONTH, offset);
    }

    /**
     * method : 25、把date的小时数增加offset
     */
    public static Date hoursAddOrSub(Date date, int offset) {
        return addOrSub(date, Calendar.HOUR_OF_DAY, offset);
    }

    /**
     * method : 26、把date的分钟数增加offset
     */
    public static Date minutesAddOrSub(Date date, int offset) {
        return addOrSub(date, Calendar.MINUTE, offset);
    }

    /**
     * method : 27、把date的秒数增加offset
     */
    public static Date secondsAddOrSub(Date date, int offset) {
        return addOrSub(date, Calendar.SECOND, offset);
    }

    /**
     * method : 28、是否为TimeString 样式的String 00:00 - 23:59
     */
    public static boolean isTimeString(String toCheck) {
        if (!StringUtils.isNotBlank(toCheck)) {
            return false;
        }
        return toCheck.trim().matches("([0-1][0-9]|2[0-3]):[0-5][0-9]|24:00");
    }

    /**
     * method : 29、比较两个TimeString 的大小 大于0表示ts1 > ts2，等于0表示相等，小于0表示ts1 < ts2
     */
    public static int compareHHmmInString(String ts1, String ts2) {
        return ts1.compareTo(ts2);
    }

    /**
     * method : 30、检测是否在开始与结束之间，前闭后开区间 -1： start 不小于end 0: 不在start 与end之间 1:
     * 在start与end之间
     */
    public static int betweenHHmmInString(String ts, String start, String end) {
        if (compareHHmmInString(start, end) >= 0)
            return -1;
        if (compareHHmmInString(ts, start) < 0)
            return 0;
        if (compareHHmmInString(end, ts) <= 0)
            return 0;
        return 1;
    }

    /**
     * method : 31、检测两个时间是否相等, 00:00 == 24:00
     */
    public static boolean equalsInTimeString(String ts1, String ts2) {
        if (ts1.equals(ts2))
            return true;
        if (("00:00".equals(ts1) || "24:00".equals(ts1)) && ("00:00".equals(ts2) || "24:00".equals(ts2))) {
            return true;
        }
        return false;
    }

    /**
     * 获取yyyyMMddHHmmssSSS
     */
    public static String getMillisecond() {
        return formatDateTime(new Date(), MILLI_SECOND_FORMAT);
    }

    /**
     * 获取yyyyMMddHHmmss
     */
    public static String getSecondNoSymbol() {
        return formatDateTime(new Date(), SECOND_FORMAT_NO_SYMBOL);
    }

    /**
     * 获取yyyyMMddHHmmss
     */
    public static String getSecondNoSymbol(Date date) {
        return formatDateTime(date, SECOND_FORMAT_NO_SYMBOL);
    }

    /**
     * method : 32、检测是否为同一个小时和分钟. 小时为24小时制 本方法只比较HH:mm不会判断哪年哪月哪日 1: cal>c;
     * 0:cal=c; -1:cal<c
     */
    public static int compareHHmm(Calendar cal, Calendar c) {
        if (cal.get(Calendar.HOUR_OF_DAY) > c.get(Calendar.HOUR_OF_DAY)) {
            return 1;
        } else if (cal.get(Calendar.HOUR_OF_DAY) == c.get(Calendar.HOUR_OF_DAY)) {
            if (cal.get(Calendar.MINUTE) > c.get(Calendar.MINUTE))
                return 1;
            else if (cal.get(Calendar.MINUTE) == c.get(Calendar.MINUTE))
                return 0;
            else
                return -1;
        } else
            return -1;
    }

    /**
     * method : 33、检测是否在开始与结束之间，闭区间 小时为24小时制 本方法只比较HH:mm不会判断哪年哪月哪日 -1： start
     * 不小于end 0: 不在start 与end之间 1: 在start与end之间
     */
    public static int betweenHHmm(Calendar cal, Calendar start, Calendar end) {
        if (compareHHmm(start, end) != -1)
            return -1;
        if (compareHHmm(cal, start) == -1)
            return 0;
        if (compareHHmm(end, cal) == -1)
            return 0;
        return 1;
    }

    /**
     * method : 34、检测是否为同在一天. 1: cal>c; 0:cal=c; -1:cal<c
     */
    public static boolean compareDay(Calendar cal, Calendar c) {
        return (cal.get(Calendar.YEAR) == c.get(Calendar.YEAR) && cal.get(Calendar.MONTH) == c.get(Calendar.MONTH)
                && cal.get(Calendar.DAY_OF_MONTH) == c.get(Calendar.DAY_OF_MONTH));
    }

    /**
     * method : 35、将00:00格式的字符串转为Calender
     */
    public static Calendar string2calendar(String timeString) throws Exception {
        if (!isTimeString(timeString))
            throw new Exception("Wrong argument : timeString format error " + timeString);
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.HOUR_OF_DAY, getHour(timeString));
        cal.set(Calendar.MINUTE, getMinute(timeString));
        return cal;
    }

    /**
     * method : 36、将calendar转为HH:MM格式的字符串
     */
    public static String calendar2TimeString(Calendar cal) {
        return (cal.get(Calendar.HOUR_OF_DAY) > 9 ? cal.get(Calendar.HOUR_OF_DAY) : "0" + cal.get(Calendar.HOUR_OF_DAY)) + ":"
                + (cal.get(Calendar.MINUTE) > 9 ? cal.get(Calendar.MINUTE) : "0" + cal.get(Calendar.MINUTE));
    }

    /**
     * method : 37、TimeString 中得到小时信息 比如 23:25那么获取23
     */
    public static int getHour(String timeString) {
        return Integer.parseInt(timeString.substring(0, 2));
    }

    /**
     * method : 38、TimeString 中得到分钟信息 比如 23:25那么获取25
     */
    public static int getMinute(String timeString) {
        return Integer.parseInt(timeString.substring(3, 5));
    }

    /**
     * method : 39、从Date的字符串中只得到月日信息
     */
    public static String getDateOnlyFromDate(String dateString) {
        return dateString.substring(5, 10);
    }

    /**
     * method : 40、将calendar转为MM-dd格式的字符串
     */
    public static String calendar2DateString(Calendar cal) {
        return (cal.get(Calendar.MONTH) + 1 > 9 ? cal.get(Calendar.MONTH) + 1 : "0" + (cal.get(Calendar.MONTH) + 1)) + "-"
                + (cal.get(Calendar.DAY_OF_MONTH) > 9 ? cal.get(Calendar.DAY_OF_MONTH) : "0" + cal.get(Calendar.DAY_OF_MONTH));
    }

    /**
     * 日期月份相减
     *
     * @param beginDate
     * @param endDate
     * @return
     */
    public static Integer monthSub(Date beginDate, Date endDate) {
        Integer monthNum = 0;
        Integer yearNumber = 0;
        Calendar startCalendar = Calendar.getInstance();
        Calendar endCalendar = Calendar.getInstance();

        startCalendar.setTime(beginDate);
        endCalendar.setTime(endDate);

        yearNumber = endCalendar.get(Calendar.YEAR) - startCalendar.get(Calendar.YEAR);
        monthNum = endCalendar.get(Calendar.MONTH) - startCalendar.get(Calendar.MONTH);

        return yearNumber * 12 + monthNum;
    }

    /**
     * 获取当前日期月份的第N天
     * @param beginDate
     * @param index
     * @return
     */
    public static Date getMonthDayByIndex(Date beginDate, int index) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(beginDate);
        cal.set(Calendar.DAY_OF_MONTH,index);
        return cal.getTime();
    }

    /**
     * 获取今天0点到明天0点
     * @return
     */
    public static Map<String,Date> retTodayAndTomorrow(){
        Map<String,Date> dateMap = new HashMap<>();
        Date start = getDayDate(new Date()); //获取今天 0点0分0秒 日期
        Calendar cal = Calendar.getInstance();
        cal.setTime(start);
        cal.add(Calendar.DATE, 1);
        Date end = cal.getTime(); //获取明天 0点0分0秒 日期
        dateMap.put("startDay", start);
        dateMap.put("endDay", end);
        return dateMap;
    }

    /**
     * 获取当前时间到明天0点的秒数（至少返回1，以防redis用0作为超时时间报错）
     */
    public static int getToTomorrowSeconds(){
        Calendar curDate = Calendar.getInstance();
        Calendar tommorowDate = new GregorianCalendar(curDate.get(Calendar.YEAR), curDate.get(Calendar.MONTH),
                curDate.get(Calendar.DATE) + 1, 0, 0, 0);
        return Math.max((int) ((tommorowDate.getTimeInMillis() - curDate.getTimeInMillis()) / 1000), 1);
    }

    /**
     * 获取今天零点零分零秒 时间搓
     * @return
     */
    public static long getTodayZeroTimeInMillis(){
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.MILLISECOND, 0);
        return cal.getTimeInMillis();
    }

    /**
     * 获取当前月第一天，零点零分零秒 时间搓
     * @return
     */
    public static long getMonthFirstDayTimeInMillis(){
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.MONTH, 0);
        calendar.set(Calendar.DAY_OF_MONTH, 1);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTimeInMillis();
    }

    /**
     * 获取下一月第一天，零点零分零秒 时间搓
     * @return
     */
    public static long getNextMonthFirstDayTimeInMillis(){
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        calendar.add(Calendar.MONTH, 1);
        calendar.set(Calendar.DAY_OF_MONTH, 1);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTimeInMillis();
    }

    /**
     * 获取相前时间与传入时间的相差几天
     * @param date
     * @return
     */
    public static long getDifferDay(Date date){
        Date nowDate = new Date();
        if(nowDate.after(date)){
            return 0l;
        }
        return Math.abs(date.getTime() - nowDate.getTime())/(1000*24*60*60) +
                (Math.abs(date.getTime() - nowDate.getTime())%(1000*24*60*60) == 0? 0:1);
    }

    private static Date addOrSub(Date date, int type, int offset) {
        if (date == null)
            return null;
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.get(type);
        cal.set(type, cal.get(type) + offset);
        return cal.getTime();
    }

    private static String getStr(Date date, SimpleDateFormat format) {
        if (date == null) {
            return "";
        }
        return format.format(date);
    }

    private static Date getDate(String dateStr, SimpleDateFormat format) {
        if ("".equals(dateStr) || dateStr == null)
            return null;
        try {
            return format.parse(dateStr);
        } catch (ParseException e) {
            LOG.error("format yyyy-MM-dd HH:mm:ss error:", e);
        }
        return null;
    }

    /**
     * 根据日期 获取 该日期当周 周一日期
     * 如果日期为周一，那么获取上周一
     */
    public static Date getMondayDate(Date day) {
        Calendar calWeek = Calendar.getInstance();
        calWeek.setTime(day);
        if (Calendar.SUNDAY == calWeek.get(Calendar.DAY_OF_WEEK)) {
            calWeek.add(Calendar.DATE, -6);
        } else if (Calendar.MONDAY == calWeek.get(Calendar.DAY_OF_WEEK)) {
            calWeek.add(Calendar.DATE, -7);
        } else if (Calendar.TUESDAY == calWeek.get(Calendar.DAY_OF_WEEK)) {
            calWeek.add(Calendar.DATE, -1);
        } else if (Calendar.WEDNESDAY == calWeek.get(Calendar.DAY_OF_WEEK)) {
            calWeek.add(Calendar.DATE, -2);
        } else if (Calendar.THURSDAY == calWeek.get(Calendar.DAY_OF_WEEK)) {
            calWeek.add(Calendar.DATE, -3);
        } else if (Calendar.FRIDAY == calWeek.get(Calendar.DAY_OF_WEEK)) {
            calWeek.add(Calendar.DATE, -4);
        } else if (Calendar.SATURDAY == calWeek.get(Calendar.DAY_OF_WEEK)) {
            calWeek.add(Calendar.DATE, -5);
        }
        return getDayDate(calWeek.getTime());
    }

    /**
     * 根据日期获取 该日期当月1号的日期
     * 如果参数日期为1号，那么获取上月1号的日期
     */
    public static Date getFirstDayDate(Date day) {
        Calendar calMonth = Calendar.getInstance();
        calMonth.setTime(day);
        if (1 == calMonth.get(Calendar.DAY_OF_MONTH)) {
            calMonth.setTime(daysAddOrSub(day, -1));
        }
        calMonth.set(Calendar.DAY_OF_MONTH, 1);
        return getDayDate(calMonth.getTime());
    }

    /**
     * 获取00:00:00时间点
     *
     * @param date
     * @return Date 年月日不变，时分秒改为当天的00:00:00.000
     */
    public static Date getDayStartTime(Date date) {
        if (null == date) {
            return null;
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTime();
    }

    /**
     * 获取23:59:59时间点
     *
     * @param date
     * @return Date 年月日不变，时分秒改为当天的23:59:59.999
     */
    public static Date getDayEndTime(Date date) {
        if (null == date) {
            return null;
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        calendar.set(Calendar.MILLISECOND, 999);
        return calendar.getTime();
    }

    /**
     * 根据日期字符串获取该日期的开始时间 2015-12-01 00:00:00.000
     *
     * @param strDate
     * @return
     */
    public static Date getDayStartTime(String strDate) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(getDayDate(strDate));
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return calendar.getTime();
    }

    /**
     * 根据日期字符串获取该日期的结束时间 2015-12-01 23:59:59.999
     *
     * @param strDate
     * @return
     */
    public static Date getDayEndTime(String strDate) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(getDayDate(strDate));
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        calendar.set(Calendar.MILLISECOND, 999);
        return calendar.getTime();
    }

    /**
     * 计算两个日期之间相差的天数
     * @param smdate 较小的时间
     * @param bdate  较大的时间
     * @return 相差天数
     * @throws ParseException
     */
    public static int daysBetween(Date smdate, Date bdate) {
        smdate = getDayDate(smdate);
        bdate = getDayDate(bdate);
        Calendar cal = Calendar.getInstance();
        cal.setTime(smdate);
        long time1 = cal.getTimeInMillis();
        cal.setTime(bdate);
        long time2 = cal.getTimeInMillis();
        long betweenDays = (time2 - time1) / (1000 * 3600 * 24);
        return Integer.parseInt(String.valueOf(betweenDays));
    }

    /**
     * 日期增加或减少天数
     *
     * @param date
     * @param day 负数为减少，正数为增加
     * @return
     */
    public static Date changeByDay(Date date, int day) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.DAY_OF_MONTH, day);
        return cal.getTime();
    }

    /**
     * 获取昨日日期
     * @return
     * @throws Exception
     */
    public static String getYesterday() {
        SimpleDateFormat dft = new SimpleDateFormat("yyyy-MM-dd");
        Calendar date = Calendar.getInstance();
        date.setTime(new Date());
        date.set(Calendar.DATE, date.get(Calendar.DATE) - 1);
        Date endDate = null;
        try {
            endDate = dft.parse(dft.format(date.getTime()));
        } catch (ParseException e) {
            throw new RuntimeException(e);
        }
        return dft.format(endDate);
    }



    /**
     * String 转时间
     *
     * @param timeStr
     * @return
     */
    public static LocalDateTime parseTime(String timeStr) {
        return LocalDateTime.parse(timeStr, DEFAULT_DATETIME_FORMATTER);
    }


    /**
     * String 转时间
     *
     * @param timeStr
     * @param format  时间格式
     * @return
     */
    public static LocalDateTime parseTime(String timeStr, TimeFormat format) {
        return LocalDateTime.parse(timeStr, format.formatter);
    }


    public static LocalDate parseLocalDate(String timeStr, TimeFormat format) {
        return LocalDate.parse(timeStr, format.formatter);
    }


    /**
     * 时间转 String
     *
     * @param time
     * @return
     */
    public static String parseTime(LocalDateTime time) {
        return DEFAULT_DATETIME_FORMATTER.format(time);
    }


    /**
     * 时间转 String
     *
     * @param time
     * @param format 时间格式
     * @return
     */
    public static String parseTime(LocalDateTime time, TimeFormat format) {
        return format.formatter.format(time);
    }


    public static String parseLocalDate(LocalDate time, TimeFormat format) {
        return format.formatter.format(time);
    }


    /**
     * 获取当前时间
     *
     * @return
     */
    public static String getCurrentDatetime() {
        return DEFAULT_DATETIME_FORMATTER.format(LocalDateTime.now());
    }


    /**
     * 获取当前时间
     *
     * @param format 时间格式
     * @return
     */
    public static String getCurrentDatetime(TimeFormat format) {
        return format.formatter.format(LocalDateTime.now());
    }


    /**
     * 时间格式
     */
    public enum TimeFormat {


        /**
         * 短时间格式
         */
        SHORT_DATE_PATTERN_CHINESE("yyyy年MM月dd日"), SHORT_DATE_PATTERN_LINE("yyyy-MM-dd"), SHORT_DATE_PATTERN_SLASH("yyyy/MM/dd"), SHORT_DATE_PATTERN_DOUBLE_SLASH("yyyy\\MM\\dd"), SHORT_DATE_PATTERN_NONE("yyyyMMdd"), SHORT_DATE_PATTERN_MINUTES("yyyyMMddHHmm"), SHORT_DATE_PATTERN_HOURS("yyyyMMddHH"),

        /**
         * 长时间格式
         */
        LONG_DATE_PATTERN_LINE("yyyy-MM-dd HH:mm:ss"), LONG_DATE_PATTERN_SLASH("yyyy/MM/dd HH:mm:ss"), LONG_DATE_PATTERN_DOUBLE_SLASH("yyyy\\MM\\dd HH:mm:ss"), LONG_DATE_PATTERN_NONE("yyyyMMdd HH:mm:ss"),


        /**
         * 长时间格式 带毫秒
         */
        LONG_DATE_PATTERN_WITH_MILSEC_LINE("yyyy-MM-dd HH:mm:ss.SSS"), LONG_DATE_PATTERN_WITH_MILSEC_SLASH("yyyy/MM/dd HH:mm:ss.SSS"), LONG_DATE_PATTERN_WITH_MILSEC_DOUBLE_SLASH("yyyy\\MM\\dd HH:mm:ss.SSS"), LONG_DATE_PATTERN_WITH_MILSEC_NONE("yyyyMMdd HH:mm:ss.SSS");


        private transient DateTimeFormatter formatter;


        TimeFormat(String pattern) {
            formatter = DateTimeFormatter.ofPattern(pattern);
        }
    }


    /**
     * 获取Period（时间段）
     *
     * @param lt 较小时间
     * @param gt 较大时间
     * @return
     */
    public static Period LocalDateDiff(LocalDate lt, LocalDate gt) {
        Period p = Period.between(lt, gt);
        return p;
    }


    /**
     * 获取时间间隔，并格式化为XXXX年XX月XX日
     *
     * @param lt 较小时间
     * @param gt 较大时间
     * @return
     */
    public static String localDateDiffFormat(LocalDate lt, LocalDate gt) {
        Period p = Period.between(lt, gt);
        String str = String.format(" %d年 %d月 %d日", p.getYears(), p.getMonths(), p.getDays());
        return str;
    }


    /**
     * 获取Duration（持续时间）
     *
     * @param lt 较小时间
     * @param gt 较大时间
     * @return
     */
    public static Duration localTimeDiff(LocalTime lt, LocalTime gt) {
        Duration d = Duration.between(lt, gt);
        return d;
    }


    /**
     * 获取时间间隔（毫秒）
     *
     * @param lt 较小时间
     * @param gt 较大时间
     * @return
     */
    public static long millisDiff(LocalTime lt, LocalTime gt) {
        Duration d = Duration.between(lt, gt);
        return d.toMillis();
    }


    /**
     * 获取时间间隔（秒）
     *
     * @param lt 较小时间
     * @param gt 较大时间
     * @return
     */
    public static long secondDiff(LocalTime lt, LocalTime gt) {
        Duration d = Duration.between(lt, gt);
        return d.getSeconds();
    }


    /**
     * 获取时间间隔（天）
     *
     * @param lt 较小时间
     * @param gt 较大时间
     * @return
     */
    public static long daysDiff(LocalDate lt, LocalDate gt) {
        long daysDiff = ChronoUnit.DAYS.between(lt, gt);
        return daysDiff;
    }

    /**
     * 获取时间间隔（小时）
     *
     * @param lt 较小时间
     * @param gt 较大时间
     * @return
     */
    public static long hoursDiff(LocalDate lt, LocalDate gt) {
        long hoursDiff = ChronoUnit.HOURS.between(lt, gt);
        return hoursDiff;
    }


    /**
     * 创建一个新的日期，它的值为上月的第一天
     *
     * @param date
     * @return
     */
    public static LocalDate getFirstDayOfLastMonth(LocalDate date) {
        return date.with((temporal) -> temporal.with(DAY_OF_MONTH, 1).plus(-1, MONTHS));


    }

    /**
     * 创建一个新的日期，它的值为上月的最后一天
     *
     * @param date
     * @return
     */
    public static LocalDate getLastDayOfLastMonth(LocalDate date) {
        return date.with((temporal) -> temporal.with(DAY_OF_MONTH, temporal.range(DAY_OF_MONTH).getMaximum()).plus(-1, MONTHS));


    }

    /**
     * 创建一个新的日期，它的值为当月的第一天
     *
     * @param date
     * @return
     */
    public static LocalDate getFirstDayOfMonth(LocalDate date) {
        return date.with(firstDayOfMonth());


    }

    /**
     * 创建一个新的日期，它的值为当月的最后一天
     *
     * @param date
     * @return
     */
    public static LocalDate getLastDayOfMonth(LocalDate date) {
        return date.with(lastDayOfMonth());
    }

    /**
     * 创建一个新的日期，它的值为下月的第一天
     *
     * @param date
     * @return
     */
    public static LocalDate getFirstDayOfNextMonth(LocalDate date) {
        return date.with(firstDayOfNextMonth());


    }

    /**
     * 创建一个新的日期，它的值为下月的最后一天
     *
     * @param date
     * @return
     */
    public static LocalDate getLastDayOfNextMonth(LocalDate date) {
        return date.with((temporal) -> temporal.with(DAY_OF_MONTH, temporal.range(DAY_OF_MONTH).getMaximum()).plus(1, MONTHS));


    }


    /**
     * 创建一个新的日期，它的值为上年的第一天
     *
     * @param date
     * @return
     */
    public static LocalDate getFirstDayOfLastYear(LocalDate date) {
        return date.with((temporal) -> temporal.with(DAY_OF_YEAR, 1).plus(-1, YEARS));
    }

    /**
     * 创建一个新的日期，它的值为上年的最后一天
     *
     * @param date
     * @return
     */
    public static LocalDate getLastDayOfLastYear(LocalDate date) {
        return date.with((temporal) -> temporal.with(DAY_OF_YEAR, temporal.range(DAY_OF_YEAR).getMaximum()).plus(-1, YEARS));
    }


    /**
     * 创建一个新的日期，它的值为当年的第一天
     *
     * @param date
     * @return
     */
    public static LocalDate getFirstDayOfYear(LocalDate date) {
        return date.with(firstDayOfYear());
    }


    /**
     * 创建一个新的日期，它的值为今年的最后一天
     *
     * @param date
     * @return
     */
    public static LocalDate getLastDayOfYear(LocalDate date) {
        return date.with(lastDayOfYear());
    }


    /**
     * 创建一个新的日期，它的值为明年的第一天
     *
     * @param date
     * @return
     */
    public static LocalDate getFirstDayOfNextYear(LocalDate date) {
        return date.with(firstDayOfNextYear());
    }


    /**
     * 创建一个新的日期，它的值为明年的最后一天
     *
     * @param date
     * @return
     */
    public static LocalDate getLastDayOfNextYear(LocalDate date) {
        return date.with((temporal) -> temporal.with(DAY_OF_YEAR, temporal.range(DAY_OF_YEAR).getMaximum()).plus(1, YEARS));
    }


    /**
     * 创建一个新的日期，它的值为同一个月中，第一个符合星期几要求的值
     *
     * @param date
     * @return
     */
    public static LocalDate getFirstInMonth(LocalDate date, DayOfWeek dayOfWeek) {
        return date.with(firstInMonth(dayOfWeek));
    }


    /**
     * 创建一个新的日期，并将其值设定为日期调整后或者调整前，第一个符合指定星 期几要求的日期
     *
     * @param date
     * @return
     */
    public static LocalDate getNext(LocalDate date, DayOfWeek dayOfWeek) {
        return date.with(next(dayOfWeek));
    }


    /**
     * 创建一个新的日期，并将其值设定为日期调整后或者调整前，第一个符合指定星 期几要求的日期
     *
     * @param date
     * @return
     */
    public static LocalDate getPrevious(LocalDate date, DayOfWeek dayOfWeek) {
        return date.with(previous(dayOfWeek));
    }


    /**
     * 创建一个新的日期，并将其值设定为日期调整后或者调整前，第一个符合指定星 期几要求的日期，如果该日期已经符合要求，直接返回该对象
     *
     * @param date
     * @return
     */
    public static LocalDate getNextOrSame(LocalDate date, DayOfWeek dayOfWeek) {
        return date.with(nextOrSame(dayOfWeek));
    }


    /**
     * 创建一个新的日期，并将其值设定为日期调整后或者调整前，第一个符合指定星 期几要求的日期，如果该日期已经符合要求，直接返回该对象
     *
     * @param date
     * @return
     */
    public static LocalDate getPreviousOrSame(LocalDate date, DayOfWeek dayOfWeek) {
        return date.with(previousOrSame(dayOfWeek));
    }

    /**
     * 判断是否是周末
     *
     * @param date
     * @return Boolean false:说明不是周末 true :说明是周末
     */
    public static Boolean checkIsWeek(LocalDate date) {
        if ((date != getNextOrSame(LocalDate.now(), DayOfWeek.MONDAY)) && (date != getNextOrSame(LocalDate.now(), DayOfWeek.SATURDAY))) {
            return Boolean.FALSE;
        }
        return Boolean.TRUE;
    }

    public static String dateToStr(Date dateDate) {
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        String dateString = formatter.format(dateDate);
        return dateString;

    }

    public static Long getTimeDifference(Long time) {
        return (System.currentTimeMillis() - time)/1000;
    }
}