/**
 * Project Name:activity-center-api<br>
 * File Name:Probability.java<br>
 * Package Name:cn.com.duiba.tuia.adx.center.api.common<br>
 * Date:2017年3月30日上午10:35:09<br>
 * Copyright (c) 2017, duiba.com.cn All Rights Reserved.<br>
 *
 */
package cn.com.duiba.tuia.adx.center.api.common;

import com.alibaba.fastjson.JSON;

import java.math.BigDecimal;
import java.util.Random;

/**
 * ClassName: Probability <br/>
 * Function: 概率. <br/>
 * Reason: 精确表示概率. <br/>
 * 部分代码与注释参考 org.apache.commons.lang3.math.Fraction
 * date: 2017年3月30日 上午10:35:09 <br/>
 *
 * @author wubo
 * @version 
 * @since JDK 1.7
 */
public class Probability extends Number{
    
    private static final long serialVersionUID = 1L;

    private final static Random random = new Random();
    
    private transient int hashCode = 0;
    /** numerator:分子.*/
    private int numerator;
    /** denominator:分母.*/
    private int denominator = 100;
    
    public Probability(){
        super();
    }
    
    /**
     * Creates a new instance of Probability.
     *
     * @param numerator
     * @param denominator
     */
    public Probability(int numerator,int denominator){
        checkArg(numerator,denominator);
        this.numerator = numerator;
        this.denominator = denominator;
    }
    
    /**
     * Creates a new instance of Probability.
     *
     */
    public Probability(int persent){
        this(persent,100);
    }
    
    public int getNumerator() {
        return numerator;
    }

    
    public void setNumerator(int numerator) {
        this.numerator = numerator;
    }

    
    public int getDenominator() {
        return denominator;
    }

    
    public void setDenominator(int denominator) {
        this.denominator = denominator;
    }

    /**
     * getProbability:Creates a new instance with x/y. <br/>
     * @param numerator
     * @param denominator
     * @return
     */
    public static Probability getProbability(int numerator,int denominator){
        return new Probability(numerator,denominator);
    }
    
    /**
     * getProbability:Creates a new instance with p%. <br/>
     * @param percent
     * @return
     */
    public static Probability getProbability(int percent){
        return new Probability(percent,100);
    }

    public static Probability getProbability(Integer percent) {
        if (percent == null) {
            return new Probability(0, 100);
        } else {
            return new Probability(percent, 100);
        }
    }
    
    /**
     * getProbability:Creates a new instance with string value. <br/>
     * @param value e.g. 0.xx
     * @return
     */
    public static Probability getProbability(String value){
        if (value == null) {
            throw new IllegalArgumentException("The string must not be null");
        }
        
        Double dvalue = Double.parseDouble(value.trim());
        
        if (dvalue < 0){
            throw new IllegalArgumentException("Probability cannot be less than 0 ");
        }
        
        if (dvalue > 1){
            throw new IllegalArgumentException("Probability cannot be greater than 1 ");
        }
        
        if (dvalue == 1){
            return getProbability(100);
        }
        
        if (dvalue == 0) {
            return getProbability(0);
        }
        
        //get numerator char,
        //example:
        //0.111  -> 111
        //0.011  -> 11
        //0.110  -> 11
        //0.101  -> 101
        int length = value.length();
        
        
        int pl = getUselessPrefixLength(value);
        int sl = getUselessSuffixLength(value);
        
        int nl = length - pl - sl;
        char[] nc = new char[nl];
        
        for(int i = 0;i<nl;i++){
            nc[i] = value.charAt(pl++);
        }
        
        //calculate denominator
        int denominator = 1;
        
        int dl = length - sl - 2;
        
        for(int i=0;i<dl;i++){
            denominator*=10;
        }
        
        return new Probability(Integer.valueOf(new String(nc)), denominator);
        
    }
    
    /*
     * 获取无效前缀字符的位数(小数前面的0和小数点字符)
     * example:
     * 0.011 ->  3
     * 0.1   ->  2
     */
    private static int getUselessPrefixLength(String value){
        int i = 2;
        for(;i<value.length();i++){
            if(value.charAt(i)!='0'){
                break;
            }
        }
        return i;
    }
    
    /*
     * 获取无效后缀字符的位数(小数后面跟的0)
     * example:
     * 0.100 -> 2
     * 0.1   -> 0
     */
    private static int getUselessSuffixLength(String value){
        int i = value.length();
        for(;i>0;i--){
            if(value.charAt(i-1) !='0'){
                break;
            }
        }
        return value.length()-i;
    }
    
    /*
     * check numerator and denominator<br/>
     * denominator > numerator >= 0
     */
    private void checkArg(int numerator,int denominator){
        if(numerator<0){
            throw new IllegalArgumentException("Probability cannot be less than 0 ");
        }
        
        if(numerator>denominator){
            throw new IllegalArgumentException("Probability cannot be greater than 1 ");
        }
        
        if(denominator==0){
            throw new IllegalArgumentException("Denominator cannot be 0 ");
        }
    }

    /**
     * <p>Gets the probability as an <code>int</code>. This returns the whole number
     * part of the probability.</p>
     *
     * @return the whole number probability part
     */
    @Override
    public int intValue() {
        return numerator / denominator;
    }

    /**
     * <p>Gets the probability as a <code>long</code>. This returns the whole number
     * part of the probability.</p>
     *
     * @return the whole number probability part
     */
    @Override
    public long longValue() {
        return (long) numerator / denominator;
    }

    /**
     * <p>Gets the probability as a <code>float</code>. This calculates the probability
     * as the numerator divided by denominator.</p>
     *
     * @return the probability as a <code>float</code>
     */
    @Override
    public float floatValue() {
        return ((float) numerator) / ((float) denominator);
    }

    /**
     * <p>Gets the probability as a <code>double</code>. This calculates the probability
     * as the numerator divided by denominator.</p>
     *
     * @return the probability as a <code>double</code>
     */
    @Override
    public double doubleValue() {
        return ((double) numerator) / ((double) denominator);
    }

    /**
     * <p>Gets the probability as a percent <code>int</code>. This returns the whole number
     * part of the probability.</p>
     *
     * @return the whole number probability part
     */
    public int intPercentValue() {
        return numerator*100/denominator;
    }

    /**
     * <p>Gets the probability as a <code>String</code>. This calculates the probability
     * as the numerator divided by denominator.</p>
     *
     * @return the probability as a <code>String</code>
     */
    public String stringValue() {
        return Double.toString(doubleValue());
    }
    
    /**
     * <p>Gets the probability keep scale decimal places as a <code>String</code>. This calculates the probability
     * as the numerator divided by denominator. </p>
     *
     * @return the probability as a <code>String</code>
     */
    public String stringValue(int scale) {
        return BigDecimal.valueOf(doubleValue()).setScale(scale,BigDecimal.ROUND_HALF_UP).toString();
    }

    /**
     * <p>Gets the probability as a percent <code>String</code>. This calculates the probability
     * as the numerator divided by denominator.</p>
     *
     * @return the probability as a <code>String</code>
     */
    public String stringPercentValue() {
        return Double.toString(((double) (numerator*100)) / ((double) denominator));
    }

    /**
     * <p>Gets the probability keep scale decimal places as a percent <code>String</code>. This calculates the probability
     * as the numerator divided by denominator. </p>
     *
     * @return the probability as a <code>String</code>
     */
    public String stringPercentValue(int scale) {
        return BigDecimal.valueOf(((double) (numerator*100)) / ((double) denominator)).setScale(scale, BigDecimal.ROUND_HALF_UP).toString();
    }
    
    public String toPercentString(){
        return this.stringPercentValue(2)+"%";
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj == null) {
            return false;
        }
        if (obj == this) {
            return true;
        }
        if (!(obj instanceof Probability)) {
            return false;
        }
        final Probability other = (Probability) obj;
        return other.getDenominator()*this.numerator == this.denominator * other.getNumerator();
    }
    @Override
    public int hashCode() {
        if (hashCode == 0) {
            // hashcode update should be atomic.
            hashCode = 37 * (37 * 17 + getNumerator()) + getDenominator();
        }
        return hashCode;
    }

    @Override
    public String toString() {
        return JSON.toJSONString(this);
    }

    /**
     * 根据当前概率,随机返回true
     * @return boolean
     */
    public boolean random() {
        return random.nextInt(denominator) < numerator;
//        return RandomUtils.nextInt(0, denominator) < numerator;
    }
    
}
