package cn.com.duiba.tuia.engine.activity.remote.sdk;

import cn.com.duiba.tuia.ssp.center.api.dto.PageResultDto;
import cn.com.duiba.tuia.ssp.center.api.dto.Result;
import cn.com.duiba.tuia.ssp.center.api.dto.sdk.*;
import cn.com.duiba.tuia.ssp.center.api.query.SdkAggreCrashLogQuery;
import cn.com.duiba.tuia.ssp.center.api.query.SdkCrashLogQuery;
import cn.com.duiba.tuia.ssp.center.api.remote.sdk.RemoteSdkCrashLogService;
import cn.com.duiba.tuia.ssp.center.api.tool.RemoteExcuteTemplateHandle;
import cn.com.duiba.tuia.utils.JsonUtils;
import com.alibaba.fastjson.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.List;

/**
 * @author K. L. Mao
 * @create 2020-01-02
 */
@Service
public class SdkCrashLogService {

    private Logger logger = LoggerFactory.getLogger(getClass());

    @Resource
    private RemoteSdkCrashLogService remoteSdkCrashLogService;

    /**
     * SDK 崩溃日志列表
     * @param crashLogQuery
     */
    public PageResultDto<SdkCrashLogDTO> listSdkCrashLog(SdkCrashLogQuery crashLogQuery) {
        Result<PageResultDto<SdkCrashLogDTO>> result = RemoteExcuteTemplateHandle.execute(() -> remoteSdkCrashLogService.listSdkCrashLog(crashLogQuery),
                logger, JSONObject.toJSONString(crashLogQuery));
        Assert.isTrue(result != null && result.isSuccess(), "SDK 崩溃日志列表，查询失败，errMsg:" + result.getDesc());
        return result.getData();
    }

    /**
     * SDK 崩溃日志详情
     * @param crashLogId
     */
    public SdkCrashLogDetailDTO crashLogDetail(Long crashLogId) {
        Result<SdkCrashLogDetailDTO> result = RemoteExcuteTemplateHandle.execute(() -> remoteSdkCrashLogService.sdkCrashLogDetail(crashLogId),
                logger, JSONObject.toJSONString(crashLogId));
        Assert.isTrue(result != null && result.isSuccess(), "SDK 崩溃日志详情，查询失败，errMsg:" + result.getDesc());
        return result.getData();
    }

    /**
     * SDK 崩溃日志处理
     * @param crashLogId
     */
    public void resolveAction(Long crashLogId) {
        Result<Void> result = RemoteExcuteTemplateHandle.execute(() -> remoteSdkCrashLogService.resolveAction(crashLogId),
                logger, JSONObject.toJSONString(crashLogId));
        Assert.isTrue(result != null && result.isSuccess(), "SDK 崩溃日志处理，查询失败，errMsg:" + result.getDesc());
    }

    /**
     * SDK 崩溃日志批量处理
     * @param sdkLogIds
     */
    public void resolveActionV2(List<Long> sdkLogIds, Integer status) {
        Result<Void> result = RemoteExcuteTemplateHandle.execute(() -> remoteSdkCrashLogService.resolveActionV2(sdkLogIds, status),
                logger, JSONObject.toJSONString(sdkLogIds));
        Assert.isTrue(result != null && result.isSuccess(), "SDK 崩溃日志处理 V2，查询失败，errMsg:" + result.getDesc());
    }

    /**
     * SDK 异常概览和趋势线
     * @param crashLogQuery
     */
    public SdkCrashOverviewDTO overview(SdkCrashLogQuery crashLogQuery) {
        Result<SdkCrashOverviewDTO> result = RemoteExcuteTemplateHandle.execute(() -> remoteSdkCrashLogService.overview(crashLogQuery),
                logger, JSONObject.toJSONString(crashLogQuery));
        Assert.isTrue(result != null && result.isSuccess(), "SDK 异常概览和趋势线，查询失败，errMsg:" + result.getDesc());
        return result.getData();
    }

    /**
     * 插入崩溃日志
     * @param sdkCrashLogDTO
     * @return
     */
    public void insertSdkCrashLog(SdkCrashLogDTO sdkCrashLogDTO) {
        Result<Void> result = RemoteExcuteTemplateHandle.execute(() -> remoteSdkCrashLogService.insertSdkCrashLog(sdkCrashLogDTO),
                logger, JsonUtils.objectToString(sdkCrashLogDTO));
        Assert.isTrue(result != null && result.isSuccess(), "插入 sdk 崩溃信息，请求失败，appId:" + sdkCrashLogDTO.getAppId());
    }

    /**
     *  聚合的崩溃日志
     * @param crashLogQuery
     * @return
     */
    public Result<PageResultDto<SdkAggreCrashLogDTO>> listAggreSdkCrashLog(SdkCrashLogQuery crashLogQuery){
        Result<PageResultDto<SdkAggreCrashLogDTO>> result = RemoteExcuteTemplateHandle.execute(() -> remoteSdkCrashLogService.listAggreSdkCrashLog(crashLogQuery),
                logger, JSONObject.toJSONString(crashLogQuery));
        Assert.isTrue(result != null && result.isSuccess(), "SDK 聚合崩溃日志列表，查询失败，errMsg:" + result.getDesc());
        return result;
    }

    /**
     *  聚合sdk版本信息
     * @param crashLogQuery
     * @return
     */
    public Result<PageResultDto<SdkVersionAggreDTO>> listAggreSdkVersion(SdkAggreCrashLogQuery crashLogQuery){
        Result<PageResultDto<SdkVersionAggreDTO>> result = RemoteExcuteTemplateHandle.execute(() -> remoteSdkCrashLogService.listAggreSdkVersion(crashLogQuery),
                logger, JSONObject.toJSONString(crashLogQuery));
        Assert.isTrue(result != null && result.isSuccess(), "SDK 聚合版本信息列表，查询失败，errMsg:" + result.getDesc());
        return result;
    }

    /**
     *  聚合机型信息
     * @param crashLogQuery
     * @return
     */
    public Result<PageResultDto<SdkModeDeviceAggreDTO>> listAggreSdkModeDevice(SdkAggreCrashLogQuery crashLogQuery){
        Result<PageResultDto<SdkModeDeviceAggreDTO>> result = RemoteExcuteTemplateHandle.execute(() -> remoteSdkCrashLogService.listAggreSdkModeDevice(crashLogQuery),
                logger, JSONObject.toJSONString(crashLogQuery));
        Assert.isTrue(result != null && result.isSuccess(), "SDK 聚合机型信息列表，查询失败，errMsg:" + result.getDesc());
        return result;
    }

    /**
     *  聚合操作系统信息
     * @param crashLogQuery
     * @return
     */
    public Result<PageResultDto<SdkOsInfoAggreDTO>> listAggreSdkOsInfo(SdkAggreCrashLogQuery crashLogQuery){
        Result<PageResultDto<SdkOsInfoAggreDTO>> result = RemoteExcuteTemplateHandle.execute(() -> remoteSdkCrashLogService.listAggreSdkOsInfo(crashLogQuery),
                logger, JSONObject.toJSONString(crashLogQuery));
        Assert.isTrue(result != null && result.isSuccess(), "SDK 聚合系统信息列表，查询失败，errMsg:" + result.getDesc());
        return result;
    }

    /**
     *  查看更多记录
     * @param crashLogQuery
     * @return
     */
    public Result<PageResultDto<SdkCrashLogDTO>> listSdkCrashLogByIds(SdkAggreCrashLogQuery crashLogQuery){
        Result<PageResultDto<SdkCrashLogDTO>> result = RemoteExcuteTemplateHandle.execute(() -> remoteSdkCrashLogService.listSdkCrashLogByIds(crashLogQuery),
                logger, JSONObject.toJSONString(crashLogQuery));
        Assert.isTrue(result != null && result.isSuccess(), "SDK 查看更多记录列表，查询失败，errMsg:" + result.getDesc());
        return result;
    }

    public Result<SdkAggreCrashLogDTO> aggreDetail(SdkAggreCrashLogQuery crashLogQuery){
        Result<SdkAggreCrashLogDTO> result = RemoteExcuteTemplateHandle.execute(() -> remoteSdkCrashLogService.aggreSdkCrashLog(crashLogQuery),
                logger, JSONObject.toJSONString(crashLogQuery));
        Assert.isTrue(result != null && result.isSuccess(), "SDK 查看聚合日志记录详情，查询失败，errMsg:" + result.getDesc());
        return result;
    }

}
