/**
 * Project Name:tuia-core-api<br>
 * File Name:DuiBaItemClient.java<br>
 * Package Name:cn.com.duiba.tuia.core.api.client<br>
 * Date:2016年11月11日下午6:21:04<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */

package cn.com.duiba.tuia.api;

import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import cn.com.duiba.tuia.pangea.center.api.localservice.apollopangu.ApolloPanGuService;
import cn.com.duiba.tuia.ssp.center.api.remote.advertselect.dto.media.req.AuditedAdvertAndMaterial4AdDto;
import cn.com.duiba.tuia.ssp.center.api.remote.media.RemoteMeituanService;
import cn.com.duiba.tuia.ssp.center.api.remote.media.dto.CanServeAdvertDto;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;

import cn.com.duiba.tuia.domain.model.AdvQueryParam;
import cn.com.duiba.tuia.domain.model.AdvertSelectInvokeRsp;
import cn.com.duiba.tuia.ssp.center.api.dto.SlotMsInfoDto;
import cn.com.duiba.tuia.ssp.center.api.dto.StrategyCacheDto;
import cn.com.duiba.tuia.ssp.center.api.dto.advertselect.AdvertSelectedDto;
import cn.com.duiba.tuia.ssp.center.api.remote.RemoteManagerShieldStrategyService;
import cn.com.duiba.tuia.ssp.center.api.remote.RemoteMediaVisibleTagsService;
import cn.com.duiba.tuia.ssp.center.api.remote.RemoteToAdvertService;
import cn.com.duiba.tuia.ssp.center.api.remote.advertselect.dto.advert.AdvertAndTagSelected4AdRsp;
import cn.com.duiba.tuia.ssp.center.api.remote.advertselect.dto.advert.AdvertAndTagSelectedRsp;
import cn.com.duiba.tuia.ssp.center.api.remote.advertselect.remoteservice.advert.RemoteAdvertSelectV2Service;

/**
 * ClassName: DuiBaItemClient <br/>
 * Function: 兑吧ItemClient. <br/>
 * date: 2016年11月11日 下午6:21:04 <br/>
 *
 * @author leiliang
 * @version
 * @since JDK 1.6
 */
@Service
public class TuiaMediaClientService {

    private static Logger log = LoggerFactory.getLogger(TuiaMediaClientService.class);

    private static final int ELECT_APP_KEY = 1;

    // 美团这个媒体的广告位
//    @Value("${tuia.meituan.slotid:0}")
//    private Long MEITUAN_SLOT;

    @Resource
    protected RedisTemplate<String,Long>  redisTemplate;

    @Resource
    private ExecutorService executorService;

    @Autowired
    private RemoteMediaVisibleTagsService remoteMediaVisibleTagsService;

    @Autowired
    private RemoteAdvertSelectV2Service remoteAdvertSelectV2Service;

    @Resource
    private RemoteManagerShieldStrategyService                   remoteManagerShieldStrategyService;

    @Resource
    private RemoteToAdvertService remoteToAdvertService;

    @Autowired
    private RemoteMeituanService remoteMeituanService;

    @Autowired
    private ApolloPanGuService apolloPanGuService;

    private final LoadingCache<Integer, List<AdvertSelectedDto>> ELECT_APP_CACHE = CacheBuilder.newBuilder().initialCapacity(4000).maximumSize(5000)
            .refreshAfterWrite(5, TimeUnit.MINUTES).expireAfterWrite(3, TimeUnit.HOURS).build(new CacheLoader<Integer, List<AdvertSelectedDto>>() {
                @Override
                public List<AdvertSelectedDto> load(Integer key) {
                    return getElectAppIds();
                }

                @Override
                public ListenableFuture<List<AdvertSelectedDto>> reload(final Integer key, List<AdvertSelectedDto> oldValue) {
                    ListenableFutureTask<List<AdvertSelectedDto>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });

    private List<AdvertSelectedDto> getElectAppIds() {
        //获取已经开启了广告互选的媒体列表
        List<AdvertSelectedDto> advertSelectedList;
        try {
            advertSelectedList = remoteMediaVisibleTagsService.selectAppIds();
        } catch (Exception e) {
            log.error("remoteMediaVisibleTagsService.selectAppIdsById exception", e);
            AdvertSelectedDto dto = new AdvertSelectedDto();
            dto.setAppId(-1L);
            return Lists.newArrayList(dto);
        }
        if (CollectionUtils.isEmpty(advertSelectedList)) {
            return Collections.emptyList();
        }
        return advertSelectedList;
    }


    private LoadingCache<Long, SlotMsInfoDto> slotCache = CacheBuilder.newBuilder().initialCapacity(4000).maximumSize(5000)
            .refreshAfterWrite(1L, TimeUnit.HOURS).expireAfterWrite(3, TimeUnit.HOURS)
            .build(new CacheLoader<Long, SlotMsInfoDto>() {
                @Override
                public SlotMsInfoDto load(Long key) {
                    return get(key).orElse(null);
                }

                @Override
                public ListenableFuture<SlotMsInfoDto> reload(Long key, SlotMsInfoDto oldValue) {
                    ListenableFutureTask<SlotMsInfoDto> task = ListenableFutureTask.create(() -> get(key).orElse(oldValue));
                    executorService.submit(task);
                    return task;
                }
            });

    private Optional<SlotMsInfoDto> get(Long slotId) {
        try {
            return Optional.ofNullable(remoteToAdvertService.getSlotMsInfo(slotId));
        } catch (Exception e) {
            log.error("remoteToAdvertService.getSlotMsInfo happened error ,slotId:{},error:{}", slotId, e);
            return Optional.empty();
        }
    }


    public SlotMsInfoDto getSlotInfo(Long slotId) {
        try {
            return slotCache.getUnchecked(slotId);
        } catch (Exception e) {
            return null;
        }
    }
    /**
     * 判断请求的媒体是否开启了广告互选的功能
     *
     * @param appId 请求的媒体ID
     * @param advQueryParam
     * @return
     */
    public boolean isElectApp(Long appId, AdvQueryParam advQueryParam) throws Exception {
        if (appId == null) {
            return false;
        }
        List<AdvertSelectedDto> advertSelectedDtoList;
        try {
            advertSelectedDtoList = ELECT_APP_CACHE.get(ELECT_APP_KEY);
        } catch (Exception e) {
            log.error("isElectApp error", e);
            AdvertSelectedDto dto = new AdvertSelectedDto();
            dto.setAppId(-1L);
            advertSelectedDtoList = Lists.newArrayList(dto);
        }

        if(CollectionUtils.isEmpty(advertSelectedDtoList)){
            return false;
        }
        if(advertSelectedDtoList.get(0) != null && advertSelectedDtoList.get(0).getAppId() == -1){
            ELECT_APP_CACHE.refresh(ELECT_APP_KEY);
            throw new  IllegalAccessException(String.format("获取互选媒体缓存异常，appId=%s", appId));
        }

        //
        AdvertSelectedDto advertSelectedDto = advertSelectedDtoList.stream().filter(e -> appId.equals(e.getAppId())).findFirst().orElse(null);
        advQueryParam.setAdvertSelectedDto(advertSelectedDto);
        return advertSelectedDto != null;
    }

    private final LoadingCache<Long, Optional<AdvertSelectInvokeRsp>> MEDIA_SELECT_ADVERT_CACHE = CacheBuilder.newBuilder().initialCapacity(4000).maximumSize(5000)
            .refreshAfterWrite(5, TimeUnit.MINUTES).expireAfterWrite(3, TimeUnit.HOURS).build(new CacheLoader<Long, Optional<AdvertSelectInvokeRsp>>() {
                @Override
                public Optional<AdvertSelectInvokeRsp> load(Long slotId) {
                    AdvertSelectInvokeRsp rsp = getMediaSelectConfig(slotId);
                    return Optional.ofNullable(rsp);
                }

                @Override
                public ListenableFuture<Optional<AdvertSelectInvokeRsp>> reload(final Long key, Optional<AdvertSelectInvokeRsp> oldValue) {
                    ListenableFutureTask<Optional<AdvertSelectInvokeRsp>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });

    private AdvertSelectInvokeRsp getMediaSelectConfig(Long slotId) {

        AdvertSelectInvokeRsp rsp = new AdvertSelectInvokeRsp();
        AdvertAndTagSelected4AdRsp mediaSelectConfig;
        try {

            Set<String> slotSet = new HashSet<>();
            try {
                String meituanSlotStr = apolloPanGuService.getIdMapStrByKeyStr("meituan-slots");
                String[] split = meituanSlotStr.split(",");
                slotSet.addAll(Arrays.asList(split));
            } catch (Exception e) {
                log.error("查询盘古配置并序列化过程失败",e);
            }

            //判断广告位,如果是美团的，则进行特殊处理
            if(slotSet.contains(String.valueOf(slotId))){

                //根据美团的广告位，获取美团的互选广告以及素材
                mediaSelectConfig = getMeiTuanSelectAdvert(312089L);
            }else {
                // 原来逻辑
                // 获取媒体屏蔽列表
                mediaSelectConfig = remoteAdvertSelectV2Service.getSlotConfig4Ad(slotId);
            }

            rsp.setInvokeSuccess(true);
            rsp.setRsp(mediaSelectConfig);

        } catch (Exception e) {
            log.error("remoteAdvertSelectV2Service.getSlotConfig exception slotId:{}",slotId, e);
            rsp.setInvokeSuccess(false);
        }
        return rsp;
    }

    /**
     * 根据美团的广告位，获取美团的互选广告以及素材
     * @param slotId
     * @return
     */
    private AdvertAndTagSelected4AdRsp getMeiTuanSelectAdvert(Long slotId){

        AdvertAndTagSelected4AdRsp mediaSelectConfig = new AdvertAndTagSelected4AdRsp();

        List<CanServeAdvertDto> canServeAdvertDtos = remoteMeituanService.canServeAdvert(slotId);

        if(CollectionUtils.isEmpty(canServeAdvertDtos)){
            return mediaSelectConfig;
        }

        Set<Long> advertSets = canServeAdvertDtos.stream().map(CanServeAdvertDto::getAdvertId).collect(Collectors.toSet());

        // 映射广告ID->对象
        Map<Long, List<CanServeAdvertDto>> canServeAdvertMap = canServeAdvertDtos.stream().collect(Collectors.groupingBy(CanServeAdvertDto::getAdvertId));

        Set<AuditedAdvertAndMaterial4AdDto> auditedAdLists = Sets.newHashSet();

        // 构建返回对象
        advertSets.forEach(advertId ->{

            List<CanServeAdvertDto> canServeAdvertList = canServeAdvertMap.get(advertId);

            Set<AuditedAdvertAndMaterial4AdDto.Material> materialSet = canServeAdvertList.stream().map(dto -> {
                AuditedAdvertAndMaterial4AdDto.Material material = new AuditedAdvertAndMaterial4AdDto.Material();

                material.setId(dto.getMaterialId());
                material.setGmtModified(dto.getGmtModified());
                return material;
            }).collect(Collectors.toSet());

            AuditedAdvertAndMaterial4AdDto material4AdDto = new AuditedAdvertAndMaterial4AdDto();

            // 互选广告开始时间
            Date startDate = materialSet.stream().map(AuditedAdvertAndMaterial4AdDto.Material::getGmtModified).min(Date::compareTo).get();

            material4AdDto.setAdvertId(advertId);
            material4AdDto.setGmtModified(startDate);
            material4AdDto.setMaterialIds(materialSet);

            auditedAdLists.add(material4AdDto);
        });

        mediaSelectConfig.setSelectedAdvertIds(advertSets);
        mediaSelectConfig.setSelectedAdvertAndMaterialIds(auditedAdLists);


        return mediaSelectConfig;
    }

    public Optional<AdvertSelectInvokeRsp> getAppSelectConfigCache(Long slotId) {
        if (slotId == null) {
            return Optional.empty();
        }
        return MEDIA_SELECT_ADVERT_CACHE.getUnchecked(slotId);
    }

    private final LoadingCache<Long, Optional<StrategyCacheDto>> CACHE = CacheBuilder.newBuilder().initialCapacity(4000)
            .refreshAfterWrite(10, TimeUnit.MINUTES).expireAfterWrite(3, TimeUnit.HOURS).build(new CacheLoader<Long, Optional<StrategyCacheDto>>() {
        @Override
        public Optional<StrategyCacheDto> load(Long slotId)  {
            StrategyCacheDto strategyCacheDto = getStrategyBySlotId(slotId);
            return Optional.ofNullable(strategyCacheDto);
        }

        @Override
        public ListenableFuture<Optional<StrategyCacheDto>> reload(final Long key, Optional<StrategyCacheDto> oldValue)  {
            ListenableFutureTask<Optional<StrategyCacheDto>> task = ListenableFutureTask.create(() -> load(key));
            executorService.submit(task);
            return task;
        }
    });

    public Optional<StrategyCacheDto> getCacheStrategyBySlotId(Long slotId) {
        return CACHE.getUnchecked(slotId);
    }

    public void refreshMediaSelectAdvertCache(List<Long> slotIds) {
        if (CollectionUtils.isEmpty(slotIds)) {
            return;
        }
        slotIds.forEach(slotId -> {
            MEDIA_SELECT_ADVERT_CACHE.refresh(slotId);
            CACHE.refresh(slotId);

            // 防止大量广告位刷新，增加10ms的sleep
            try {
                TimeUnit.MILLISECONDS.sleep(50);
            } catch (Exception e) {
                log.error("refreshMediaSelectAdvertCache InterruptedException", e);
            }
        });
    }

    public void clearCache() {
        CACHE.invalidateAll();
        MEDIA_SELECT_ADVERT_CACHE.invalidateAll();
        ELECT_APP_CACHE.invalidateAll();
    }

    public void  updateSlotStrategy(Long slotId){
        CACHE.refresh(slotId);
    }

    /**
     * 查询广告位选择的屏蔽策略信息.<br>
     * [使用说明]
     * <ol>
     * <li>当该广告位没有选择屏蔽策略时，返回null</li>
     * <li>其他情况，返回屏蔽策略信息</li>
     * </ol>
     *
     * @param slotId the slot id
     * @return the strategy by slot id
     */
    private StrategyCacheDto getStrategyBySlotId(Long slotId) {
        try{
            if(null != slotId){
                StrategyCacheDto cacheDto = remoteManagerShieldStrategyService.queryStrategyBySlotId(slotId);//redisTemplate.opsForHash().get("SSP_CENTER_K19_", slotId);
                    if(null != cacheDto){
                        cacheDto.setSlotId(slotId);
                    }else {
                        // 返回为空的情况，id给-1L ： 为了区分接口正常返回null和出异常返回null
                        cacheDto = new StrategyCacheDto();
                        cacheDto.setId(-1L);
                    }
                    return cacheDto;
            }
        }catch(Exception e){
            log.error("getStrategyBySlotId error slotId:{}",slotId, e);
        }
        return null;
    }
}
