package cn.com.duiba.tuia.bo.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Maps;

import cn.com.duiba.tuia.bo.AdvertBannedTagBo;
import cn.com.duiba.tuia.constants.ErrorCode;
import cn.com.duiba.tuia.dao.advert_tag.AdvertTagDAO;
import cn.com.duiba.tuia.domain.dataobject.AdvertOrientationPackageDO;
import cn.com.duiba.tuia.domain.dataobject.AdvertTagDO;
import cn.com.duiba.tuia.exception.TuiaException;
import cn.com.duiba.tuia.service.AdvertOrientationService;

/**
 * @author ZhouFeng zhoufeng@duiba.com.cn
 * @version $Id: AdvertBannedTagBoImpl.java , v 0.1 2017/4/13 下午1:45 ZhouFeng Exp $
 */
@Service
public class AdvertBannedTagBoImpl implements AdvertBannedTagBo {

    @Autowired
    private AdvertOrientationService advertOrientationService;

    @Autowired
    private AdvertTagDAO advertTagDAO;

    /**
     * 广告标签缓存
     */
    private final Cache<String, List<String>> CACHE = CacheBuilder.newBuilder().initialCapacity(1000).
            recordStats().expireAfterWrite(1, TimeUnit.HOURS).build();


    @Override
    public List<String> selectBannedTagByAdvertIdAndOrientataionPackageId(final Long advertId, final Long
            orientationPackageId)
            throws TuiaException {
        try {
            return CACHE.get(formateCacheKey(advertId, orientationPackageId), new Callable<List<String>>() {
                @Override
                public List<String> call() throws Exception {
                    AdvertTagDO advertTagDO = advertTagDAO.selectByAdvertId(advertId);
                    if (advertTagDO == null) {
                        return Collections.emptyList();
                    }
                    return splitBannedString(advertTagDO.getAdvertBannedTagNums());
                }
            });
        } catch (ExecutionException e) {
            throw new TuiaException(ErrorCode.E9999999, e);
        }
    }

    @Override
    public void deleteBannedCacheByAdvertIdAndOrientataionPackageId(Long advertId, Long orientationPackageId) {
        //清除缓存
        CACHE.invalidate(formateCacheKey(advertId, orientationPackageId));
    }

    @Override
    public void initBannedTags(List<Long> advertIds, List<AdvertTagDO> advertTagDOS) throws TuiaException {

        Map<Long, String> advertBannedTagNumsMap = Maps.newHashMap();
        advertTagDOS.forEach(advertTagDO -> advertBannedTagNumsMap.put(advertTagDO.getAdvertId(), advertTagDO.getAdvertBannedTagNums()));

        for (AdvertTagDO advertTagDO : advertTagDOS) {
            CACHE.put(formateCacheKey(advertTagDO.getAdvertId(), 0L), splitBannedString(advertTagDO.getAdvertBannedTagNums()));
        }

        //查询定向配置包下的标签
        List<AdvertOrientationPackageDO> orientationList = advertOrientationService.getOrientationList(advertIds);
        if(CollectionUtils.isEmpty(orientationList)){
            return;
        }
        for (AdvertOrientationPackageDO packageDO : orientationList) {
            CACHE.put(formateCacheKey(packageDO.getAdvertId(), packageDO.getId()), splitBannedString(advertBannedTagNumsMap.get(packageDO.getAdvertId())));
        }


    }

    /**
     * 格式化缓存key
     *
     * @param advertId
     * @param orientationPackageId
     * @return
     */
    private String formateCacheKey(Long advertId, Long orientationPackageId) {
        return new StringBuilder().append(advertId).append("-").append(orientationPackageId).toString();
    }

    private List<String> splitBannedString(String bannedStr) {
        if (StringUtils.isBlank(bannedStr)) {
            return Collections.emptyList();
        }
        String[] tags = bannedStr.split(",");
        List<String> list = new ArrayList<>(tags.length);
        list.addAll(Arrays.asList(tags));

        return list;
    }
}
