package cn.com.duiba.tuia.cache;

import cn.com.duiba.tuia.dao.material.AdvertMaterialActDAO;
import cn.com.duiba.tuia.dao.tag.NewTradeAdvertDAO;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * Created by ZhangShun at 2020/5/19
 */
@Service
public class ActPreCache {

    @Resource
    private AdvertMaterialActDAO advertMaterialActDAO;
    @Resource
    private NewTradeAdvertDAO newTradeAdvertDAO;
    @Resource
    private ExecutorService executorService;

    /**
     * 新行业标签缓存
     *
     * 新行业标签名称  >  广告列表
     * String        >  List<String>
     * 视频          >  [123,3234]
     */
    private LoadingCache<String, List<Long>> ACT_PRE_NEW_TRADE_CACHE = CacheBuilder.newBuilder()
            .initialCapacity(30).maximumSize(50)
            .expireAfterAccess(30, TimeUnit.MINUTES)
            .refreshAfterWrite(15, TimeUnit.MINUTES)
            .build(new CacheLoader<String, List<Long>>() {
                public List<Long> load(String newTradeName) {
                    return newTradeAdvertDAO.selectNewTradeAdvertList(newTradeName);
                }
                public ListenableFutureTask<List<Long>> reload(final String newTradeName, List<Long> oldValue) {
                    ListenableFutureTask<List<Long>> task = ListenableFutureTask.create(() -> load(newTradeName));
                    executorService.submit(task);
                    return task;
                }
            });

    /**
     * 广告预发券素材缓存
     *
     * 广告ID  >  素材图片URL
     * Long   >  String
     * 123     > "//yun.dui88.com/tuia/img/gtwtojh7n2.png"
     */
    private LoadingCache<Long, String> ACT_PRE_ADVERT_MATERIAL_CACHE = CacheBuilder.newBuilder()
            .initialCapacity(1000).maximumSize(3000)
            .expireAfterAccess(3, TimeUnit.HOURS)
            .refreshAfterWrite(15, TimeUnit.MINUTES)
            .build(new CacheLoader<Long, String>() {
                public String load(Long advertId) {
                    return StringUtils.defaultString(advertMaterialActDAO.selectImageUrlByAdvertId(advertId));
                }
                public ListenableFutureTask<String> reload(final Long advertId, String oldValue) {
                    ListenableFutureTask<String> task = ListenableFutureTask.create(() -> load(advertId));
                    executorService.submit(task);
                    return task;
                }
            });

    /**
     * 刷新缓存
     * @param advertId
     */
    public void refresh(Long advertId, String newTradeName) {
        if (Objects.nonNull(advertId)) {
            ACT_PRE_ADVERT_MATERIAL_CACHE.refresh(advertId);
        }
        if (StringUtils.isNotBlank(newTradeName)) {
            ACT_PRE_NEW_TRADE_CACHE.refresh(newTradeName);
        }
    }

    /**
     * 缓存查询 新行业标签 对应的广告列表
     */
    public List<Long> getAdvertList(String newTradeName) {
        try {
            List<Long> list = ACT_PRE_NEW_TRADE_CACHE.getUnchecked(newTradeName);
            return (null == list) ? Collections.emptyList() : list;
        } catch (Exception e) {
            return Collections.emptyList();
        }
    }

    /**
     * 缓存查询 新行业标签 对应的广告列表
     */
    public String getActPreImageUrl(Long advertId) {
        try {
            return StringUtils.defaultString(ACT_PRE_ADVERT_MATERIAL_CACHE.getUnchecked(advertId), StringUtils.EMPTY);
        } catch (Exception e) {
            return StringUtils.EMPTY;
        }
    }

}
