package cn.com.duiba.tuia.cache;

import cn.com.duiba.tuia.activity.center.api.dto.staticpage.VerifyAllResultQuery;
import cn.com.duiba.tuia.activity.center.api.dto.staticpage.VerifyPageResultDto;
import cn.com.duiba.tuia.activity.center.api.remoteservice.staticpage.RemoteStaticPageVerifyService;
import cn.com.duiba.tuia.dao.promotetest.AdvertNewPromoteTestDao;
import cn.com.duiba.tuia.dao.promotetest.AdvertPromoteTestDAO;
import cn.com.duiba.tuia.domain.dataobject.*;
import cn.com.duiba.tuia.exception.TuiaException;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@Service
@RefreshScope
public class AdvertABTestCacheService extends BaseCacheService {

    @Resource
    private ExecutorService executorService;
    @Autowired
    private AdvertPromoteTestDAO advertPromoteTestDAO;

    @Autowired
    private AdvertNewPromoteTestDao advertNewPromoteTestDao;
    @Autowired
    private RemoteStaticPageVerifyService remoteStaticPageVerifyService;

    //abtest缓存 有消息同步，不用设置过期时间来保证避免读取旧值
    private LoadingCache<Long, ABTestFilterUrlDO> advertABTestFilterList = CacheBuilder.newBuilder().initialCapacity(1000).maximumSize(1200)
            .refreshAfterWrite(5, TimeUnit.MINUTES)
            .build(
                    new CacheLoader<Long, ABTestFilterUrlDO>() {
                        public ABTestFilterUrlDO load(Long advertId) {
                            return getAdvertABTestByCache(advertId);
                        }

                        public ListenableFutureTask<ABTestFilterUrlDO> reload(final Long key, ABTestFilterUrlDO prevGraph) {
                            ListenableFutureTask<ABTestFilterUrlDO> task = ListenableFutureTask.create(() -> {
                                return getAdvertABTestByCache(key);
                            });
                            executorService.submit(task);
                            return task;
                        }
                    });

    //缓存 有消息同步，不用设置过期时间来保证避免读取旧值
    private LoadingCache<Long, Map<String, Boolean>> landPageRepoFilterMap = CacheBuilder.newBuilder()
            .initialCapacity(1000)
            .maximumSize(1200)
            .refreshAfterWrite(5, TimeUnit.MINUTES)
            .build(
                    new CacheLoader<Long, Map<String, Boolean>>() {
                        public Map<String, Boolean> load(Long advertId) {
                            return getLandPageRepoInfoByCache(advertId);
                        }

                        public ListenableFutureTask<Map<String, Boolean>> reload(final Long key, Map<String, Boolean> prevGraph) {
                            ListenableFutureTask<Map<String, Boolean>> task = ListenableFutureTask.create(() -> {
                                return getLandPageRepoInfoByCache(key);
                            });
                            executorService.submit(task);
                            return task;
                        }
                    });

    /**
     * 初始化缓存
     * @param advertList
     */
    public void init(List<Long> advertList) {
        for (Long advertId : advertList) {
            try {
                ABTestFilterUrlDO abTestFilterUrlDO = getAdvertABTestInfo(advertId);
                advertABTestFilterList.put(advertId, abTestFilterUrlDO);
            } catch (Exception e) {
                logger.error("初始化缓存异常", e);
            }

            try {
                Map<String, Boolean> map = getLandPageRepoInfo(advertId);
                landPageRepoFilterMap.put(advertId, map);
            } catch (Exception e) {
                logger.error("初始化缓存异常:getLandPageRepoInfo", e);
            }
        }
    }

    /**
     * 更新缓存
     * @param advertId
     */
    public void updateAdvertABTestCache(Long advertId) {
        try {
            advertABTestFilterList.refresh(advertId);
        } catch (Exception e) {
            logger.error("updateAdvertABTestCache error", e);
        }

        try {
            landPageRepoFilterMap.refresh(advertId);
        } catch (Exception e) {
            logger.error("updateAdvertABTestCache.landPageRepoFilterMap error", e);
        }
    }

    /**
     *
     * 根据广告ID获取广告下配置了ABTest的链接是否都支持爱奇艺
     * @param advertId
     * @return
     */
    public ABTestFilterUrlDO getAdvertABTestCache(Long advertId) {
        if (advertId == null) {
            return null;
        }
        try {
            return advertABTestFilterList.get(advertId);
        } catch (Exception e) {
            return null;
        }
    }

    public ABTestFilterUrlDO getAdvertABTestByCache(Long advertId) {
        try {
            if (null == advertId) {
                return null;
            }
            return getAdvertABTestInfo(advertId);
        } catch (Exception e) {
            logger.error(e.getMessage()+advertId.toString(), e);
            return null;
        }
    }

    public Map<String, Boolean> getLandPageRepoInfoCache(Long advertId) {
        if (advertId == null) {
            return null;
        }
        try {
            return landPageRepoFilterMap.get(advertId);
        } catch (Exception e) {
            return null;
        }
    }

    public Map<String, Boolean> getLandPageRepoInfoByCache(Long advertId) {
        if (null == advertId) {
            return Collections.emptyMap();
        }

        try {
            return getLandPageRepoInfo(advertId);
        } catch (Exception e) {
            logger.error(e.getMessage() + advertId.toString(), e);
        }
        return Collections.emptyMap();
    }

    public ABTestFilterUrlDO getAdvertABTestInfo(Long advertId) throws TuiaException {
        //1.查询广告有效的，并配置了ABTest链接的列表 findPromoteUrlsByAdvertId方法 是从tb_advert_promote_test表查落地页测试信息.这是老逻辑
        List<AdvertPromoteTestDO> promoteTestDOList = advertPromoteTestDAO.findPromoteUrlsByAdvertId(advertId);

        // 如果老逻辑没有才从新逻辑里查
        if(CollectionUtils.isEmpty(promoteTestDOList)){
            //新落地 落地页测试计划单独拆处了表
            promoteTestDOList= advertNewPromoteTestDao.findNewPromoteUrlsByAdvertId(advertId);
        }


        //如果为空，没有生效或可用的测试链接
        if (CollectionUtils.isEmpty(promoteTestDOList)) {
            return new ABTestFilterUrlDO(false);
        }

        List<String> urlList = promoteTestDOList.stream().map(AdvertPromoteTestDO::getPromoteUrl).collect(Collectors.toList());
        //2.调全量获取爱奇艺映射关系
        List<String> urlMappingList = getUrlMappingList(urlList);
        if (CollectionUtils.isEmpty(urlMappingList)) {
            return new ABTestFilterUrlDO(false);
        }
        //3.以1的结果为主，2的结果为辅 比对构建对应的缓存
        for (AdvertPromoteTestDO promoteTestDO : promoteTestDOList) {
            if (!isBaiqiDomain(promoteTestDO.getPromoteUrl())) {
                return new ABTestFilterUrlDO(false);
            }

            if (!urlMappingList.contains(promoteTestDO.getPromoteUrl())) {
                return new ABTestFilterUrlDO(false);
            }
        }

        return new ABTestFilterUrlDO(true);
    }

    /**
     * 判断该广告的落地页库链接是否爱奇艺审核通过
     *
     * @param advertId 广告Id
     * @return 落地页链接-是否爱奇艺审核通过映射
     */
    public Map<String, Boolean> getLandPageRepoInfo(Long advertId) {
        //1.查询广告有效的，并配置了ABTest链接的列表
        List<String> urlList = advertPromoteTestDAO.selectLandPageRepoUrlsByAdvertId(advertId);

        //如果为空，没有生效或可用的测试链接
        if (CollectionUtils.isEmpty(urlList)) {
            return Collections.emptyMap();
        }

        //2.调全量获取爱奇艺映射关系
        List<String> urlMappingList = getUrlMappingList(urlList);
        if (CollectionUtils.isEmpty(urlMappingList)) {
            return Collections.emptyMap();
        }
        //3.以1的结果为主，2的结果为辅 比对构建对应的缓存
        Map<String, Boolean> map = new HashMap<>(urlList.size());
        for (String url : urlList) {
            map.put(url, isBaiqiDomain(url) && urlMappingList.contains(url));
        }
        return map;
    }

    public static boolean isBaiqiDomain(String promoteUrl) {
        if (StringUtils.isBlank(promoteUrl)) {
            return false;
        }

        String baiqiDomainStr = "/item/detail3";
        String baiqiDomainStr2 = "id=";
        //是否同时包含 /item/detail和id=
        return promoteUrl.contains(baiqiDomainStr) && promoteUrl.contains(baiqiDomainStr2);
    }

    private List<String> getUrlMappingList(List<String> urlList) {
        try {
            VerifyAllResultQuery verifyAllResultQuery = new VerifyAllResultQuery();
            verifyAllResultQuery.setPlatformType(1);
            verifyAllResultQuery.setPageType(2);
            verifyAllResultQuery.setTuiaUrls(urlList);
            List<VerifyPageResultDto> list = remoteStaticPageVerifyService.selectAllVerifyResult(verifyAllResultQuery);
            if (CollectionUtils.isEmpty(list)) {
                return new ArrayList<>();
            }

            List<String> resultList = list.stream().filter(item -> item.getLastStatus().equals(2)).map(VerifyPageResultDto::getTuiaUrl).collect(Collectors.toList());
            return resultList;
        } catch (Exception e) {
            return new ArrayList<>();
        }
    }
}
