package cn.com.duiba.tuia.cache;

import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import com.alibaba.fastjson.JSON;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFutureTask;

import cn.com.duiba.tuia.dao.compensate.CompensateAdvertDAO;
import cn.com.duiba.tuia.dao.compensate.CompensateConsumeDAO;
import cn.com.duiba.tuia.domain.dataobject.CompensateAdvertDO;
import cn.com.duiba.tuia.domain.dataobject.CompensateConsumeDO;
import cn.com.duiba.tuia.message.rocketmq.RefreshCacheMqProducer;
import cn.com.duiba.wolf.redis.RedisAtomicClient;
import cn.com.duiba.wolf.redis.RedisLock;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.tuia.advert.cache.RedisCommonKeys;

/**
 * 广告计划赔付缓存
 *
 * @author peanut.huang
 * @date 2019/10/17
 * @since JDK 1.8
 */
@Service
public class AdvertCompensateCacheService extends BaseCacheService {

    @Resource
    private ExecutorService                 executorService;
    @Resource
    private CompensateAdvertDAO             compensateAdvertDAO;
    @Resource
    private CompensateConsumeDAO            compensateConsumeDAO;
    @Resource
    private StringRedisTemplate             stringRedisTemplate;
    @Resource(name="redisTemplate")
    private RedisAtomicClient               redisAtomicClient;
    @Resource
    private RefreshCacheMqProducer          refreshCacheMqProducer;


    private static final String             COMPENSATE_CONSUME_KEY_SPILT = ":";


    private static final String             COMPENSATE_SEND_MSG_KEY= "compensate_send_msg";



    /**
     * 赔付消耗缓存
     */
    private LoadingCache<String, Optional<CompensateConsumeDO>> compensateConsumeCache = CacheBuilder.newBuilder()
                                                                                                    .initialCapacity(1000).maximumSize(1200)
                                                                                                    .refreshAfterWrite(30, TimeUnit.MINUTES)
                                                                                                    .expireAfterWrite(2, TimeUnit.HOURS)
                                                                                                    .build(initConsumeCacheLoader());

    private CacheLoader<String, Optional<CompensateConsumeDO>> initConsumeCacheLoader(){
        return  new CacheLoader<String,  Optional<CompensateConsumeDO>>() {
            @Override
            public Optional<CompensateConsumeDO> load(@NotNull String key) throws Exception{
                String [] keyArray = key.split(COMPENSATE_CONSUME_KEY_SPILT);

                // 计划id
                Long advertId = Long.valueOf(keyArray[0]);

                // 赔付日期
                String compensateDate = keyArray[1];

                return Optional.ofNullable(compensateConsumeDAO.selectLastConsume(advertId, compensateDate));
            }

            @Override
            public ListenableFutureTask<Optional<CompensateConsumeDO>> reload(final String key, Optional<CompensateConsumeDO> compensateAdvertDO) throws Exception {
                ListenableFutureTask<Optional<CompensateConsumeDO>> task = ListenableFutureTask.create(() -> load(key));
                executorService.submit(task);
                return task;
            }
        };
    }

    public void refreshCompensateConsume(String key){
        compensateConsumeCache.refresh(key);
    }


    public Optional<CompensateConsumeDO> getCompensateConsume(Long advertId, String compensateDate){
        if(advertId == null || StringUtils.isBlank(compensateDate)){
            return Optional.empty();
        }
            
        // key
        String key = advertId + COMPENSATE_CONSUME_KEY_SPILT + compensateDate;

        try {
          return compensateConsumeCache.get(key);
        } catch (Exception e) {
            logger.error("getCompensateConsume error, advertId={}, compensateDate={}", advertId, compensateDate);
            return Optional.empty();
        }
    }

    /**
     * 赔付计划缓存 有消息同步，不用设置过期时间来保证避免读取旧值
     */
    private LoadingCache<Long, Optional<CompensateAdvertDO>> compensateAdvertListCache = CacheBuilder.newBuilder()
                                                                                .initialCapacity(1000).maximumSize(1200)
                                                                                .refreshAfterWrite(5, TimeUnit.MINUTES)
                                                                                .build(initAdvertCacheLoader());


    private CacheLoader<Long, Optional<CompensateAdvertDO>> initAdvertCacheLoader(){
        return  new CacheLoader<Long, Optional<CompensateAdvertDO>>() {
            @Override
            public Optional<CompensateAdvertDO> load(@NotNull Long advertId) throws Exception{
                return Optional.ofNullable(compensateAdvertDAO.selectValidByAdvertId(advertId));
            }

            @Override
            public ListenableFutureTask<Optional<CompensateAdvertDO>> reload(final Long key, Optional<CompensateAdvertDO> compensateAdvertDO) throws Exception {
                ListenableFutureTask<Optional<CompensateAdvertDO>> task = ListenableFutureTask.create(() -> load(key));
                executorService.submit(task);
                return task;
            }
        };
    }


    /**
     * 根据计划获取赔付记录
     *
     * @param advertId 计划id
     * @return
     */
    public Optional<CompensateAdvertDO> getCompensateAdvert(Long advertId){
        try {
            return compensateAdvertListCache.get(advertId);
        } catch (Exception e) {
            logger.error("getCompensateAdvert error advertId={}", advertId, e);
            return Optional.empty();
        }
    }

    /**
     * 广告计划是否需要赔付
     *
     * @param advertId 计划id
     * @return
     */
    public boolean isNeedCompensate(Long advertId){
        Optional<CompensateAdvertDO> optional = getCompensateAdvert(advertId);
        return optional.isPresent();
    }

    /**
     * 处理消耗金额：计算当前计划的赔付是否达标
     *
     * @param fee              当前扣费金额
     * @param advertId         计划id
     * @param compensateDate   赔付日期
     */
    public void handleConsume(Long fee, Long advertId, String compensateDate) {
        try {
            // 无赔付日期，不需要赔付
            if(StringUtils.isBlank(compensateDate)){
                return;
            }

            Optional<CompensateAdvertDO> compensateOptional = this.getCompensateAdvert(advertId);
            if(!compensateOptional.isPresent()){
                return;
            }

            CompensateAdvertDO compensateAdvertDO = compensateOptional.get();


            // 计划要赔付的金额
            Long compensateAmount = compensateAdvertDO.getCompensateAmount();

            // 累计赔付金额 ： 当前扣费 + 历史扣费 + 今日扣费
            Long totalConsume = fetchTotalConsume(fee, advertId, compensateDate);

            // 赔付完成：累计赔付金额 >= 要赔付的金额
            if(totalConsume >= compensateAmount){

                // 注意1秒钟频次控制
                try(RedisLock lock = redisAtomicClient.getLock(COMPENSATE_SEND_MSG_KEY, 1)) {
                    if(lock != null){

                        // 发消息变更赔付计划状态为关闭, msg = advertId:compensateDate:0
                        String msg = advertId + COMPENSATE_CONSUME_KEY_SPILT + compensateDate + COMPENSATE_CONSUME_KEY_SPILT + 0;
                        logger.info("send compensate status to end. msg={}", msg);
                        refreshCacheMqProducer.sendMsg(RedisCommonKeys.KC143.toString(), msg);
                    }

                } catch (Exception e) {
                    logger.error("send compensate COMPENSATE_SEND_MSG_KEY error", e);
                }
            }
        }catch (Exception e){
            logger.error("handleConsume error, fee = {}. advertId={}, compensateDate={}", fee, advertId, compensateDate, e);
        }

    }


    /**
     * 累计赔付金额 ： 当前扣费 + 历史扣费 + 今日扣费
     *
     * @param fee              当前扣费金额
     * @param advertId         计划id
     * @param compensateDate   赔付日期
     * @return
     */
    private Long fetchTotalConsume(Long fee, Long advertId, String compensateDate) {

        // 1.最近回流的扣费 -- 从数据表中获取
        Optional<CompensateConsumeDO> compensateConsumeOptional = getCompensateConsume(advertId, compensateDate);
        if(compensateConsumeOptional.isPresent()){
            CompensateConsumeDO compensateConsumeDO = compensateConsumeOptional.get();

            // compensateConsumeDO.getCurDate() 为昨天时，实时取今天
            boolean isYesterday = isYesterday(compensateConsumeDO.getCurDate());
            if(isYesterday){
                return compensateConsumeDO.getTotalConsume() + getTodayConsume(fee, advertId, compensateDate);
            }

            // compensateConsumeDO.getCurDate() 不为昨天, 实时取昨天 + 今天
            return compensateConsumeDO.getTotalConsume() + getTodayConsume(fee, advertId, compensateDate) + getYesterdayConsume(advertId, compensateDate);

        }

        // 以往扣费为0, 今天与昨天的扣费 -- 从redis中获取, 当前扣费 fee进行累加
        return getConsumeFromRedis(fee, advertId, compensateDate);
    }


    private boolean isYesterday(Date curDate) {
        return DateUtils.getDayStr(curDate).equals(DateUtils.getYesterday());
    }

    /**
     * 从redis中获取今日与昨日的消耗
     *
     * @param fee               当前扣费
     * @param advertId          计划id
     * @param compensateDate    赔付日期
     * @return
     */
    private Long getConsumeFromRedis(Long fee, Long advertId, String compensateDate) {

        // 获取今日消耗
        Long todayConsume = getTodayConsume(fee, advertId, compensateDate);

        // 获取昨日消耗
        Long yesterdayConsume = getYesterdayConsume(advertId, compensateDate);

        return todayConsume + yesterdayConsume;
    }

    /**
     * 获取赔付计划昨日消耗
     *
     * @param advertId          计划id
     * @param compensateDate    赔付日期
     * @return
     */
    private long getYesterdayConsume(Long advertId, String compensateDate) {

        String yesterdayDate = DateUtils.getYesterday();

        // 昨日消耗redis key
        String yesterdayKey = advertId + COMPENSATE_CONSUME_KEY_SPILT + compensateDate + COMPENSATE_CONSUME_KEY_SPILT + yesterdayDate;

        String consume = stringRedisTemplate.boundValueOps(yesterdayKey).get();
        return consume == null ? 0L : Long.valueOf(consume);
    }

    /**
     * 获取今日消耗
     *
     * @param fee             当前扣费
     * @param advertId        计划id
     * @param compensateDate  赔付日期
     * @return
     */
    private long getTodayConsume(Long fee, Long advertId, String compensateDate) {
        String todayDate = DateUtils.getDayStr(DateTime.now().toDate());

        // 今天消耗redis key= advertId:compensateDate:todayDate
        String todayKey = advertId + COMPENSATE_CONSUME_KEY_SPILT + compensateDate + COMPENSATE_CONSUME_KEY_SPILT + todayDate;

        // 如果今日redis key不存在，初始set并设置过期时间: 这里会不会有并发问题？？
        if(!stringRedisTemplate.hasKey(todayKey)){
            stringRedisTemplate.boundValueOps(todayKey).set(String.valueOf(fee), 2, TimeUnit.DAYS);
            return fee;
        }else {
            // 非第一次，当前扣费累加
            return stringRedisTemplate.boundValueOps(todayKey).increment(fee);
        }
    }

    /**
     * refresh
     *
     * @param message
     */
    public void refreshCompensateList(String message) {

        String [] msgArray = message.split(":");

        Long advertId = Long.valueOf(msgArray[0]);
        Integer type = Integer.valueOf(msgArray[1]);

        // 关闭
        if(0 == type){
            compensateAdvertListCache.invalidate(advertId);
        }

        // 开启
        if(1 == type){
            compensateAdvertListCache.refresh(advertId);
        }
    }
}
