package cn.com.duiba.tuia.cache;

import cn.com.duiba.tuia.dao.advert.AdvertCouponBackDAO;
import cn.com.duiba.tuia.dao.advert.AdvertKeywordMatchDAO;
import cn.com.duiba.tuia.dao.engine.AdvertOrientationPackageDAO;
import cn.com.duiba.tuia.dao.promotetest.AdvertPromoteTestDAO;
import cn.com.duiba.tuia.domain.dataobject.*;
import cn.com.duiba.tuia.enums.PatrolTypeEnum;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@Service
@RefreshScope
public class AdvertKeywordCacheService extends BaseCacheService {

    @Resource
    private ExecutorService executorService;
    @Autowired
    private AdvertKeywordMatchDAO advertKeywordMatchDAO;
    @Autowired
    private AdvertCouponBackDAO advertCouponBackDAO;
    @Autowired
    private AdvertPromoteTestDAO advertPromoteTestDAO;
    @Autowired
    private AdvertMapCacheManager advertMapCacheManager;
    @Autowired
    private AdvertOrientationPackageDAO advertOrientationPackageDAO;

    /**
     * 匹配结果缓存
     */
    private LoadingCache<Long, AdvertFilterKeywordDO> Slot_Match_Result_List = CacheBuilder.newBuilder().initialCapacity(1000)
            .refreshAfterWrite(5, TimeUnit.MINUTES).expireAfterWrite(3, TimeUnit.HOURS)
            .build(
                    new CacheLoader<Long, AdvertFilterKeywordDO>() {

                        public AdvertFilterKeywordDO load(Long slotId) {
                            return getFilterKeywordInfo(slotId).orElse(null);
                        }

                        public ListenableFutureTask<AdvertFilterKeywordDO> reload(final Long slotId, AdvertFilterKeywordDO oldVal) {
                            ListenableFutureTask<AdvertFilterKeywordDO> task = ListenableFutureTask.create(() -> getFilterKeywordInfo(slotId).orElse(oldVal));
                            executorService.submit(task);
                            return task;
                        }
                    });

    /**
     * 获取缓存
     * @param soltId
     * @return
     */
    public AdvertFilterKeywordDO getShieldKeywordListBySlotId(Long soltId) {
        try {
            return Slot_Match_Result_List.get(soltId);
        } catch (Exception e) {
            logger.error("获取缓存异常", e);
            return null;
        }
    }

    /**
     * 初始化匹配结果集
     * @return
     */
    public List<AdvertKeywordMatchDO> loadMatchResultListBySlotId(Long slotId) {
        try {
            return advertKeywordMatchDAO.getMatchResultBySlotId(slotId);
        } catch (Exception e) {
            logger.error("初始化关键词匹配结果集合异常", e);
        }

        return new ArrayList<>();
    }

    /**
     * 根据广告位ID获取过滤对象信息
     * @param slotId
     * @return
     */
    public Optional<AdvertFilterKeywordDO> getFilterKeywordInfo(Long slotId) {
        AdvertFilterKeywordDO advertFilterKeywordDO = new AdvertFilterKeywordDO();
        advertFilterKeywordDO.setGeneralFilterKeywordDO(new GeneralFilterKeywordDO());
        advertFilterKeywordDO.setNewTradeFilterKeywordDO(new NewTradeFilterKeywordDO());
        try {
            List<AdvertKeywordMatchDO> list = loadMatchResultListBySlotId(slotId);
            if (CollectionUtils.isNotEmpty(list)) {
                //构建广告屏蔽列表，缩略图
                buildAdvertIdShieldList(advertFilterKeywordDO, list);
                //构建普通素材
                buildMaterialList(advertFilterKeywordDO, list);
                //构建落地页屏蔽列表
                buildLandingPageList(advertFilterKeywordDO, list);
            }
        } catch (Exception e) {
            logger.error("异常", e);
        }
        return Optional.ofNullable(advertFilterKeywordDO);
    }

    /**
     * 构建广告维度屏蔽列表
     * @param advertFilterKeywordDO
     * @param list
     */
    private void buildAdvertIdShieldList(AdvertFilterKeywordDO advertFilterKeywordDO, List<AdvertKeywordMatchDO> list) {
        Set<Long> advertIdList = list.stream().filter(item -> item.getShieldType().equals(0L) && item.getPatrolType().equals(PatrolTypeEnum.Thumbnail.getCode())).map(AdvertKeywordMatchDO::getAdvertId).collect(Collectors.toSet());
        advertFilterKeywordDO.getGeneralFilterKeywordDO().setAdvertIdList(advertIdList);

        Set<Long> newTradeAdvertIds = list.stream().filter(item -> item.getShieldType().equals(1L) && item.getPatrolType().equals(PatrolTypeEnum.Thumbnail.getCode())).map(AdvertKeywordMatchDO::getAdvertId).collect(Collectors.toSet());
        advertFilterKeywordDO.getNewTradeFilterKeywordDO().setAdvertIdList(newTradeAdvertIds);

        buildAdvertShieldListByLandPage(advertFilterKeywordDO, list);
    }

    /**
     * 当广告所有落地页都被屏蔽时上升到广告维度屏蔽
     * @param advertFilterKeywordDO
     * @param list
     */
    private void buildAdvertShieldListByLandPage(AdvertFilterKeywordDO advertFilterKeywordDO, List<AdvertKeywordMatchDO> list) {
        try {
            Set<Long> advertIdList = list.stream().filter(item -> item.getShieldType().equals(0L) && isLandPage(item)).map(AdvertKeywordMatchDO::getAdvertId).collect(Collectors.toSet());
            //主链接
            List<AdvertCouponBackDO> couponBackDOList = advertCouponBackDAO.getAdvertCouponBackList(new ArrayList<>(advertIdList));
            //ABTest 新媒体 配置链接
            List<AdvertPromoteTestDO> promoteTestDOList = advertPromoteTestDAO.selectPromoteUrlList(new ArrayList<>(advertIdList));
            Map<Long, Set<Long>> configMap = new HashMap<>();
            Map<Long, Set<Long>> shieldMap = new HashMap<>();
            initConfigMap(configMap, couponBackDOList, promoteTestDOList);
            initShieldMap(shieldMap, list);

            Set<Long> landPageShieldList = new HashSet<>();
            for (Map.Entry<Long, Set<Long>> entry : configMap.entrySet()) {
                if (shieldMap.containsKey(entry.getKey())) {
                    entry.getValue().removeAll(shieldMap.get(entry.getKey()));
                    if (entry.getValue().size() == 0) {
                        landPageShieldList.add(entry.getKey());
                    }
                }
            }
            if (CollectionUtils.isNotEmpty(landPageShieldList)) {
                advertFilterKeywordDO.getGeneralFilterKeywordDO().getAdvertIdList().addAll(landPageShieldList);
            }

            buildNewTradeAdvertShieldListByLandPage(advertFilterKeywordDO, list);
        } catch (Exception e) {
            logger.error("落地页屏蔽上升广告维度异常", e);
        }
    }

    private void buildNewTradeAdvertShieldListByLandPage(AdvertFilterKeywordDO advertFilterKeywordDO, List<AdvertKeywordMatchDO> list) {
        try {
            Set<Long> advertIdList = list.stream().filter(item -> item.getShieldType().equals(1L) && isLandPage(item)).map(AdvertKeywordMatchDO::getAdvertId).collect(Collectors.toSet());
            //主链接
            List<AdvertCouponBackDO> couponBackDOList = advertCouponBackDAO.getAdvertCouponBackList(new ArrayList<>(advertIdList));
            //ABTest 新媒体 配置链接
            List<AdvertPromoteTestDO> promoteTestDOList = advertPromoteTestDAO.selectPromoteUrlList(new ArrayList<>(advertIdList));
            Map<Long, Set<Long>> configMap = new HashMap<>();
            Map<Long, Set<Long>> shieldMap = new HashMap<>();
            initConfigMap(configMap, couponBackDOList, promoteTestDOList);
            initNewTradeShieldMap(shieldMap, list);

            Set<Long> landPageShieldList = new HashSet<>();
            for (Map.Entry<Long, Set<Long>> entry : configMap.entrySet()) {
                if (shieldMap.containsKey(entry.getKey())) {
                    entry.getValue().removeAll(shieldMap.get(entry.getKey()));
                    if (entry.getValue().size() == 0) {
                        landPageShieldList.add(entry.getKey());
                    }
                }
            }
            if (CollectionUtils.isNotEmpty(landPageShieldList)) {
                advertFilterKeywordDO.getNewTradeFilterKeywordDO().getAdvertIdList().addAll(landPageShieldList);
            }
        } catch (Exception e) {
            logger.error("落地页屏蔽上升广告维度异常", e);
        }
    }

    private void initShieldMap(Map<Long, Set<Long>> shieldMap, List<AdvertKeywordMatchDO> list) {
        List<AdvertKeywordMatchDO> advertIdList = list.stream().filter(item -> item.getShieldType().equals(0L) && isLandPage(item)).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(advertIdList)) {
            advertIdList.forEach(item -> {
                if (shieldMap.containsKey(item.getAdvertId())) {
                    shieldMap.get(item.getAdvertId()).add(item.getPatrolType());
                } else {
                    Set<Long> config = new HashSet<>();
                    config.add(item.getPatrolType());
                    shieldMap.put(item.getAdvertId(), config);
                }
            });
        }
    }

    private void initNewTradeShieldMap(Map<Long, Set<Long>> shieldMap, List<AdvertKeywordMatchDO> list) {
        List<AdvertKeywordMatchDO> advertIdList = list.stream().filter(item -> item.getShieldType().equals(1L) && isLandPage(item)).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(advertIdList)) {
            advertIdList.forEach(item -> {
                if (shieldMap.containsKey(item.getAdvertId())) {
                    shieldMap.get(item.getAdvertId()).add(item.getPatrolType());
                } else {
                    Set<Long> config = new HashSet<>();
                    config.add(item.getPatrolType());
                    shieldMap.put(item.getAdvertId(), config);
                }
            });
        }
    }

    private void initConfigMap(Map<Long, Set<Long>> configMap, List<AdvertCouponBackDO> couponBackDOList, List<AdvertPromoteTestDO> promoteTestDOList) {
        //先初始化主链接
        if (CollectionUtils.isNotEmpty(couponBackDOList)) {
            couponBackDOList.forEach(item -> {
                if (configMap.containsKey(item.getAdvertId())) {
                    configMap.get(item.getAdvertId()).add(PatrolTypeEnum.Normal_LandingPage.getCode());
                } else {
                    Set<Long> config = new HashSet<>();
                    config.add(PatrolTypeEnum.Normal_LandingPage.getCode());
                    configMap.put(item.getAdvertId(), config);
                }
            });
        }
        //初始化配置链接 abtest 新媒体 配置
        if (CollectionUtils.isNotEmpty(promoteTestDOList)) {
            promoteTestDOList.forEach(item -> {
                Long type = null;
                if (item.getUrlType().equals(0)) {
                    type = PatrolTypeEnum.ABTest_LandingPage.getCode();
                }
                if (item.getUrlType().equals(1)) {
                    type = PatrolTypeEnum.NEW_Media_LandingPage.getCode();
                }
                if (item.getUrlType().equals(2)) {
                    type = PatrolTypeEnum.Config_LandingPage.getCode();
                }
                if (item.getUrlType().equals(4)) {
                    type = PatrolTypeEnum.Material_LandingPage.getCode();
                }
                if (configMap.containsKey(item.getAdvertId())) {
                    configMap.get(item.getAdvertId()).add(type);
                } else {
                    Set<Long> config = new HashSet<>();
                    config.add(type);
                    configMap.put(item.getAdvertId(), config);
                }
            });
        }
    }

    private boolean isLandPage(AdvertKeywordMatchDO advertKeywordMatchDO) {
        if (advertKeywordMatchDO.getPatrolType().equals(PatrolTypeEnum.Normal_LandingPage.getCode()) ||
                advertKeywordMatchDO.getPatrolType().equals(PatrolTypeEnum.ABTest_LandingPage.getCode()) ||
                advertKeywordMatchDO.getPatrolType().equals(PatrolTypeEnum.NEW_Media_LandingPage.getCode()) ||
                advertKeywordMatchDO.getPatrolType().equals(PatrolTypeEnum.Config_LandingPage.getCode()) ||
                advertKeywordMatchDO.getPatrolType().equals(PatrolTypeEnum.Special_LandingPage.getCode()) ||
                advertKeywordMatchDO.getPatrolType().equals(PatrolTypeEnum.Material_LandingPage.getCode())) {
            return true;
        }
        return false;
    }

    /**
     * 普通素材
     * @param advertFilterKeywordDO
     * @param list
     */
    private void buildMaterialList(AdvertFilterKeywordDO advertFilterKeywordDO, List<AdvertKeywordMatchDO> list) {
        //普通素材 key=广告ID value=素材列表
        Map<Long, Set<Long>> shieldMaterList = new HashMap<>();
        Map<Long, List<AdvertKeywordMatchDO>> map = list.stream().filter(item -> item.getShieldType().equals(0L) && item.getPatrolType().equals(PatrolTypeEnum.Normal_Material.getCode()))
                .collect(Collectors.groupingBy(AdvertKeywordMatchDO::getAdvertId));

        for(Map.Entry<Long, List<AdvertKeywordMatchDO>> entry : map.entrySet()){
            Long advertId = entry.getKey();
            List<AdvertKeywordMatchDO> matchDOList = entry.getValue();
            Set<String> materialIds = matchDOList.stream().map(AdvertKeywordMatchDO::getExtContent).collect(Collectors.toSet());
            Set<Long> materialList = new HashSet<>();
            materialIds.forEach(item -> {materialList.add(Long.valueOf(item));});
            shieldMaterList.put(advertId, materialList);
        }
        advertFilterKeywordDO.getGeneralFilterKeywordDO().setShieldMaterialList(shieldMaterList);

        buildNewTradeMaterialList(advertFilterKeywordDO, list);
    }

    private void buildNewTradeMaterialList(AdvertFilterKeywordDO advertFilterKeywordDO, List<AdvertKeywordMatchDO> list) {
        Map<Long, Set<Long>> shieldMaterList = new HashMap<>();
        Map<Long, List<AdvertKeywordMatchDO>> map = list.stream().filter(item -> item.getShieldType().equals(1L) && item.getPatrolType().equals(PatrolTypeEnum.Normal_Material.getCode()))
                .collect(Collectors.groupingBy(AdvertKeywordMatchDO::getAdvertId));

        for(Map.Entry<Long, List<AdvertKeywordMatchDO>> entry : map.entrySet()){
            Long advertId = entry.getKey();
            List<AdvertKeywordMatchDO> matchDOList = entry.getValue();
            Set<String> materialIds = matchDOList.stream().map(AdvertKeywordMatchDO::getExtContent).collect(Collectors.toSet());
            Set<Long> materialList = new HashSet<>();
            materialIds.forEach(item -> {materialList.add(Long.valueOf(item));});
            shieldMaterList.put(advertId, materialList);
        }
        advertFilterKeywordDO.getNewTradeFilterKeywordDO().setShieldMaterialList(shieldMaterList);
    }

    /**
     * 主链接列表
     * @param advertFilterKeywordDO
     * @param list
     */
    private void buildNormalLandingPageList(AdvertFilterKeywordDO advertFilterKeywordDO, List<AdvertKeywordMatchDO> list) {
        Set<Long> advertIdList = list.stream().filter(item -> item.getShieldType().equals(0L) && item.getPatrolType().equals(PatrolTypeEnum.Normal_LandingPage.getCode())).map(AdvertKeywordMatchDO::getAdvertId).collect(Collectors.toSet());
        advertFilterKeywordDO.getGeneralFilterKeywordDO().setShieldNormalLandingList(advertIdList);

        Set<Long> newTradeAdvertIdList = list.stream().filter(item -> item.getShieldType().equals(1L) && item.getPatrolType().equals(PatrolTypeEnum.Normal_LandingPage.getCode())).map(AdvertKeywordMatchDO::getAdvertId).collect(Collectors.toSet());
        advertFilterKeywordDO.getNewTradeFilterKeywordDO().setShieldNormalLandingList(newTradeAdvertIdList);
    }

    /**
     * abtest落地页
     */
    private void buildABTestLandingPageList(AdvertFilterKeywordDO advertFilterKeywordDO, List<AdvertKeywordMatchDO> list) {
        Map<Long, Set<String>> shieldABTestList = new HashMap<>();
        Map<Long, List<AdvertKeywordMatchDO>> map = list.stream().filter(item -> item.getShieldType().equals(0L) && item.getPatrolType().equals(PatrolTypeEnum.ABTest_LandingPage.getCode()))
                .collect(Collectors.groupingBy(AdvertKeywordMatchDO::getAdvertId));

        for(Map.Entry<Long, List<AdvertKeywordMatchDO>> entry : map.entrySet()){
            Long advertId = entry.getKey();
            List<AdvertKeywordMatchDO> matchDOList = entry.getValue();
            Set<String> urls = matchDOList.stream().map(AdvertKeywordMatchDO::getTypeUrl).collect(Collectors.toSet());
            shieldABTestList.put(advertId, urls);
        }
        advertFilterKeywordDO.getGeneralFilterKeywordDO().setShieldABTestLandingPageList(shieldABTestList);

        //行业关键词
        buildNewTradeABTestLandingPageList(advertFilterKeywordDO, list);
    }

    /**
     * 新行业
     * @param advertFilterKeywordDO
     * @param list
     */
    private void buildNewTradeABTestLandingPageList(AdvertFilterKeywordDO advertFilterKeywordDO, List<AdvertKeywordMatchDO> list) {
        Map<Long, Set<String>> shieldABTestList = new HashMap<>();
        Map<Long, List<AdvertKeywordMatchDO>> map = list.stream().filter(item -> item.getShieldType().equals(1L) && item.getPatrolType().equals(PatrolTypeEnum.ABTest_LandingPage.getCode()))
                .collect(Collectors.groupingBy(AdvertKeywordMatchDO::getAdvertId));

        for(Map.Entry<Long, List<AdvertKeywordMatchDO>> entry : map.entrySet()){
            Long advertId = entry.getKey();
            List<AdvertKeywordMatchDO> matchDOList = entry.getValue();
            Set<String> urls = matchDOList.stream().map(AdvertKeywordMatchDO::getTypeUrl).collect(Collectors.toSet());
            shieldABTestList.put(advertId, urls);
        }
        advertFilterKeywordDO.getNewTradeFilterKeywordDO().setShieldABTestLandingPageList(shieldABTestList);
    }

    /**
     * 配置落地页
     * @param advertFilterKeywordDO
     * @param list
     */
    private void buildConfigLandingPageList(AdvertFilterKeywordDO advertFilterKeywordDO, List<AdvertKeywordMatchDO> list) {
        Map<Long, Set<Long>> shieldConfigList = new HashMap<>();
        Map<Long, List<AdvertKeywordMatchDO>> map = list.stream().filter(item -> item.getShieldType().equals(0L) && item.getPatrolType().equals(PatrolTypeEnum.Config_LandingPage.getCode()))
                .collect(Collectors.groupingBy(AdvertKeywordMatchDO::getAdvertId));

        for(Map.Entry<Long, List<AdvertKeywordMatchDO>> entry : map.entrySet()){
            Long advertId = entry.getKey();
            List<AdvertKeywordMatchDO> matchDOList = entry.getValue();
            Set<String> orinPkgList = matchDOList.stream().map(AdvertKeywordMatchDO::getExtContent).collect(Collectors.toSet());
            Set<Long> orinPkgIds = new HashSet<>();
            orinPkgList.forEach(item -> { orinPkgIds.add(Long.valueOf(item)); });
            shieldConfigList.put(advertId, orinPkgIds);
        }

        advertFilterKeywordDO.getGeneralFilterKeywordDO().setShieldConfigLandingPageList(shieldConfigList);
        advertFilterKeywordDO.getGeneralFilterKeywordDO().setOnlyPromoteUrl(setOnlyPromoteUrlList());
        //新行业关键词过滤
        buildNewTradeConfigLandingPageList(advertFilterKeywordDO, list);
    }

    /**
     * 设置只有推广链接额的
     * @return
     */
    private Map<Long, Map<Long, Long>> setOnlyPromoteUrlList() {
        Map<Long, Map<Long, Long>> onlyMap = Maps.newHashMap();
        try {
            List<Long> validAdvertIds = advertMapCacheManager.getValidAdvertIds();
            if (CollectionUtils.isNotEmpty(validAdvertIds)) {
                for (Long advertId : validAdvertIds) {
                    List<AdvertOrientationPackageDO> list = advertOrientationPackageDAO.selectByAdvertIdList(advertId);
                    if (CollectionUtils.isNotEmpty(list)) {
                        Map<Long, Long> orientMap = Maps.newHashMap();
                        for (AdvertOrientationPackageDO orientationPackageDO : list) {
                            orientMap.put(orientationPackageDO.getId(), Long.valueOf(orientationPackageDO.getEnableAdvertUrl()));
                        }
                        onlyMap.put(advertId, orientMap);
                    }
                }
            }
        } catch (Exception e) {
            logger.warn("setOnlyPromoteUrlList error", e);
        }
        return onlyMap;
    }

    /**
     * 新行业关键词过滤
     * @param advertFilterKeywordDO
     * @param list
     */
    private void buildNewTradeConfigLandingPageList(AdvertFilterKeywordDO advertFilterKeywordDO, List<AdvertKeywordMatchDO> list) {
        Map<Long, Set<Long>> shieldConfigList = new HashMap<>();
        Map<Long, List<AdvertKeywordMatchDO>> map = list.stream().filter(item -> item.getShieldType().equals(1L) && item.getPatrolType().equals(PatrolTypeEnum.Config_LandingPage.getCode()))
                .collect(Collectors.groupingBy(AdvertKeywordMatchDO::getAdvertId));

        for(Map.Entry<Long, List<AdvertKeywordMatchDO>> entry : map.entrySet()){
            Long advertId = entry.getKey();
            List<AdvertKeywordMatchDO> matchDOList = entry.getValue();
            Set<String> orinPkgList = matchDOList.stream().map(AdvertKeywordMatchDO::getExtContent).collect(Collectors.toSet());
            Set<Long> orinPkgIds = new HashSet<>();
            orinPkgList.forEach(item -> { orinPkgIds.add(Long.valueOf(item)); });
            shieldConfigList.put(advertId, orinPkgIds);
        }

        advertFilterKeywordDO.getNewTradeFilterKeywordDO().setShieldConfigLandingPageList(shieldConfigList);
        advertFilterKeywordDO.getNewTradeFilterKeywordDO().setOnlyPromoteUrl(setOnlyPromoteUrlList());
    }

    /**
     * 新媒体落地页
     * @param advertFilterKeywordDO
     * @param list
     */
    private void buildNewMediaLandingPageList(AdvertFilterKeywordDO advertFilterKeywordDO, List<AdvertKeywordMatchDO> list) {
        Set<Long> advertIdList = list.stream().filter(item -> item.getShieldType().equals(0L) && item.getPatrolType().equals(PatrolTypeEnum.NEW_Media_LandingPage.getCode())).map(AdvertKeywordMatchDO::getAdvertId).collect(Collectors.toSet());
        advertFilterKeywordDO.getGeneralFilterKeywordDO().setShieldNewMediaLandingPageList(advertIdList);

        Set<Long> newTradeAdvertIdList = list.stream().filter(item -> item.getShieldType().equals(1L) && item.getPatrolType().equals(PatrolTypeEnum.NEW_Media_LandingPage.getCode())).map(AdvertKeywordMatchDO::getAdvertId).collect(Collectors.toSet());
        advertFilterKeywordDO.getNewTradeFilterKeywordDO().setShieldNewMediaLandingPageList(newTradeAdvertIdList);
    }

    /**
     * 落地页构建
     * @param advertFilterKeywordDO
     * @param list
     */
    private void buildLandingPageList(AdvertFilterKeywordDO advertFilterKeywordDO, List<AdvertKeywordMatchDO> list) {
        //主链接
        buildNormalLandingPageList(advertFilterKeywordDO, list);
        //测试落地页
        buildABTestLandingPageList(advertFilterKeywordDO, list);
        //新媒体
        buildNewMediaLandingPageList(advertFilterKeywordDO, list);
        //配置
        buildConfigLandingPageList(advertFilterKeywordDO, list);
        //素材绑定落地页测试
        buildMaterialLandingPageList(advertFilterKeywordDO, list);
    }

    /**
     * 素材绑定落地页测试
     * @param advertFilterKeywordDO
     * @param list
     */
    private void buildMaterialLandingPageList(AdvertFilterKeywordDO advertFilterKeywordDO, List<AdvertKeywordMatchDO> list) {
        Map<Long, Set<String>> shieldMaterialLandingPageMap = new HashMap<>();
        Map<Long, List<AdvertKeywordMatchDO>> map = list.stream().filter(item -> item.getShieldType().equals(0L) && item.getPatrolType().equals(PatrolTypeEnum.Material_LandingPage.getCode()))
                .collect(Collectors.groupingBy(AdvertKeywordMatchDO::getAdvertId));

        for(Map.Entry<Long, List<AdvertKeywordMatchDO>> entry : map.entrySet()){
            Long advertId = entry.getKey();
            List<AdvertKeywordMatchDO> matchDOList = entry.getValue();
            Set<String> urls = matchDOList.stream().map(AdvertKeywordMatchDO::getTypeUrl).collect(Collectors.toSet());
            shieldMaterialLandingPageMap.put(advertId, urls);
        }
        advertFilterKeywordDO.getGeneralFilterKeywordDO().setShieldMaterialLandingPageList(shieldMaterialLandingPageMap);

        buildNewTradeMaterialLandingPageList(advertFilterKeywordDO, list);
    }

    private void buildNewTradeMaterialLandingPageList(AdvertFilterKeywordDO advertFilterKeywordDO, List<AdvertKeywordMatchDO> list) {
        Map<Long, Set<String>> shieldMaterialLandingPageMap = new HashMap<>();
        Map<Long, List<AdvertKeywordMatchDO>> map = list.stream().filter(item -> item.getShieldType().equals(1L) && item.getPatrolType().equals(PatrolTypeEnum.Material_LandingPage.getCode()))
                .collect(Collectors.groupingBy(AdvertKeywordMatchDO::getAdvertId));

        for(Map.Entry<Long, List<AdvertKeywordMatchDO>> entry : map.entrySet()){
            Long advertId = entry.getKey();
            List<AdvertKeywordMatchDO> matchDOList = entry.getValue();
            Set<String> urls = matchDOList.stream().map(AdvertKeywordMatchDO::getTypeUrl).collect(Collectors.toSet());
            shieldMaterialLandingPageMap.put(advertId, urls);
        }
        advertFilterKeywordDO.getNewTradeFilterKeywordDO().setShieldMaterialLandingPageList(shieldMaterialLandingPageMap);
    }
}
