/**
 * Project Name:engine-service File Name:WelfareFlowAdvertCacheService.java Package Name:cn.com.duiba.tuia.cache
 * Date:2018年7月11日下午4:50:44 Copyright (c) 2018, duiba.com.cn All Rights Reserved.
 */

package cn.com.duiba.tuia.cache;

import cn.com.duiba.tuia.dao.advertNewtrade.AdvertNewTradeDAO;
import cn.com.duiba.tuia.dao.advertNewtrade.NewtradeGoodAppDAO;
import cn.com.duiba.tuia.dao.advertNewtrade.TagNewTradeDAO;
import cn.com.duiba.tuia.domain.dataobject.AdvertNewTradeDO;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * ClassName:WelfareFlowAdvertCacheService <br/>
 * Function: 福利流量广告发券缓存. <br/>
 * Date: 2018年7月11日 下午4:50:44 <br/>
 * 
 * @author chencheng
 * @version
 * @since JDK 1.8
 * @see
 */
@Component
public class AdvertNewTradeCacheService extends BaseCacheService {

    @Resource
    private ExecutorService       executorService;
    
    @Autowired
    private AdvertNewTradeDAO advertNewTradeDAO;

    @Autowired
    private NewtradeGoodAppDAO newtradeGoodAppDAO;

    @Autowired
    private TagNewTradeDAO tagNewTradeDAO;

    //广告标签对应的 综合行业ID
    private final LoadingCache<String, Optional<String>> NEW_TRADE_TAG_NUM_TO_ID_CACH = CacheBuilder.newBuilder()
            .initialCapacity(100).expireAfterWrite(1, TimeUnit.HOURS)
            .build(new CacheLoader<String, Optional<String>>() {
                @Override
                public Optional<String> load(String key) throws Exception {
                    return Optional.ofNullable(queryNewTradeIdByNewTradeNum(key));
                }

                @Override
                public ListenableFuture<Optional<String>> reload(String key, Optional<String> oldValue) throws Exception {
                    ListenableFutureTask<Optional<String>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });

    private String queryNewTradeIdByNewTradeNum(String key) {
        String tradeId = "";
        if(StringUtils.isNotBlank(key)){
            tradeId = tagNewTradeDAO.queryNewTradeIdByNewTradeNum(key);
            if(StringUtils.isBlank(tradeId)){
                tradeId = "";
            }
        }
        return tradeId;
    }

    private List<Long> queryTagGoodAppIds(String tagNum) {
        //查询广告标签对应的新行业
        String newTrade = tagNewTradeDAO.selectNewTradeByTagNum(tagNum);
        if (StringUtils.isBlank(newTrade)) {
            return Collections.emptyList();
        }
        //查询新行业对应的优质媒体
        return newtradeGoodAppDAO.selectNewTradeGoodAppIds(newTrade);
    }

    public String getTagNewTradeId(String tagNum) {
        Optional<String> newTradeId = NEW_TRADE_TAG_NUM_TO_ID_CACH.getUnchecked(tagNum);
        return newTradeId.orElse("");
    }

    //广告标签对应的优质媒体缓存
    private final LoadingCache<String, Optional<String>> TAG_NEW_TRADE = CacheBuilder.newBuilder()
            .initialCapacity(30).expireAfterWrite(1, TimeUnit.HOURS)
            .build(new CacheLoader<String, Optional<String>>() {
                @Override
                public Optional<String> load(String key) throws Exception {
                    return Optional.ofNullable(queryTagNewTrade(key));
                }

                @Override
                public ListenableFuture<Optional<String>> reload(String key, Optional<String> oldValue) throws Exception {
                    ListenableFutureTask<Optional<String>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });

    private String queryTagNewTrade(String tagNum) {
        //查询广告标签对应的新行业
        return tagNewTradeDAO.selectNewTradeByTagNum(tagNum);
    }

    //该方法我后期优化要用
    public String getTagNewTrade(String tagNum) {
        Optional<String> newTrade = TAG_NEW_TRADE.getUnchecked(tagNum);

        return newTrade.orElse("");
    }

    public void refreshTagNewTrade() {
        TAG_NEW_TRADE.invalidateAll();
        NEW_TRADE_TAG_NUM_TO_ID_CACH.invalidateAll();
    }

    /**
     * 广告对应的新行业标签缓存
     */
    private final LoadingCache<Long, Optional<AdvertNewTradeDO>> advertNewTradeCache = CacheBuilder.newBuilder()
            .initialCapacity(1000).expireAfterWrite(1,TimeUnit.HOURS)
            .build(new CacheLoader<Long, Optional<AdvertNewTradeDO>>() {
                @Override
                public Optional<AdvertNewTradeDO> load(Long key) throws Exception {
                    return Optional.ofNullable(advertNewTradeDAO.selectAdvertNewTradeById(key));
                }
                
                @Override
                public ListenableFuture<Optional<AdvertNewTradeDO>> reload(Long key, Optional<AdvertNewTradeDO> oldValue) throws Exception {
                    ListenableFutureTask<Optional<AdvertNewTradeDO>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }

                @Override
                public Map<Long, Optional<AdvertNewTradeDO>> loadAll(Iterable<? extends Long> keys){

                    List<Long> advertIds = Lists.newArrayList(keys);

                    List<AdvertNewTradeDO> advertNewTradeDOS = advertNewTradeDAO.selectAdvertNewTradeByAdIds(advertIds);

                    Map<Long, AdvertNewTradeDO> advertNewTradeDOMap = advertNewTradeDOS.stream()
                            .filter(Objects::nonNull)
                            .collect(Collectors.toMap(AdvertNewTradeDO::getAdvertId, Function.identity(), (o, n) -> n));

                    Map<Long, Optional<AdvertNewTradeDO>> resultMap = new HashMap<>();

                    advertIds.forEach(advertId -> resultMap.put(advertId, Optional.ofNullable(advertNewTradeDOMap.get(advertId))));

                    return resultMap;
                }
            });
    
    /**
     * 
     * getAdvertNewTradeName:(获取广告缓存行业). <br/>
     *
     * @author chencheng
     * @param advertId
     * @return
     * @since JDK 1.8
     */
    public AdvertNewTradeDO getAdvertNewTradeName(Long advertId) {
        try {
            AdvertNewTradeDO advertNewTradeDO =  advertNewTradeCache.get(advertId).orElse(new AdvertNewTradeDO(advertId, AdvertNewTradeDO.DEFULT_NEW_TRADE_NAME));
            // 行业默认值
            if (StringUtils.isBlank(advertNewTradeDO.getNewTradeName())) {
                advertNewTradeDO.setNewTradeName(AdvertNewTradeDO.DEFULT_NEW_TRADE_NAME);
            }
            return advertNewTradeDO;
        } catch (ExecutionException e) {
            logger.error("getAdvertNewTradeName error", e);
            return null;
        }
    }

    /**
     *
     * getAdvertNewTradeNameByAdIds:(批量获取广告缓存行业). <br/>
     *
     * @param advertIds
     * @return
     */
    public Map<Long, String> getAdvertNewTradeNameByAdIds(List<Long> advertIds) {
        try {
            Map<Long, Optional<AdvertNewTradeDO>> advertNewTradeMap = advertNewTradeCache.getAll(advertIds);

            Map<Long, String> resultMap = new HashMap<>();

            advertNewTradeMap.forEach((k, v) ->{

                AdvertNewTradeDO advertNewTradeDO = v.orElse(new AdvertNewTradeDO());

                String newTradeName = advertNewTradeDO.getNewTradeName() == null ? AdvertNewTradeDO.DEFULT_NEW_TRADE_NAME : advertNewTradeDO.getNewTradeName();

                resultMap.put(k, newTradeName);
            });

            return resultMap;
        } catch (ExecutionException e) {
            logger.error("getAdvertNewTradeNameByAdIds error", e);
            return new HashMap<>();
        }
    }
    
}
