/**
 * Project Name:engine-service<br>
 * File Name:ConsumerCacheService.java<br>
 * Package Name:cn.com.duiba.tuia.cache<br>
 * Date:2016年11月25日下午2:22:47<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */

package cn.com.duiba.tuia.cache;

import java.util.*;

import cn.com.duiba.tuia.constants.RepeatAdvertTestConstant;
import cn.com.duiba.tuia.domain.model.AdvQueryParam;
import cn.com.duiba.tuia.domain.model.RepeatTestGroup;
import cn.com.duiba.tuia.domain.vo.ConsumerRecordJsonVO;
import cn.com.duiba.tuia.utils.RecordTimeUtils;
import cn.com.duiba.wolf.utils.DateUtils;
import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.tuia.domain.dataobject.ConsumerInteractiveRecordDO;
import cn.com.duiba.tuia.domain.model.FilterResult;
import cn.com.duiba.tuia.exception.TuiaException;
import cn.com.duiba.tuia.service.ConsumerRecordSerivce;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.tuia.advert.constants.SystemConfigKeyConstant;

/**
 * ClassName: ConsumerCacheService <br/>
 * Function: 用户缓存服务. <br/>
 * date: 2016年11月25日 下午2:22:47 <br/>
 *
 * @author leiliang
 * @version 1.0
 * @since JDK 1.6
 */
@Slf4j
@Service
public class AdvertOrderCacheService extends BaseCacheService {

    @Autowired
    private ServiceManager            serviceManager;
    
    @Autowired
    private ConsumerRecordSerivce     consumerRecordSerivce;

    /**
     * 获取(全局配置的重复领取天数)领取过该广告列表.
     *
     * @param consumerVOList 用户列表
     * @return 领取过的广告类别
     */
    public List<Long> getReceiveAdvertIds(List<ConsumerInteractiveRecordDO> consumerVOList, AdvQueryParam advQueryParam) throws TuiaException {
        if (consumerVOList.isEmpty()) {
            return Collections.emptyList();
        }
        List<Long> receiveAdvertIds = Lists.newArrayList();
        Set<String> adxAdverts = Sets.newHashSet();
        try {
            DBTimeProfile.enter("getReceiveAdvertIds");
            Integer singleRepeatDays = serviceManager.getIntValue(SystemConfigKeyConstant.ADVERT_SINGLE_REPEAT_DAY);
            Date orderTime = RecordTimeUtils.getTodayMMdd(DateUtils.changeByDay(new Date(), ~singleRepeatDays + 1));

            // [重复发券测试需求]测试组重新计算订单时间
            RepeatTestGroup repeatTestGroup = advQueryParam.getRepeatTestGroup();
            if (isRepeatTestGroup(repeatTestGroup)) {
                orderTime = DateUtils.hoursAddOrSub(new Date(), -repeatTestGroup.getInterval());
            }

            for (ConsumerInteractiveRecordDO record : consumerVOList) {
                ConsumerRecordJsonVO vo = JSON.parseObject(record.getJson(), ConsumerRecordJsonVO.class);
                if (orderTime.before(record.getGmtCreate())) {
                    if (vo != null && vo.getDspId() != null) {
                        adxAdverts.add(String.valueOf(vo.getDspId()) + ";" + String.valueOf(record.getAdvertId()));
                    } else {
                        receiveAdvertIds.add(record.getAdvertId());
                    }
                }
            }
            //dsp adx 广告  24h 同一用户  已经曝光的广告列表   dspId+";"+advertId
            advQueryParam.setAfterAdxAdverts(adxAdverts);
            return receiveAdvertIds;
        } catch (Exception e) {
            logger.warn("getReceiveAdvertIds error", e);
        } finally {
            DBTimeProfile.release();
        }
        return Collections.emptyList();
    }

    /**
     * 查询广告id的发券次数，可以重复曝光的广告领取次数是当天的次数
     *
     * @param filterResult
     * @param advertIds
     * @return
     * @throws TuiaException
     */
    public Map<String, String> getAdvertLaunchedTimes(FilterResult filterResult, List<Long> advertIds) throws TuiaException {

        //可以重复曝光的广告
        List<Long> multipleExposureIds = filterResult.getMultipleExposureIds();
        Map<Long,Integer> limitAdvertTodayMap = filterResult.getLimitAdvertTodayMap();
        Map<Long,Integer> allUserTimes = filterResult.getAllUserTimesMap();
        
        //从数据库中取出的是所有的id，需要筛选出advertIds对应的数据
        Map<String, String> newUsed = new HashMap<>(advertIds.size());
        Integer value;
        for (Long advertId : advertIds) {

            value = allUserTimes.get(advertId);
            if(multipleExposureIds.contains(advertId)){
                value = limitAdvertTodayMap.get(advertId);
            }

            if (value != null) {
                newUsed.put(advertId.toString(), value.toString());
            }
        }

        return newUsed;
    }

    /**
     * 查询广告id的发券次数.
     *
     * @param allUserTimesMap the all user times map
     * @param advertId the advert id
     * @return the advert launched times
     */
    public int getAdvertLaunchedTimes(Map<Long,Integer> allUserTimesMap, Long advertId){
        if(MapUtils.isEmpty(allUserTimesMap)){
            return 0;
        }
        return Optional.ofNullable(allUserTimesMap.get(advertId)).orElse(0);
    }

    /**
     * 查询adx广告id的发券次数.
     *
     * @param allUserTimesMap the all user times map
     * @param key dspId+";"+advertid
     * @return the advert launched times
     */
    public int getAdxAdvertLaunched(Map<String,Integer> allUserTimesMap, String key){
        if(MapUtils.isEmpty(allUserTimesMap)){
            return 0;
        }
        return Optional.ofNullable(allUserTimesMap.get(key)).orElse(0);
    }

    /**
     * [重复发券测试需求]判断是否分流到测试组
     *
     * @param group 测试组
     * @return 是否分流到测试组
     */
    private boolean isRepeatTestGroup(RepeatTestGroup group) {
        return null != group && !Objects.equals(group.getGroup(), RepeatAdvertTestConstant.CONTROL_GROUP) && null != group.getInterval();
    }
}
