package cn.com.duiba.tuia.cache;

import cn.com.duiba.tuia.dao.engine.AdvertAdjustDAO;
import cn.com.duiba.tuia.dao.engine.AdvertAdjustRatioDAO;
import cn.com.duiba.tuia.dao.engine.AdvertOrientationPackageDAO;
import cn.com.duiba.tuia.domain.dataobject.AdjustRatioDo;
import cn.com.duiba.tuia.domain.dataobject.AdvertOrientationPackageDO;
import cn.com.duiba.tuia.enums.AdvertSubtypeEnum;
import cn.com.duiba.tuia.tool.StringTool;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * Created by MichealWang on 2018/11/5 0005.
 * 自定义调整因子下 媒体自定义转化出价缓存
 */
@Component
public class AdvertPackageAppAdjustFeeCacheService  extends BaseCacheService{

    private static final Long DEFAULT_APP_ID = -1L;
    @Autowired
    private AdvertOrientationPackageDAO advertOrientationPackageDAO;
    @Resource
    private ExecutorService executorService;
    @Autowired
    private AdvertAdjustDAO adjustDAO;
    @Autowired
    private AdvertAdjustRatioDAO adjustRatioDAO;

    private final LoadingCache<String, Long> advertPackageAppAdjustFeeCache = CacheBuilder.newBuilder()
            .initialCapacity(1000).expireAfterWrite(1, TimeUnit.HOURS)
            .build(new CacheLoader<String, Long>() {
                @Override
                public Long load(String key)  {
                    return selectAdvertPackageAppAdjustFee(key);
                }

                @Override
                public ListenableFuture<Long> reload(String key, Long oldValue) {
                    ListenableFutureTask<Long> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }

            });

    /**
     * 根据 AdvertId,packagerId,appId 获取对应的 转化出价
     * @param key
     * @return
     */
    private Long selectAdvertPackageAppAdjustFee(String key) {
        try {
            // 根据广告id配置id 定向媒体id查询出价
            List<Long> adPkgApp = StringTool.getLongListByStr(key);

            Long advertId = adPkgApp.get(0);
            Long packageId = adPkgApp.get(1);
            Long appId = adPkgApp.get(2);


            //1、查询ae是否有自定义媒体转化出价
            Long price = adjustDAO.getAppCustomPrice(advertId, packageId, appId);
            if(price!=null){
                return price;
            }
            //2、查询签收率和该配置的成本
            AdvertOrientationPackageDO packageDO = advertOrientationPackageDAO.selectById(packageId);
            if (packageDO == null){
                return null;
            }
            List<AdjustRatioDo> ratioDos = adjustRatioDAO.selectAdjustRatios(advertId, AdvertSubtypeEnum.SIGN_FOR.getSubtype());
            Map<Long, String> map = ratioDos.stream()
                    .filter(dto -> null != dto && null != dto.getAdjustRatio()).collect(Collectors.toMap(dto -> dto.getDataType()==0?DEFAULT_APP_ID:dto.getAppId(),
                    AdjustRatioDo::getAdjustRatio, (oldVal, newVal) -> newVal));

            String adjustRatio = map.getOrDefault(appId, map.get(DEFAULT_APP_ID));

            Double cost = packageDO.getAdjustCost() * Double.parseDouble(adjustRatio);
            return new BigDecimal(cost).setScale(0, BigDecimal.ROUND_HALF_UP).longValue();
        } catch (Exception e) {
            logger.error("AdvertPackageAppAdjustFeeCacheService.selectAdvertPackageAppAdjustFee key={{}}", key, e);
            return null;
        }
    }


    /**
     * 根据 AdvertId,packagerId,appId 获取对应的 转化出价
     * @param key
     * @return
     */
    public Long getCpaFee(String key)  {
        try {
            return advertPackageAppAdjustFeeCache.get(key);
        }catch (Exception e){
            logger.error("AdvertPackageAppAdjustFeeCacheService.getCpaFee key={{}}",key,e);
            //如果出现异常则降级处理返回0 本次不投这个广告
            Long demotionFee = selectAdvertPackageAppAdjustFee(key);
            return demotionFee==null?0L:demotionFee;
        }
    }

    /** 更新缓存*/
    public void invalidateAdjustFee(String key) {
        advertPackageAppAdjustFeeCache.invalidate(key);
        logger.info("AdvertPackageAppAdjustFeeCacheService.invalidateAdjustFee key {{}}",key);
    }
}
