package cn.com.duiba.tuia.cache;

import cn.com.duiba.tuia.activity.center.api.dto.staticpage.VerifyAllResultQuery;
import cn.com.duiba.tuia.activity.center.api.dto.staticpage.VerifyPageResultDto;
import cn.com.duiba.tuia.activity.center.api.remoteservice.staticpage.RemoteStaticPageVerifyService;
import cn.com.duiba.tuia.dao.advert.AdvertCouponBackDAO;
import cn.com.duiba.tuia.dao.engine.AdvertOrientationPackageDAO;
import cn.com.duiba.tuia.dao.promotetest.AdvertPromoteTestDAO;
import cn.com.duiba.tuia.domain.dataobject.AdvertCouponBackDO;
import cn.com.duiba.tuia.domain.dataobject.AdvertPromoteTestDO;
import cn.com.duiba.tuia.domain.dataobject.OrientPkgFilterUrlDO;
import cn.com.duiba.tuia.domain.vo.AdvertOrientationPackageVO;
import cn.com.duiba.tuia.service.AdvertOrientationService;
import com.google.common.base.Splitter;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@Service
@RefreshScope
public class AdvertPkgCacheService extends BaseCacheService {

    @Resource
    private ExecutorService executorService;
    @Autowired
    private AdvertPromoteTestDAO advertPromoteTestDAO;
    @Autowired
    private AdvertCouponBackDAO advertCouponBackDAO;
    @Autowired
    private AdvertOrientationPackageDAO advertOrientationPackageDAO;
    @Autowired
    private RemoteStaticPageVerifyService remoteStaticPageVerifyService;
    @Autowired
    private AdvertOrientationService advertOrientationService;

    //有消息同步，不用设置过期时间来保证避免读取旧值
    private LoadingCache<String, OrientPkgFilterUrlDO> advertPkgFilterList = CacheBuilder.newBuilder().initialCapacity(1000).maximumSize(1200)
            .refreshAfterWrite(5, TimeUnit.MINUTES)
            .build(
                    new CacheLoader<String, OrientPkgFilterUrlDO>() {
                        public OrientPkgFilterUrlDO load(String key) {
                            return getAdvertPkgByCache(key);
                        }

                        public ListenableFutureTask<OrientPkgFilterUrlDO> reload(final String key, OrientPkgFilterUrlDO prevGraph) {
                            ListenableFutureTask<OrientPkgFilterUrlDO> task = ListenableFutureTask.create(() -> {
                                return getAdvertPkgByCache(key);
                            });
                            executorService.submit(task);
                            return task;
                        }
                    });

    /**
     * 根据KEY获取缓存信息
     * @param key
     * @return
     */
    public OrientPkgFilterUrlDO getAdvertOrientPkgCache(String key) {
        if (StringUtils.isBlank(key)) {
            return null;
        }
        try {
            return advertPkgFilterList.get(key);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 构建缓存
     * @param key
     * @return
     */
    public OrientPkgFilterUrlDO getAdvertPkgByCache(String key) {
        List<String> strs = Splitter.on("|").splitToList(key);//advertId|orientId
        Long advertId = Long.parseLong(strs.get(0));
        Long orientId = Long.parseLong(strs.get(1));
        OrientPkgFilterUrlDO orientPkgFilterUrlDO = new OrientPkgFilterUrlDO();
        //1.查询广告推广链接 是否支持爱奇艺
        AdvertCouponBackDO advertCouponBackDO = advertCouponBackDAO.getAdvertCouponBack(advertId);
        if (null == advertCouponBackDO) {
            return new OrientPkgFilterUrlDO(false, false);
        }
        //2.广告配置下链接是否支持爱奇艺
        //默认配置
        String orientUrl = "";
        if (orientId.equals(0L)) {
            orientId = advertOrientationPackageDAO.selectPkgIdByAdvertId(advertId);
        }
        AdvertPromoteTestDO advertPromoteTestDO = advertPromoteTestDAO.selectVaildOrientPkgUrlByAdvertId(advertId, orientId);
        if (Objects.nonNull(advertPromoteTestDO) && StringUtils.isNotBlank(advertPromoteTestDO.getPromoteUrl())) {
            orientUrl = advertPromoteTestDO.getPromoteUrl();
        }
        List<String> urlList = Lists.newArrayList();
        if (StringUtils.isNotBlank(advertCouponBackDO.getPromoteURL())) {
            urlList.add(advertCouponBackDO.getPromoteURL());
        }
        if (StringUtils.isNotBlank(orientUrl)) {
            urlList.add(orientUrl);
        }
        List<String> resultUrlList = getUrlMappingList(urlList);
        if (CollectionUtils.isEmpty(resultUrlList)) {
            return new OrientPkgFilterUrlDO(false, false);
        }
        if (resultUrlList.contains(advertCouponBackDO.getPromoteURL())) {
            orientPkgFilterUrlDO.setAdvertSupport(true);
        }
        if (resultUrlList.contains(orientUrl)) {
            orientPkgFilterUrlDO.setPkgSupport(true);
        }

        return orientPkgFilterUrlDO;
    }

    /**
     * 初始化缓存列表
     * @param advertList
     */
    public void init(List<Long> advertList) {
        if (CollectionUtils.isEmpty(advertList)) {
            return;
        }

        for (Long advertId : advertList) {
            List<AdvertOrientationPackageVO> orientationPackageVOList = advertOrientationService.getOrientationList(advertId);
            if (CollectionUtils.isEmpty(orientationPackageVOList)) {
                continue;
            }

            orientationPackageVOList.forEach(item-> {
                String pkgId = item.isDefaultOrientation() ? "0" : item.getOrientationId().toString();
                String key = advertId + "|" + pkgId;
                OrientPkgFilterUrlDO orientPkgFilterUrlDO = getAdvertPkgByCache(key);
                advertPkgFilterList.put(key, orientPkgFilterUrlDO);
            });
        }
    }

    /**
     * 更新广告配置缓存
     * @param advertId 广告ID
     */
    public void updateAdvertOrientPkgCache(Long advertId) {
        //更新广告配置缓存
        List<AdvertOrientationPackageVO> orientationPackageVOList = advertOrientationService.getOrientationList(advertId);
        if (CollectionUtils.isEmpty(orientationPackageVOList)) {
            return;
        }

        orientationPackageVOList.forEach(item-> {
            String pkgId = item.isDefaultOrientation() ? "0" : item.getOrientationId().toString();
            String key = advertId + "|" + pkgId;
            advertPkgFilterList.refresh(key);
        });
    }

    /**
     * 更新广告配置缓存
     * @param key
     */
    public void updateAdvertOrientPkgCache(String key) {
        try {
            advertPkgFilterList.refresh(key);
        } catch (Exception e) {
            logger.error("刷新缓存异常", e);
        }
    }

    private List<String> getUrlMappingList(List<String> urlList) {
        try {
            VerifyAllResultQuery verifyAllResultQuery = new VerifyAllResultQuery();
            verifyAllResultQuery.setPlatformType(1);
            verifyAllResultQuery.setPageType(2);
            verifyAllResultQuery.setTuiaUrls(urlList);
            List<VerifyPageResultDto> list = remoteStaticPageVerifyService.selectAllVerifyResult(verifyAllResultQuery);
            if (CollectionUtils.isEmpty(list)) {
                return new ArrayList<>();
            }

            List<String> resultUrlList = list.stream().filter(item -> item.getLastStatus().equals(2)).map(VerifyPageResultDto::getTuiaUrl).collect(Collectors.toList());
            return resultUrlList;
        } catch (Exception e) {
            logger.error("查询爱奇艺映射异常", e);
            return new ArrayList<>();
        }
    }
}
