package cn.com.duiba.tuia.cache;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import javax.annotation.Resource;

import cn.com.duiba.tuia.utils.DateUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Service;

import cn.com.duiba.tuia.dao.advert.PackageBudgetConsumeDAO;
import cn.com.duiba.tuia.dao.engine.AdvertOrientationPackageDAO;
import cn.com.duiba.tuia.domain.dataobject.AdvertOrientationPackageDO;
import cn.com.duiba.tuia.domain.dataobject.PackageBudgetConsumeDO;
import cn.com.duiba.tuia.domain.vo.AdvertOrientationPackageVO;
import cn.com.duiba.tuia.domain.vo.AdvertVO;
import cn.com.duiba.tuia.exception.TuiaException;
import cn.com.duiba.tuia.service.AdvertInvalidHandleService;
import cn.com.duiba.tuia.service.AdvertOrientationService;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.tuia.advert.constants.SystemConfigKeyConstant;

import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Splitter;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;

/**
 * Created by hujinliang on 2017/8/14.
 */
@Service
public class AdvertPkgPutFlagCacheServiceImpl implements AdvertPkgPutFlagCacheService {
    private static Logger logger = LoggerFactory.getLogger(AdvertPkgPutFlagCacheService.class);

    @Autowired
    private ServiceManager serviceManager;
    @Autowired
    private AdvertOrientationService advertOrientationService;
    @Autowired
    private PackageBudgetConsumeDAO packageBudgetConsumeDAO;
    @Autowired
    private AdvertMapCacheManager advertMapCacheManager;
    @Autowired
    private AdvertOrientationPackageDAO orientationPackageDAO;
    @Autowired
    private AdvertInvalidHandleService advertInvalidHandleService;
    @Resource
    private ExecutorService executorService;

    /**
     * 广告配置的key是以广告id、配置包id、当天为key的，全量肯定是大于有效广告1600的，
     * 所以当天的缓存1000是不够的，所以增加了容量
     * 而且数据超过一天肯定是无效的，所以加了过期时间24小时
     */
    @SuppressWarnings("squid:S3776")
    private LoadingCache<String, Boolean> advertPkgPutFlagCache = CacheBuilder.newBuilder().initialCapacity(2000).
            recordStats().refreshAfterWrite(600 , TimeUnit.SECONDS)
            .build(
                    new CacheLoader<String, Boolean>() {
                        public Boolean load(String key) {
                            List<String> strs= Splitter.on("-").splitToList(key);
                            Long advertId=Long.parseLong(strs.get(0));
                            Long orientId=Long.parseLong(strs.get(1));
                            //查询定向配置包
                            AdvertOrientationPackageDO pkgDO = advertOrientationService.getOrientation(advertId,orientId);
                            if (pkgDO == null||null==pkgDO.getId()) {
                                return false;
                            }
                            try {
                                return checkConsume(advertId, orientId, pkgDO.getBudgetPerDay(), null);
                            } catch (Exception e) {
                                logger.error("checkConsume error, advertId:{},orientId:{}",advertId, orientId, e);
                                return false;
                            } 
                        }

                        @Override
                        public ListenableFuture<Boolean> reload(String key, Boolean oldValue) throws Exception {
                            ListenableFutureTask<Boolean> task = ListenableFutureTask.create(() -> load(key));
                            executorService.submit(task);
                            return task;
                        }
                    });


    @EventListener(cn.com.duiba.boot.event.MainContextRefreshedEvent.class)
    public void afterInit() throws Exception {
        initPkgPutCache();
    }


    public void initPkgPutCache() {
        long start = System.currentTimeMillis();
        try {
            List<Long> advertIds = advertMapCacheManager.getValidAdvertIds();
            if (advertIds.isEmpty()) {
                return;
            }
            Map<Long, List<AdvertOrientationPackageDO>> advertsMap = Maps.newHashMap();
            List<AdvertOrientationPackageDO> packages = orientationPackageDAO.selectListByAdvertIds(advertIds);
            if (CollectionUtils.isEmpty(packages)) {
                return;
            }
            for (AdvertOrientationPackageDO pack : packages) {
                List<AdvertOrientationPackageDO> list = advertsMap.get(pack.getAdvertId());
                if (null == list) {
                    list = Lists.newArrayList();
                    advertsMap.put(pack.getAdvertId(), list);
                }
                if (pack.getIsDefault() == 1) {
                    pack.setId(0L);
                }
                list.add(pack);
            }

            for (Entry<Long, List<AdvertOrientationPackageDO>> entry : advertsMap.entrySet()) {
                Long advertId = entry.getKey();
                List<AdvertOrientationPackageDO> advertPacks = entry.getValue();
                String key = "";
                for (AdvertOrientationPackageDO od : advertPacks) {
                    key = getCacheKey(advertId, od.getId(), getCurDate());
                    boolean flag = checkConsume(advertId, od.getId(), od.getBudgetPerDay(), null);
                    advertPkgPutFlagCache.put(key, flag);
                }
            }
        } catch (Exception e) {
            logger.error("", e);
        } finally {
            logger.info("initPkgPutCache consume : " + (System.currentTimeMillis() - start) + " ms");
        }
    }


    public void invalidPkgPutCache(Long advertId) {
        boolean hasDefault = false;
        List<AdvertOrientationPackageVO> packages = advertOrientationService.getOrientationList(advertId);
        for (AdvertOrientationPackageVO pa : packages) {
            String key = "";
            if (pa.isDefaultOrientation()) {
                hasDefault = true;
                key = getCacheKey(pa.getAdvertId(), 0L, DateUtils.getDayStr(new Date()));
            } else {
                key = getCacheKey(pa.getAdvertId(), pa.getId(), DateUtils.getDayStr(new Date()));
            }
            advertPkgPutFlagCache.refresh(key);
        }
        if (!hasDefault) {
            advertPkgPutFlagCache.refresh(getCacheKey(advertId, 0L, DateUtils.getDayStr(new Date())));
        }
    }
    public void invalidPkgPutCache(Long advertId,Long packageId) {
        String key = getCacheKey(advertId, packageId, DateUtils.getDayStr(new Date()));
        advertPkgPutFlagCache.refresh(key);
    }
    /**
     * 日预算自动释放校验
     *
     * @return
     * @throws TuiaException
     */
    public boolean beforeAdvertPkgPutFlag() throws TuiaException, ParseException {
        String value = serviceManager.getStrValue(SystemConfigKeyConstant.ADVERT_PACKAGE_BUDGET_TIME_RELEASE);
        if (StringUtils.isBlank(value)) {
            return Boolean.FALSE;
        }
        Date curDate = new Date();

        Date configDate = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").parse(DateUtils.getDayStr(curDate) + " " + value + ":00");
        Date endDate = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").parse(DateUtils.getDayStr(curDate) + " 23:59:59");
        //当前时间  大于  配置时间 ， 且  当前日期 小于 凌晨，则释放默认配置
        if (curDate.after(configDate) && curDate.before(endDate)) {
            return Boolean.TRUE;
        }
        return Boolean.FALSE;
    }

    /**
     * @param advertId     广告id
     * @param pkgId        配置包id
     * @param budgetPerDay 定向配置包日预算
     * @return true(可投放)，false(不可投放)
     * @throws TuiaException
     */
    public Boolean getAdvertPkgPutFlag( long advertId,  long pkgId,  Long budgetPerDay) {

        try {
            String key = getCacheKey(advertId, pkgId, getCurDate());
            Boolean ifBudget = advertPkgPutFlagCache.getIfPresent(key);
            if (ifBudget != null) {
                return ifBudget;
            }
            advertPkgPutFlagCache.refresh(key);
        } catch (Exception e) {
            logger.error("判断定向配置包日预算常，advertId:{},pkgId:{},异常：", advertId, pkgId, e);
        }
        return false;
    }

    /**
     * 更新缓存中当前配置包有效状态
     *
     * @param msgBody
     * @throws TuiaException
     * @throws ParseException 
     */
    public void updateAdvertPkgPutFlag(String msgBody) throws TuiaException, ParseException {
        UpdateAdvertPkgPutFlag msgObj = JSONObject.parseObject(msgBody, UpdateAdvertPkgPutFlag.class);
        if (msgObj.getOrientationPackageId() == null || msgObj.getAdvertId() == null) {
            return;
        }
        Long advertId = msgObj.getAdvertId();
        Long planId = msgObj.getOrientationPackageId();
        String curDate = msgObj.getCurDate();
        Boolean flag = msgObj.getFlag();
        String key = getCacheKey(advertId, planId, curDate);
        if (flag == null) {
            //查询定向配置包
            AdvertOrientationPackageDO pkgDO = advertOrientationService.getOrientation(advertId,msgObj.getOrientationPackageId());
            if (pkgDO == null||null==pkgDO.getId()) {
                return;
            }
            curDate = StringUtils.isBlank(curDate) ? getCurDate() : curDate;
            flag = checkConsume(advertId, planId, pkgDO.getBudgetPerDay(), curDate);
        }
        if (!flag) {
            advertInvalidHandleService.sendBudgetNotEnoughDingNotice(advertId, planId, curDate);
        }
        advertPkgPutFlagCache.put(key, flag);
    }

    /**
     * 校验当前定向配置包预算1
     *
     * @param advertId
     * @param pkgId
     * @param budgetPerDay
     * @param curDate
     * @return
     * @throws TuiaException
     * @throws ParseException 
     */
    private boolean checkConsume(long advertId, long pkgId, Long budgetPerDay, String curDate) throws TuiaException, ParseException {
        curDate = StringUtils.isBlank(curDate) ? getCurDate() : curDate;
        // 定向配置有预算,计算是否有剩余预算
        if(budgetPerDay != null){
            boolean flag = false;
            try {
                DBTimeProfile.enter("for checkConsume have bugetperday is null");
                Long consumeTotal = packageBudgetConsumeDAO.getTodayConsumeByAdvertPkg(advertId, pkgId, curDate);
                flag = checkFlag(budgetPerDay, consumeTotal);
            }finally{
                DBTimeProfile.release();
            }
            return flag;
        }
        // 广告预算为不限，本配置预算为不限，则不限制
        AdvertVO advertVO = advertMapCacheManager.getAdvertCache(advertId);
        if (null == advertVO || null == advertVO.getAdvertPlan()
                || null == advertVO.getAdvertPlan().getBudgetPerDay()) {
            return true;
        }
        // 到达释放点时间后，配置剩余预算为 ： 广告预算-广告消耗
        if (this.beforeAdvertPkgPutFlag()) {
            PackageBudgetConsumeDO advertonsumeDO = packageBudgetConsumeDAO.getTodayConsumeByAdvert(advertId, curDate);
            Long advertConsumeTotal = Optional.ofNullable(advertonsumeDO).map(dto -> Optional.ofNullable(dto.getConsumeTotal()).orElse(0L)).orElse(0L);
            return checkFlag(advertVO.getAdvertPlan().getBudgetPerDay(), advertConsumeTotal);
        }
        // 预算未释放时。广告有预算，所有配置均不限预算，则共用广告预算，不进行额外判断
        Set<Long> packIds = Sets.newHashSet();
        Long NoTotal = buildBugetLimitList(advertId, packIds);
        if(NoTotal.equals(0L)){
            return true;
        }
        // 本配置无预算，其他配有预算。本配置剩余预算为   （广告预算-有限制的配置的预算）-（所有配置的消耗-有限制的配置的消耗）
        Long totalConsume = 0L;
        Long hasConsumeTotal = 0L;
        List<PackageBudgetConsumeDO> total = packageBudgetConsumeDAO.getTodayTotalConsumeByAdvert(advertId, curDate);
        for (PackageBudgetConsumeDO consume : total) {
            totalConsume += consume.getConsumeTotal();
            if (packIds.contains(consume.getAdvertPackageId())) {
                hasConsumeTotal += consume.getConsumeTotal();
            }
        }
        Long noT = advertVO.getAdvertPlan().getBudgetPerDay() - NoTotal;//未分配预算
        return checkFlag(noT, totalConsume - hasConsumeTotal);
    }

    /**
     * 拼装有预算限制的配置包id列表
     * @param advertId
     * @param packIds
     * @return
     */
    private Long buildBugetLimitList(long advertId, Set<Long> packIds) {
        Long noTotal=0L;
        List<AdvertOrientationPackageVO> origntList = advertOrientationService.getOrientationList(advertId);
        for (AdvertOrientationPackageVO vo : origntList) {
            if (vo.getBudgetPerDay() != null) {
                noTotal += vo.getBudgetPerDay();
                if (vo.isDefaultOrientation()) {
                    packIds.add(0L);
                } else {
                    packIds.add(vo.getId());
                }
            }
        }
        return noTotal;
    }

    private boolean checkFlag(Long budgetPerDay, long consumeTotal) {
        //当前定向配置包“剩余预算” = 定向配置包日预算 - 今日定向包配置“已消耗”
        Long pkgSurplusTodayBudget = budgetPerDay - consumeTotal;
        //剩余预算 < 定向包配置预算，不可投
        if (pkgSurplusTodayBudget <= 0) {
            return Boolean.FALSE;
        }
        return Boolean.TRUE;
    }


    private String getCacheKey(long advert, long pkgId, String curDate) {
        return advert + "-" + pkgId + "-" + curDate;
    }


    private String getCurDate() {
        return DateUtils.getDayStr(new Date());
    }

}

class UpdateAdvertPkgPutFlag {
    public UpdateAdvertPkgPutFlag() {

    }

    public UpdateAdvertPkgPutFlag(Long advertId, Long orientationPackageId, String curDate, Boolean flag) {
        this.advertId = advertId;
        this.orientationPackageId = orientationPackageId;
        this.curDate = curDate;
        this.flag = flag;
    }

    //广告id
    private Long advertId;
    //定向配置id
    private Long orientationPackageId;
    //日期
    private String curDate;
    // 是否可投放标识
    private Boolean flag;

    public Long getAdvertId() {
        return advertId;
    }

    public Long getOrientationPackageId() {
        return orientationPackageId;
    }

    public String getCurDate() {
        return curDate;
    }

    public void setAdvertId(Long advertId) {
        this.advertId = advertId;
    }

    public void setOrientationPackageId(Long orientationPackageId) {
        this.orientationPackageId = orientationPackageId;
    }

    public void setCurDate(String curDate) {
        this.curDate = curDate;
    }

    public Boolean getFlag() {
        return flag;
    }

    public void setFlag(Boolean flag) {
        this.flag = flag;
    }
}
