package cn.com.duiba.tuia.cache;

import cn.com.duiba.tuia.dao.promotetest.AdvertNewPromoteTestDao;
import cn.com.duiba.tuia.dao.engine.AdvertiserAuditDao;
import cn.com.duiba.tuia.dao.promotetest.AdvertPromoteTestDAO;
import cn.com.duiba.tuia.domain.dataobject.AdvertPromoteTestDO;
import cn.com.duiba.tuia.domain.dataobject.AdvertiserCheckAdvertDO;
import cn.com.duiba.tuia.domain.vo.MaterialPromoteTestUrlsVO;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@Service
public class AdvertPromoteTestCacheService {

    private static Logger logger = LoggerFactory.getLogger(AdvertPromoteTestCacheService.class);

    @Autowired
    private AdvertPromoteTestDAO advertPromoteTestDAO;

    @Autowired
    private AdvertNewPromoteTestDao advertNewPromoteTestDao;

    @Autowired
    private AdvertiserAuditDao advertiserAuditDao;

    @Resource
    private ExecutorService executorService;


    /**
     *
     */
    private Set<Long> getAdvertiserUnAuditPromoteUrlIdByAdvertId(Long advertId){
        if(null == advertId){
            return new HashSet<>();
        }

        List<AdvertiserCheckAdvertDO> list = advertiserAuditDao.selectByAdvertId(advertId);
        if(CollectionUtils.isEmpty(list)){
            return new HashSet<>();
        }

        return list.stream().filter(advertiserCheckAdvertDO -> !Objects.equals(3, advertiserCheckAdvertDO.getCheckStatus())).
                map(AdvertiserCheckAdvertDO::getPromoteUrlId).collect(Collectors.toSet());
    }

    //有消息同步，不用设置过期时间来保证避免读取旧值
    private LoadingCache<Long, Optional<MaterialPromoteTestUrlsVO>> ADVERT_PROMOTE_TEST_CACHE = CacheBuilder.newBuilder()
            .initialCapacity(1000).refreshAfterWrite(1, TimeUnit.HOURS)
            .build(new CacheLoader<Long, Optional<MaterialPromoteTestUrlsVO>>() {
                @Override
                public Optional<MaterialPromoteTestUrlsVO> load(Long advertId) {
                    try {

                        Boolean isNewType=false;
                        //findPromoteUrlsByAdvertId方法 是从tb_advert_promote_test表查落地页测试信息.这是老逻辑
                        List<AdvertPromoteTestDO> promoteTestDOListAll = advertPromoteTestDAO.findPromoteUrlsByAdvertId(advertId);
                        // 如果老逻辑没有才从新逻辑里查
                        if (CollectionUtils.isEmpty(promoteTestDOListAll)) {
                            //新落地 落地页测试计划单独拆处了表
                            promoteTestDOListAll = advertNewPromoteTestDao.findNewPromoteUrlsByAdvertId(advertId);
                            if(CollectionUtils.isNotEmpty(promoteTestDOListAll)){
                                isNewType=true;
                            }
                        }



                        //还有消息通知
                        Set<Long> advertiserUnAuditPromoteUrlIds = getAdvertiserUnAuditPromoteUrlIdByAdvertId(advertId);
                        List<AdvertPromoteTestDO> promoteTestDOList = promoteTestDOListAll.stream().
                                filter(promoteTestDOListOne->!advertiserUnAuditPromoteUrlIds.contains(promoteTestDOListOne.getId())).
                                collect(Collectors.toList());


                        if (CollectionUtils.isEmpty(promoteTestDOList)) {
                            return Optional.empty();
                        }

                        //收集落地页
                        List<String> promoteUrlsList = promoteTestDOList.stream().map(AdvertPromoteTestDO::getPromoteUrl).collect(Collectors.toList());

                        MaterialPromoteTestUrlsVO vo = new MaterialPromoteTestUrlsVO();
                        vo.setPromoteUrls(promoteUrlsList);
                        vo.setWeight(promoteTestDOList.get(0).getWeight());
                        vo.setOrientIds(promoteTestDOList.get(0).getOrientIds());
                        vo.setNewType(isNewType);
                        return Optional.ofNullable(vo);
                    } catch (Exception ex) {
                        logger.error("AdvertPromoteTestCacheService.ADVERT_PROMOTE_TEST_CACHE is error", ex);
                        return Optional.empty();
                    }
                }
            });


    //有消息同步，不用设置过期时间来保证避免读取旧值
    private LoadingCache<Long, List<AdvertPromoteTestDO>> PROMOTE_MATERIAL_TEST_CACHE = CacheBuilder.newBuilder()
            .initialCapacity(1000).refreshAfterWrite(1, TimeUnit.HOURS)
            .build(new CacheLoader<Long, List<AdvertPromoteTestDO>>() {
                @Override
                public List<AdvertPromoteTestDO> load(Long materialId) {
                    try {

                        List<AdvertPromoteTestDO> promoteTestDOList = advertPromoteTestDAO.findPromoteUrlsByMaterialId(materialId);

                        if (CollectionUtils.isEmpty(promoteTestDOList)) {
                            //如果老的是空的 从新的里查
                            // 因为tb_advert_promote_test_bind_abtest表里 同一个广告的enable_state=1的只会有一条数据,(新增的时候代码层面作了校验)
                            // 所以不存在一个素材id会查询到多个实验计划的情况,因为只能有个落地页测试计划开启
                            promoteTestDOList = advertNewPromoteTestDao.findNewPromoteTestUrlsByMaterialId(materialId);
                        }

                        if(CollectionUtils.isNotEmpty(promoteTestDOList)){
                            AdvertPromoteTestDO advertPromoteTestDO = promoteTestDOList.get(0);
                            Long advertId = advertPromoteTestDO.getAdvertId();
                            Set<Long> advertiserUnAuditPromoteUrlIds = getAdvertiserUnAuditPromoteUrlIdByAdvertId(advertId);
                            promoteTestDOList = promoteTestDOList.stream().
                                    filter(promoteTestDOListOne->!advertiserUnAuditPromoteUrlIds.contains(promoteTestDOListOne.getId())).
                                    collect(Collectors.toList());
                        }

                        return Optional.ofNullable(promoteTestDOList).orElse(Lists.newArrayList());
                    } catch (Exception ex) {
                        logger.error("AdvertPromoteTestCacheService.PROMOTE_MATERIAL_TEST_CACHE is error", ex);
                        return Lists.newArrayList();
                    }
                }
            });

    private LoadingCache<Long, Set<String>> MATERIAL_PROMOTE_CACHE = CacheBuilder.newBuilder()
            .refreshAfterWrite(30, TimeUnit.MINUTES)
            .expireAfterWrite(1, TimeUnit.HOURS)
            .build(new CacheLoader<Long, Set<String>>() {
                @Override
                public Set<String> load(Long materialId) {
                    try {

                        List<String> promoteUrls = advertPromoteTestDAO.selectPromoteUrlsByMaterialId(materialId);
                        return Optional.ofNullable(promoteUrls).map(list -> new HashSet<>(promoteUrls)).orElse(new HashSet<>(0));
                    } catch (Exception ex) {
                        logger.error("AdvertPromoteTestCacheService.PROMOTE_MATERIAL_TEST_CACHE is error", ex);
                        return Collections.emptySet();
                    }
                }

                @Override
                public ListenableFuture<Set<String>> reload(Long key, Set<String> oldValue) throws Exception {
                    ListenableFutureTask<Set<String>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });

    private LoadingCache<Long, Set<Long>> PROMOTE_REPO_CACHE = CacheBuilder.newBuilder()
            .refreshAfterWrite(30, TimeUnit.MINUTES)
            .expireAfterWrite(1, TimeUnit.HOURS)
            .build(new CacheLoader<Long, Set<Long>>() {
                @Override
                public Set<Long> load(Long advertId) {
                    try {

                        List<Long> promoteIds = advertPromoteTestDAO.selectLandPageRepoIdsByAdvertId(advertId);

                        //TODO 别忘记发消息
                        Set<Long> advertiserUnAuditPromoteUrlIds = getAdvertiserUnAuditPromoteUrlIdByAdvertId(advertId);
                        promoteIds.removeAll(advertiserUnAuditPromoteUrlIds);

                        return Optional.ofNullable(promoteIds).map(list -> new HashSet<>(promoteIds)).orElse(new HashSet<>(0));
                    } catch (Exception ex) {
                        logger.error("AdvertPromoteTestCacheService.PROMOTE_REPO_CACHE is error", ex);
                        return Collections.emptySet();
                    }
                }

                @Override
                public ListenableFuture<Set<Long>> reload(Long key, Set<Long> oldValue) throws Exception {
                    ListenableFutureTask<Set<Long>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });


    /**
     * 获取广告对应的测试中的测试落地页
     *
     * @param advertId
     * @return
     */
    public MaterialPromoteTestUrlsVO getMaterialPromoteTest(Long advertId) {

        try {
            return ADVERT_PROMOTE_TEST_CACHE.get(advertId).orElse(null);
        } catch (Exception e) {
            logger.error("AdvertPromoteTestCacheService.getMaterialPromoteTest is error, advertId:{}", advertId, e);
            return null;
        }
    }

    /**
     * 获取素材对应的测试中的测试落地页关系
     *
     * @param materialId
     * @return
     */
    public List<AdvertPromoteTestDO> getMaterialPromoteTestRelation(Long materialId) {

        try {
            return PROMOTE_MATERIAL_TEST_CACHE.get(materialId);
        } catch (Exception e) {
            logger.error("AdvertPromoteTestCacheService.getMaterialPromoteTest is error, materialId:{}", materialId, e);
            return Lists.newArrayList();
        }
    }

    /**
     * 获取素材关联的落地页库的测试落地页
     *
     * @param materialId 素材Id
     * @return 落地页集合
     */
    public Set<String> getMaterialPromoteTestSet(Long materialId) {
        try {
            return MATERIAL_PROMOTE_CACHE.get(materialId);
        } catch (Exception e) {
            logger.error("AdvertPromoteTestCacheService.getMaterialPromoteTestSet is error, materialId:{}", materialId, e);
            return Collections.emptySet();
        }
    }

    /**
     * 查询广告对应的落地页库的有效落地页ID集合
     *
     * @param advertId 广告Id
     * @return 落地页Id集合
     */
    public Set<Long> getLandPageRepoSet(Long advertId) {
        try {
            return PROMOTE_REPO_CACHE.get(advertId);
        } catch (Exception e) {
            logger.error("AdvertPromoteTestCacheService.getLandPageRepoSet is error, advertId:{}", advertId, e);
            return Collections.emptySet();
        }
    }

    /**
     * 刷新缓存
     *
     * @param advertId
     * @param materialId
     */
    public void updateMaterialPromoteTest(Long materialId, Long advertId) {

        PROMOTE_MATERIAL_TEST_CACHE.refresh(materialId);

        ADVERT_PROMOTE_TEST_CACHE.refresh(advertId);

        MATERIAL_PROMOTE_CACHE.refresh(materialId);

        PROMOTE_REPO_CACHE.refresh(advertId);
    }

    /**
     * 刷新缓存
     *
     * @param advertId
     */
    public void refreshPromoteCache(Long advertId) {

        ADVERT_PROMOTE_TEST_CACHE.refresh(advertId);

        PROMOTE_REPO_CACHE.refresh(advertId);
    }


    public void refreshMaterialCache(Long materialId){
        PROMOTE_MATERIAL_TEST_CACHE.refresh(materialId);
    }

}
