/**
 * Project Name:engine-service File Name:TestPersentCacheService.java Package Name:cn.com.duiba.tuia.cache
 * Date:2018年7月11日下午4:50:44 Copyright (c) 2018, duiba.com.cn All Rights Reserved.
 */

package cn.com.duiba.tuia.cache;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;

import cn.com.duiba.tuia.dao.engine.AdvertRepeatLunchConfigDAO;
import cn.com.duiba.tuia.domain.dataobject.AdvertRepeatLunchConfigDO;
import cn.com.tuia.advert.enums.ResourceRepeatLunchTypeEnum;

/**
 * 
 * ClassName: TestPersentCacheService <br/>
 * Function: 测试比例或者切量占比等测试缓存. <br/>
 * date: 2018年8月3日 上午11:11:46 <br/>
 *
 * @author chencheng
 * @version 
 * @since JDK 1.8
 */
@Component
public class AdvertRepeatLunchConfigCacheService extends BaseCacheService {


    @Resource
    private ExecutorService       executorService;

    @Autowired
    private AdvertRepeatLunchConfigDAO advertRepeatLunchConfigDAO;

    /**
     * 用来限制重复发券的特殊资源名称的缓存值
     */
    private final LoadingCache<Integer, Map<String, AdvertRepeatLunchConfigDO>> repeatLunchConfigResourceSourceTypeCache = CacheBuilder.newBuilder()
            .initialCapacity(1000).expireAfterWrite(30,TimeUnit.MINUTES).refreshAfterWrite(5, TimeUnit. MINUTES)
            .build(new CacheLoader<Integer, Map<String, AdvertRepeatLunchConfigDO>>() {
                @Override
                public Map<String, AdvertRepeatLunchConfigDO> load(Integer key) throws Exception {
                    return getAdvertRepeatLunchConfigByType(key);
                }

                @Override
                public ListenableFuture<Map<String, AdvertRepeatLunchConfigDO>> reload(Integer key, Map<String, AdvertRepeatLunchConfigDO> oldValue) throws Exception {
                    ListenableFutureTask<Map<String, AdvertRepeatLunchConfigDO>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
                
            });

    /**
     * 
     * getAdvertRepeatLunchConfigByType:(根据过滤类型来聚合资源过滤配置). <br/>
     *
     * @author chencheng
     * @param key
     * @return
     * @since JDK 1.8
     */
    private Map<String, AdvertRepeatLunchConfigDO> getAdvertRepeatLunchConfigByType(Integer key) {
        ResourceRepeatLunchTypeEnum type = ResourceRepeatLunchTypeEnum.groupType(key);
        List<AdvertRepeatLunchConfigDO> list;
        switch (type) {
            case IOS_DEFULT:
                list = getPlatformTypeConfig(ResourceRepeatLunchTypeEnum.IOS_DEFULT.getCode(), ResourceRepeatLunchTypeEnum.IOS_SPECIAL.getCode());
                break;
            case UN_IOS_DEFULT:
                list = getPlatformTypeConfig(ResourceRepeatLunchTypeEnum.UN_IOS_DEFULT.getCode(), ResourceRepeatLunchTypeEnum.UN_IOS_SPECIAL.getCode());
                break;
            case TRADE_APP_DEFULT:
                list = getPlatformTypeConfig(ResourceRepeatLunchTypeEnum.TRADE_APP_DEFULT.getCode(), ResourceRepeatLunchTypeEnum.TRADE_APP_SPECIAL.getCode());
                break;
            default:
                list = advertRepeatLunchConfigDAO.selectSpecialCongfigBySourceType(key);
                break;
        }
        return list.stream().filter(Objects::nonNull).collect(Collectors.toMap(AdvertRepeatLunchConfigDO::getSourceId, dto -> dto, (oldVal, newVal) -> newVal));
    }
    
    
    /**
     * 
     * getAdvertRepeatLunchConfigResourceName:(获取重复发券间隔配置). <br/>
     *
     * @author chencheng
     * @return
     * @since JDK 1.8
     */
    public Map<String, AdvertRepeatLunchConfigDO> getAdvertRepeatLunchConfigCache(Integer key) {
        try {
            // 获取特殊资源配置
            Map<String, AdvertRepeatLunchConfigDO> resourceConfig = repeatLunchConfigResourceSourceTypeCache.getIfPresent(key);
            if (resourceConfig == null) {
                invalidateAll();
                return Maps.newHashMap();
            }else {
                return resourceConfig;
            }
        } catch (Exception e) {
            logger.info("getAdvertRepeatLunchConfigCache error, key:{}", key, e);
            return Maps.newHashMap();
        }
    }
    
    /**
     * 
     * getAdvertRepeatLunchConfigResourceName:(获取特殊资源的名称). <br/>
     *
     * @author chencheng
     * @return
     * @since JDK 1.8
     */
    public Set<String> getAdvertRepeatLunchConfigResourceName() {
      try {
          // 获取特殊资源配置
          Map<String, AdvertRepeatLunchConfigDO> resourceConfig = repeatLunchConfigResourceSourceTypeCache.getIfPresent(ResourceRepeatLunchTypeEnum.IOS_SPECIAL.getCode());
          if (resourceConfig == null) {
              invalidateAll();
              return Sets.newHashSet();
          }else {
              return resourceConfig.keySet();
          }
      } catch (Exception e) {
          logger.info("getAdvertRepeatLunchConfigResourceName error", e);
          return Sets.newHashSet();
      }
   }
        
    /**
     * 
     * invalidateAll:(缓存初始化). <br/>
     *
     * @author chencheng
     * @since JDK 1.8
     */
    public void invalidateAll() {
        repeatLunchConfigResourceSourceTypeCache.refresh(ResourceRepeatLunchTypeEnum.IOS_DEFULT.getCode());
        repeatLunchConfigResourceSourceTypeCache.refresh(ResourceRepeatLunchTypeEnum.UN_IOS_DEFULT.getCode());
        repeatLunchConfigResourceSourceTypeCache.refresh(ResourceRepeatLunchTypeEnum.TRADE_APP_DEFULT.getCode());
    }


    /**
     * 
     * getPlatformTypeConfig:(根据平台不同的type获取所有的类型). <br/>
     *
     * @author chencheng
     * @param defultType
     * @param specialType
     * @return
     * @throws ExecutionException
     * @since JDK 1.8
     */
    private List<AdvertRepeatLunchConfigDO> getPlatformTypeConfig(Integer defultType, Integer specialType){
        
        List<AdvertRepeatLunchConfigDO>  listAdvertRepeatLunchConfig = Lists.newArrayList();
        // 获取默认资源配置
        List<AdvertRepeatLunchConfigDO> defult = advertRepeatLunchConfigDAO.selectSpecialCongfigBySourceType(defultType);
        listAdvertRepeatLunchConfig.addAll(defult);
        // 获取特殊资源配置
        List<AdvertRepeatLunchConfigDO> special = advertRepeatLunchConfigDAO.selectSpecialCongfigBySourceType(specialType);
        listAdvertRepeatLunchConfig.addAll(special);
        return listAdvertRepeatLunchConfig;
    }
   

    /**
     * 
     * refresh:(刷新缓存). <br/>
     *
     * @author chencheng
     * @since JDK 1.8
     */
    public void refresh(Integer sourceType){
        if (null == sourceType) {
            return;
        }
        // 平台聚合
        ResourceRepeatLunchTypeEnum type = ResourceRepeatLunchTypeEnum.groupType(sourceType);
        repeatLunchConfigResourceSourceTypeCache.refresh(type.getCode());
    }
    
}
