package cn.com.duiba.tuia.cache;

import cn.com.duiba.tuia.dao.advert.AdvertSimilarTradeDao;
import cn.com.duiba.tuia.domain.dataobject.AdvertSimilarTradeDO;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * Created by huangch on 2018/6/1 11:18
 * description:
 *
 * @since JDK 1.6
 */
@Component
public class AdvertSimilarTradeCache {

    @Resource
    private ExecutorService executorService;

    private static final Long KEY = 1L;

    @Resource
    private AdvertSimilarTradeDao advertSimilarTradeDao;

    //行业对应的相似广告缓存
    private final LoadingCache<Long, Map<Long, List<AdvertSimilarTradeDO>>> TRADE_SIMILAR_ADVERT_CACHE = CacheBuilder.newBuilder().initialCapacity(100).
            recordStats().refreshAfterWrite(50, TimeUnit.MINUTES).expireAfterWrite(1, TimeUnit.HOURS).build(new CacheLoader<Long, Map<Long, List<AdvertSimilarTradeDO>>>() {
        @Override
        public Map<Long, List<AdvertSimilarTradeDO>> load(Long key) {

            List<AdvertSimilarTradeDO> advertSimilarTradeList = getAdvertSimilarTradeList();
            if (CollectionUtils.isEmpty(advertSimilarTradeList)) {
                return Collections.emptyMap();
            }
            return advertSimilarTradeList.stream().collect(Collectors.groupingBy(AdvertSimilarTradeDO::getAdvertId));
        }

        @Override
        public ListenableFuture<Map<Long, List<AdvertSimilarTradeDO>>> reload(final Long key, Map<Long, List<AdvertSimilarTradeDO>> oldValue) {
            ListenableFutureTask<Map<Long, List<AdvertSimilarTradeDO>>> task = ListenableFutureTask.create(() -> load(key));
            executorService.submit(task);
            return task;
        }
    });

    private List<AdvertSimilarTradeDO> getAdvertSimilarTradeList() {
        return advertSimilarTradeDao.selectLatestAdvertSimilarTradeList();
    }

    public Map<Integer, List<AdvertSimilarTradeDO>> getTradeSimilarAdvert(List<Long> consumeClickAdvertList) {
        if (CollectionUtils.isEmpty(consumeClickAdvertList)) {
            return Collections.emptyMap();
        }
        //获取所有的广告对应相似行业数据
        Map<Long, List<AdvertSimilarTradeDO>> advertSimilarTradeMap = TRADE_SIMILAR_ADVERT_CACHE.getUnchecked(KEY);
        if (org.springframework.util.CollectionUtils.isEmpty(advertSimilarTradeMap)) {
            return Collections.emptyMap();
        }
        List<AdvertSimilarTradeDO> advertSimilarTradeList = Lists.newArrayList();
        for (Long advertId : consumeClickAdvertList) {
            List<AdvertSimilarTradeDO> similarAdvertMap = advertSimilarTradeMap.get(advertId);
            if (org.springframework.util.CollectionUtils.isEmpty(similarAdvertMap)) {
                continue;
            }
            advertSimilarTradeList.addAll(similarAdvertMap);
        }
        if (CollectionUtils.isEmpty(advertSimilarTradeList)) {
            return Collections.emptyMap();
        }
        return advertSimilarTradeList.stream().collect(Collectors.groupingBy(AdvertSimilarTradeDO::getMark));
    }
}