package cn.com.duiba.tuia.cache;

import cn.com.duiba.nezha.engine.api.dto.SupportAdvertInfoDto;
import cn.com.duiba.tuia.dao.advert.AdvertSupportPlanDAO;
import cn.com.duiba.tuia.domain.dataobject.AdvertSupportHoursDto;
import cn.com.duiba.tuia.domain.dataobject.AdvertSupportLaunchDO;
import cn.com.duiba.tuia.domain.dataobject.AdvertSupportPlanDO;
import cn.com.duiba.tuia.message.rocketmq.AdvertSupportPlanRocketMqProducer;
import cn.com.duiba.tuia.pangea.center.api.localservice.apollopangu.ApolloPanGuService;
import cn.com.duiba.tuia.tool.CatUtil;
import cn.com.duiba.wolf.utils.DateUtils;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.lang3.tuple.MutablePair;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * Created by ZhangShun at 2020/5/11
 */
@Service
public class AdvertSupportPlanCacheManager extends BaseCacheService {

    private static final Integer CACHE_KEY_SUPPORT = 1;

    @Resource
    private ExecutorService executorService;
    @Resource
    private StringRedisTemplate stringRedisTemplate;
    @Resource
    private AdvertSupportPlanDAO advertSupportPlanDAO;
    @Resource
    private ApolloPanGuService apolloPanGuService;
    @Resource
    private AdvertSupportPlanRocketMqProducer advertSupportPlanRocketMqProducer;

    private static final String KEY_UPPER_DAY_SUM = "tuia-engine.advert.support.upper.day.sum";
    private static final long DEFUALT_UPPER_DAY_SUM = 20_0000L;

    /**
     * 生效的扶持计划缓存
     */
    private final LoadingCache<Integer, Map<Long, AdvertSupportPlanDO>> advertListSupportPlanCache = CacheBuilder.newBuilder()
            .maximumSize(100)
            .expireAfterAccess(6, TimeUnit.HOURS)
            .refreshAfterWrite(1, TimeUnit.HOURS)
            .build(new CacheLoader<Integer, Map<Long, AdvertSupportPlanDO>>() {
                @Override
                public Map<Long, AdvertSupportPlanDO> load(Integer key) throws Exception {
                    List<AdvertSupportPlanDO> dtoList = advertSupportPlanDAO.findValidSupportPlan();
                    if (CollectionUtils.isEmpty(dtoList)) {
                        return Collections.emptyMap();
                    }
                    Map<Long, AdvertSupportPlanDO> map = Maps.newHashMap();
                    dtoList.forEach(item -> map.put(item.getAdvertId(), item));
                    return map;
                }
                @Override
                public ListenableFuture<Map<Long, AdvertSupportPlanDO>> reload(Integer key, Map<Long, AdvertSupportPlanDO> oldValue) throws Exception {
                    ListenableFutureTask<Map<Long, AdvertSupportPlanDO>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });

    private Map<Long, AdvertSupportPlanDO> getCacheAdvertSupportMap() {
        return Optional.ofNullable(advertListSupportPlanCache.getUnchecked(CACHE_KEY_SUPPORT)).orElse(Collections.emptyMap());
    }

    private List<Long> getCacheAdvertIdList() {
        Map<Long, AdvertSupportPlanDO> map = getCacheAdvertSupportMap();
        if (MapUtils.isEmpty(map)) {
            return Collections.emptyList();
        }
        return Lists.newArrayList(map.keySet());
    }

    private Long getDefaultUpperDaySum() {
        try {
            String upperDaySumStr = apolloPanGuService.getIdMapStrByKeyStr(KEY_UPPER_DAY_SUM);
            if (StringUtils.isNotBlank(upperDaySumStr)) {
                return Long.parseLong(upperDaySumStr);
            }
        } catch (Exception e) {
            logger.warn("广告扶持获取大禹配置异常 apolloKey={}", KEY_UPPER_DAY_SUM, e);
        }
        return DEFUALT_UPPER_DAY_SUM;
    }

    /**
     * 判断广告是否在缓存中
     * @param advertId
     * @return
     */
    private boolean notExistAdvertSupportPlan(Long advertId) {
        return !getCacheAdvertSupportMap().containsKey(advertId);
    }

    /**
     * 广告扶持缓存  advertId -> AdvertSupportPlanDO
     */
    private final LoadingCache<Long, Map<Long, SupportAdvertInfoDto>> advertSupportPlanLaunchCache = CacheBuilder.newBuilder()
            .maximumSize(3000)
            .expireAfterAccess(10, TimeUnit.SECONDS)
            .refreshAfterWrite(5, TimeUnit.SECONDS)
            .build(new CacheLoader<Long, Map<Long, SupportAdvertInfoDto>>() {
                @Override
                public Map<Long, SupportAdvertInfoDto> load(Long key) throws Exception {
                    return doGetSupportPlanLaunch(key);
                }
                @Override
                public ListenableFuture<Map<Long, SupportAdvertInfoDto>> reload(Long key, Map<Long, SupportAdvertInfoDto> oldValue) throws Exception {
                    ListenableFutureTask<Map<Long, SupportAdvertInfoDto>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });

    public SupportAdvertInfoDto getSupportPlanByAdvertId(Long advertId, Map<Long, SupportAdvertInfoDto> launchMap) {
        try {
            // 不是扶持的广告,直接返回
            if (notExistAdvertSupportPlan(advertId)) {
                return null;
            }

            // 获取广告扶持计划配置信息
            AdvertSupportPlanDO planDO = getCacheAdvertSupportMap().getOrDefault(advertId, null);
            if (Objects.isNull(planDO)) {
                return null;
            }

            // 设置当前发券
            SupportAdvertInfoDto currentlaunchDto = launchMap.getOrDefault(advertId, new SupportAdvertInfoDto());
            Boolean isSupport = isValidSupportPlan(advertId, planDO, currentlaunchDto);

            if (!isSupport) {
                return null;
            }

            SupportAdvertInfoDto dto = new SupportAdvertInfoDto();
            dto.setSupportPlanId(planDO.getId());
            dto.setPlanAdvertUpper(getConfigUpper(planDO.getUpperDayAdvert(), planDO.getUpperDaySum())); // 每日单广告发券量，上限
            dto.setPlanAppUpper(getConfigUpper(planDO.getUpperDayMedia(), planDO.getUpperDaySum())); // 每日单媒体发券量，上限
            dto.setPlanPriceStart(planDO.getPriceStart()); // 出价扶持阈值：低值
            dto.setPlanPriceEnd(planDO.getPriceEnd()); // 出价扶持阈值：高值
            dto.setPlanCvrType(planDO.getTargetType()); // 扶持目标类型:1=落地页转化、2=启动、3=注册
            dto.setPlanTargetCost(planDO.getTargetCost()); // 扶持目标成本
            dto.setPlanAppCur(getPlanAppCur(planDO.getUpperDayMedia(), currentlaunchDto.getPlanAppCur(), currentlaunchDto.getPlanSumCur()));
            dto.setPlanAdvertCur(getPlanAppCur(planDO.getUpperDayAdvert(), currentlaunchDto.getPlanAdvertCur(), currentlaunchDto.getPlanSumCur()));
            // 设置是否需要扶持
            dto.setSupportAdvert(Boolean.TRUE);
            //设置是否AB分流 开关
            dto.setIsAbShuntSwitch(planDO.getAbShunt() != null && planDO.getAbShunt() == 1);

            return dto;
        } catch (Exception e) {
            return null;
        }
    }

    private Long getConfigUpper(Long upper, Long defaultUpper) {
        // 不是不限，取当前值
        if (upper != null && upper != -1L) {
            return upper;
        }
        // 总发券量不是不限，直接去总发券量
        if (defaultUpper != null && defaultUpper != -1L) {
            return defaultUpper;
        }
        // 总发券量不限，去配置
        return getDefaultUpperDaySum();
    }

    private Long getPlanAppCur(Long upper, Long cur, Long sumCur) {
        // 不是不限，取当前值
        if (upper != null && upper != -1L) {
            return cur;
        }
        // 不限，取总发券当前值
        return sumCur;
    }

    /**
     * 是否扶持
     * @param plan
     * @param currentlaunchDto
     * @return true = 需要扶持
     */
    private Boolean isValidSupportPlan(Long advertId, AdvertSupportPlanDO plan, SupportAdvertInfoDto currentlaunchDto) {
        try {
            return CatUtil.executeInCatTransaction(() -> handlerValidSupportPlan(advertId, plan, currentlaunchDto),
                    "memoryFilter", "handlerAdvertSupportPlan");
        } catch (Throwable throwable) {
            logger.info("判断广告扶持状态错误 {}", advertId, throwable);
            return false;
        }
    }

    /**
     *
     * @param advertId
     * @param plan
     * @param currentlaunchDto
     * @return
     */
    private Boolean handlerValidSupportPlan(Long advertId, AdvertSupportPlanDO plan, SupportAdvertInfoDto currentlaunchDto) {
        // 是否在配置时间内
        Long curTime = System.currentTimeMillis();
        if (!(curTime > plan.getDateStart().getTime() && curTime < plan.getDateEnd().getTime())) {
            return false;
        }
        // 配置时间段为不限
        List<AdvertSupportHoursDto> hourList = getAdvertSupportHoursDto(advertId, plan.getHourPeriod());
        if (CollectionUtils.isEmpty(hourList)) {
            return isValidSupportPlan(plan, currentlaunchDto);
        }
        // 是否在配置时间段内
        Integer curHour = Calendar.getInstance().get(Calendar.HOUR_OF_DAY);
        for (AdvertSupportHoursDto hour : hourList) {
            if (Objects.nonNull(hour.getHourStart()) && Objects.nonNull(hour.getHourEnd())
                    && curHour >= hour.getHourStart() && curHour < hour.getHourEnd()) {
                return isValidSupportPlan(plan, currentlaunchDto);
            }
        }

        return false;
    }

    private Boolean isValidSupportPlan(AdvertSupportPlanDO plan, SupportAdvertInfoDto currentlaunchDto) {
        if (!Objects.equals(-1L, plan.getUpperDaySum()) && currentlaunchDto.getPlanSumCur() > plan.getUpperDaySum()) {
            advertSupportPlanRocketMqProducer.sendMsg(plan.getId());
            return false;
        }

        if ((Objects.equals(-1L, plan.getUpperDayMedia()) || currentlaunchDto.getPlanAppCur() < plan.getUpperDayMedia()) &&
            (Objects.equals(-1L, plan.getUpperDayAdvert()) || currentlaunchDto.getPlanAdvertCur() < plan.getUpperDayAdvert()) &&
                (Objects.equals(-1L, plan.getUpperDaySum()) || currentlaunchDto.getPlanSumCur() < plan.getUpperDaySum()) )  {
            return true;
        }
        return false;
    }

    private List<AdvertSupportHoursDto> getAdvertSupportHoursDto(Long advertId, String hourPeriodStr) {
        try {
            return JSONArray.parseArray(hourPeriodStr, AdvertSupportHoursDto.class);
        } catch (Exception e) {
            logger.warn("解析广告扶持时间段信息错误 {} str={}", advertId, hourPeriodStr, e);
            return Collections.emptyList();
        }
    }

    /**
     * 查询广告每日的发券量
     */
    public Map<Long, SupportAdvertInfoDto> getCacheSupportPlanLaunch(Long appId) {
        try {
            return advertSupportPlanLaunchCache.getUnchecked(appId);
        } catch (Exception e) {
            return Collections.emptyMap();
        }
    }

    private Map<Long, SupportAdvertInfoDto> doGetSupportPlanLaunch(Long appId) {
        // 查询所有生效的扶持计划
        Map<Long, AdvertSupportPlanDO> cacheAdvertSupportMap = getCacheAdvertSupportMap();
        if (MapUtils.isEmpty(cacheAdvertSupportMap)) {
            return Collections.emptyMap();
        }

        List<Long> advertIds = Lists.newArrayList(cacheAdvertSupportMap.keySet());

        // 从redis 获取广告扶持发券量
        MutablePair<List<String>, List<String>> supportLaunch = doGetSupportPlanLaunch(appId, advertIds);
        Map<Long, Long> planSumCurMap = calculatePlanSumCurMap(advertIds, cacheAdvertSupportMap);

        Map<Long, SupportAdvertInfoDto> resMap = Maps.newHashMap();

        for (int i = 0; i < advertIds.size(); i++) {
            Long advertId = advertIds.get(i);
            try {

                SupportAdvertInfoDto dto = new SupportAdvertInfoDto();

                // 获取 advert launch
                AdvertSupportLaunchDO advertLaunch = doGetSupportPlanLaunch(advertId, supportLaunch.getLeft().get(i));
                dto.setPlanAdvertCur(Objects.isNull(advertLaunch.getLaunchCnt()) ? 0L : advertLaunch.getLaunchCnt()); // 每日单广告发券量，当前值

                // 获取 media launch
                AdvertSupportLaunchDO mediaLaunch = doGetSupportPlanLaunch(advertId, supportLaunch.getRight().get(i));
                dto.setPlanAppCur(Objects.isNull(mediaLaunch.getLaunchCnt()) ? 0L : mediaLaunch.getLaunchCnt()); // 每日单媒体发券量，当前值

                // 计算总扶持发券上限
                dto.setPlanSumCur(planSumCurMap.getOrDefault(advertId, 0L)); // 总扶持发券上限，当前值

                resMap.put(advertId, dto);
            } catch (Exception e) {
                logger.warn("获取广告扶持发券量错误 {}", advertId, e);
            }
        }

        return resMap;
    }

    public MutablePair<List<String>, List<String>> doGetSupportPlanLaunch(Long appId, List<Long> advertIds) {

        List<String> advertValues = null;
        List<String> mediaValues = null;
        List<String> sumValues = null;

        try {
            // 获取每日单广告发券量
            advertValues = stringRedisTemplate.opsForValue().multiGet(buildRedisKeys(null, advertIds ));
            // 获取每日单媒体发券量
            mediaValues = stringRedisTemplate.opsForValue().multiGet(buildRedisKeys(appId, advertIds));
        } catch (Exception e) {
            logger.warn("获取广告扶持发券量错误", e);
        }

        if (null == advertValues) {
            advertValues = Collections.emptyList();
        }

        if (null == mediaValues) {
            mediaValues = Collections.emptyList();
        }

        if (null == sumValues) {
            sumValues = Collections.emptyList();
        }

        return new MutablePair(advertValues, mediaValues);
    }

    private AdvertSupportLaunchDO doGetSupportPlanLaunch(Long advertId, String jsonStr) {
        if (StringUtils.isBlank(jsonStr)) {
            return new AdvertSupportLaunchDO();
        }
        try {
            return JSONObject.parseObject(jsonStr, AdvertSupportLaunchDO.class);
        } catch (Exception e) {
            logger.warn("获取广告扶持发券数据错误 {}, jsonStr={}", advertId, jsonStr, e);
            return new AdvertSupportLaunchDO();
        }
    }

    private List<String> buildRedisKeys(Long appId, List<Long> advertIds) {
        String keyPrefix = RedisKeys.K33.toString();
        String date = DateFormatUtils.format(System.currentTimeMillis(), "yyyyMMdd");

        List<String> keys = new ArrayList<>(advertIds.size());
        if (Objects.isNull(appId)) {
            advertIds.forEach(advertId -> {
                keys.add(keyPrefix + "_" + advertId + "_" + date);
            });
        } else {
            advertIds.forEach(advertId -> {
                keys.add(keyPrefix + "_" + advertId + "_" + appId + "_" + date);
            });
        }
        return keys;
    }

    /**
     * 统计计算扶持周期内的总发券量
     *
     * @param advertIds
     * @param cacheAdvertSupportMap
     * @return
     */
    private Map<Long, Long> calculatePlanSumCurMap(List<Long> advertIds, Map<Long, AdvertSupportPlanDO> cacheAdvertSupportMap) {
        try {
            String keyPrefix = RedisKeys.K33.toString();
            Date date = new Date();
            Map<Long, List<String>> keysMap = new HashMap<>();
            List<String> allKeys = new ArrayList<>(advertIds.size());
            for (Long advertId : advertIds) {
                AdvertSupportPlanDO advertSupportPlanDO = cacheAdvertSupportMap.get(advertId);
                Date dateStart = advertSupportPlanDO.getDateStart();
                List<String> advertKeys = new ArrayList<>();
                int limit = 0;
                // 拼接扶持生效周期内的redisKey
                while (dateStart.before(date) && limit < 7) {
                    String key = keyPrefix + "_" + advertId + "_" + DateUtils.getDayNumber(dateStart);
                    allKeys.add(key);
                    advertKeys.add(key);
                    dateStart = DateUtils.daysAddOrSub(dateStart, 1);
                    limit ++;
                }
                keysMap.put(advertId, advertKeys);
            }

            // 批量查询redisKey，返回结果组装成key-value形式
            Map<String, String> rediskeyValueMap = new HashMap<>();
            List<String> redisValues = stringRedisTemplate.opsForValue().multiGet(allKeys);
            for (int i = 0; i < allKeys.size(); i++) {
                String redisKey = allKeys.get(i);
                String redisValue = redisValues.get(i);
                rediskeyValueMap.put(redisKey, redisValue);
            }

            // 统计计算扶持周期内的发券量
            Map<Long, Long> advertSumMap = new HashMap<>(keysMap.size());
            for (Entry<Long, List<String>> entry : keysMap.entrySet()) {
                Long advertId = entry.getKey();
                List<String> redisKeys = entry.getValue();
                if (CollectionUtils.isEmpty(redisKeys)) {
                    advertSumMap.put(advertId, 0L);
                } else {
                    long sum = redisKeys.stream().mapToLong(item -> {
                        String redisValue = rediskeyValueMap.get(item);
                        AdvertSupportLaunchDO advertLaunch = doGetSupportPlanLaunch(null, redisValue);
                        return Objects.isNull(advertLaunch.getLaunchCnt()) ? 0L : advertLaunch.getLaunchCnt();
                    }).sum();
                    advertSumMap.put(advertId, sum);
                }
            }
            return advertSumMap;
        } catch (Exception e) {
            logger.warn("统计广告扶持发券量错误", e);
        }
        return Collections.emptyMap();
    }

    /**
     * 刷新缓存
     */
    public void refresh() {
        advertListSupportPlanCache.refresh(CACHE_KEY_SUPPORT);
    }

}
