/**
 * Project Name:engine-service<br>
 * File Name:ConsumerCacheService.java<br>
 * Package Name:cn.com.duiba.tuia.cache<br>
 * Date:2016年11月25日下午2:22:47<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */

package cn.com.duiba.tuia.cache;

import java.util.*;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import cn.com.duiba.tuia.constants.AdvertConstants;
import cn.com.duiba.tuia.constants.CommonConstants;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;

import cn.com.duiba.tuia.constants.TrusteeshipConstants;
import cn.com.duiba.tuia.dao.advert.AdvertAppPackageDAO;
import cn.com.duiba.tuia.dao.engine.AdvertOrientationPackageDAO;
import cn.com.duiba.tuia.dao.engine.TargetAppRecommendTypeDAO;
import cn.com.duiba.tuia.dao.slot.AppPackageSlotDao;
import cn.com.duiba.tuia.dao.slot.OrientationAppSlotDAO;
import cn.com.duiba.tuia.domain.dataobject.AdvertOrientationPackageDO;
import cn.com.duiba.tuia.domain.dataobject.AppPackageSlotDO;
import cn.com.duiba.tuia.domain.dataobject.OrientationAppSlotDO;
import cn.com.duiba.tuia.domain.dataobject.TargetAppRecommendTypeDO;
import cn.com.duiba.tuia.domain.vo.AdvertTargetAppVO;
import cn.com.duiba.tuia.service.AdvertOrientationService;
import cn.com.duiba.tuia.service.AdvertTargetAppService;
import static java.util.stream.Collectors.groupingBy;

/**
 * 
 * ClassName: AdvertTargetAppCacheService <br/>
 * Function: 定向配置的定向媒体-广告位缓存. <br/>
 * date: 2018年7月11日 上午10:19:03 <br/>
 *
 * @author chencheng
 * @version 
 * @since JDK 1.8
 */
@Component
public class AdvertTargetAppCacheService extends BaseCacheService {

    //不限
    private static final String NOT_LIMIT_TYPE = "-1";
    
    private static final String APP_JOIN = "_";
    
    @Autowired
    private AdvertTargetAppService advertTargetAppService;
    
    @Resource
    private ExecutorService executorService;
    
    @Autowired
    private AdvertOrientationPackageDAO advertOrientationPackageDAO;

    @Autowired
    private TargetAppRecommendTypeDAO targetAppRecommendTypeDAO;

    @Autowired
    private AdvertOrientationService advertOrientationService;
    
    @Autowired
    private OrientationAppSlotDAO orientationAppSlotDAO;
    @Autowired
    private AdvertAppPackageDAO advertAppPackageDao;
    @Autowired
    private AppPackageSlotDao appPackageSlotDao;
    //有消息同步，不用设置过期时间来保证避免读取旧值
    private final LoadingCache<String, AdvertTargetAppVO> ADVERT_TARGET_APP_CACHE =
            CacheBuilder.newBuilder().initialCapacity(1000)
            .refreshAfterWrite(10, TimeUnit.MINUTES)
                    .build(new CacheLoader<String, AdvertTargetAppVO>() {
        @Override
        public AdvertTargetAppVO load(String key) throws Exception {
            List<String> strs= Splitter.on("|").splitToList(key);//advertId|orientId
            Long advertId=Long.parseLong(strs.get(0));
            Long orientId=Long.parseLong(strs.get(1));
            return getAdvertTargetApp(advertId, orientId);
        }
        @Override
        public ListenableFuture<AdvertTargetAppVO> reload(String  key, AdvertTargetAppVO oldValue) throws Exception {
            ListenableFutureTask<AdvertTargetAppVO> task = ListenableFutureTask.create(() -> load(key));
            executorService.submit(task);
            return task;
        }
    });
    
    /**
     * 
     * getAdvertTargetApp:(从数据库查询配置的定向媒体-广告位信息). <br/>
     *
     * @author chencheng
     * @param advertId
     * @param orientId
     * @return
     * @since JDK 1.8
     */
    private AdvertTargetAppVO getAdvertTargetApp(Long advertId, Long orientId) {
        // 默认定向配置原始id
        AdvertTargetAppVO advertTargetAppVO = new AdvertTargetAppVO();

        try {
            Long rawOrientationId = orientId;
            Integer targetAppLimit;
            if (orientId == 0) {
                AdvertOrientationPackageDO advertOrientationPackageDO = advertOrientationPackageDAO.selectDefaultByAdvertId(advertId);
                if (null == advertOrientationPackageDO) {
                    return advertTargetAppVO;
                }
                rawOrientationId  = advertOrientationPackageDO.getId();
                targetAppLimit = advertOrientationPackageDO.getTargetAppLimit();
            } else {
                AdvertOrientationPackageDO orientation = advertOrientationPackageDAO.selectById(orientId);
                if (orientation == null || orientation.getEnableStatus() == AdvertConstants.ENABLE_STATUS_NO) {
                    return advertTargetAppVO;
                }
                targetAppLimit = orientation.getTargetAppLimit();
            }

            //定向媒体对应的流量包
            Map<Long, Long> appTargetPackage = Maps.newHashMap();

            Map<Long, Long> slotTargetPackage = Maps.newHashMap();

            // 查询定向媒体
            Set<Long> targetApps = buildTargetAppIds(advertId, orientId, rawOrientationId, appTargetPackage);

            advertTargetAppVO.setTargetApps(targetApps);

            //查询定向配置设置的广告位
            List<OrientationAppSlotDO> orientationAppSlots = orientationAppSlotDAO.selectTargetByAdvertId(advertId, rawOrientationId);

            //查询定向配置绑定的流量媒体包列表
            List<Long> packageIdList = advertAppPackageDao.getAppPackageIdsByAdvertOrientationPackageId(rawOrientationId);

            //查询流量媒体包包绑定的广告位
            List<AppPackageSlotDO> packageSlotDOS = appPackageSlotDao.selectByPackageIds(packageIdList);
            Map<Long, List<AppPackageSlotDO>> packageSlotsMap = packageSlotDOS.stream().collect(groupingBy(AppPackageSlotDO::getPackageId));

            advertTargetAppVO.setTargetAppSlots(this.buildTargetAppSlot(packageIdList, packageSlotsMap, orientationAppSlots, targetApps, slotTargetPackage));

            advertTargetAppVO.setAppTargetPackage(appTargetPackage);
            advertTargetAppVO.setSlotTargetPackage(slotTargetPackage);
            advertTargetAppVO.setTargetRecommendType(getTargetRecommendTypeMap(rawOrientationId));
            return advertTargetAppVO;
        } catch (Exception e) {
            logger.error("get advert target app from db error", e);
            return advertTargetAppVO;
        }
    }

    private Map<Long, Integer> getTargetRecommendTypeMap(Long orientationId) {
        try {
            List<TargetAppRecommendTypeDO> targetAppRecommendTypeList = targetAppRecommendTypeDAO.selectOrientationRecommendAppType(orientationId);
            if (CollectionUtils.isEmpty(targetAppRecommendTypeList)) {
                return Collections.emptyMap();
            }
            return targetAppRecommendTypeList.stream().filter(dto -> null != dto && null != dto.getRecommendType())
                    .collect(Collectors.toMap(TargetAppRecommendTypeDO::getAppId,
                    TargetAppRecommendTypeDO::getRecommendType, (oldVal, newVal) -> newVal));

        } catch (Exception e) {
            logger.error("getTargetRecommendType exception,orientationId:{}", orientationId, e);
            return Collections.emptyMap();
        }
    }


    /**
    *
    * @param packageIdList 配置绑定的包列表
    * @param packageSlotsMap 包绑定的广告位列表
    * @param orientationAppSlots 定向配置绑定的广告位
    * @param targetApps 配置的定向媒体
    * @return
    */
   private Set<String> buildTargetAppSlot(List<Long> packageIdList,Map<Long,List<AppPackageSlotDO>> packageSlotsMap,
                                           List<OrientationAppSlotDO> orientationAppSlots, Set<Long> targetApps,
                                           Map<Long, Long> slotTargetPackage) {

       Set<String> appSlots = Sets.newHashSet();
       List<Long> alreadySetAppIds = Lists.newArrayList();
       List<Long> alreadySetSlotIds = Lists.newArrayList();
       orientationAppSlots = Optional.ofNullable(orientationAppSlots).orElse(Collections.emptyList());
       for (OrientationAppSlotDO orientationAppSlotDO : orientationAppSlots) {

           StringBuilder builder = new StringBuilder();
           builder.append(orientationAppSlotDO.getAppId()).append("_").append(orientationAppSlotDO.getSlotId());
           appSlots.add(builder.toString());
           alreadySetAppIds.add(orientationAppSlotDO.getAppId());
           alreadySetSlotIds.add(orientationAppSlotDO.getSlotId());
       }

       //设置流量包设置的广告位
       Optional.ofNullable(packageIdList).ifPresent(itemList -> itemList.forEach(packageId -> {
           //包设置的广告位列表
           List<AppPackageSlotDO> appPackageSlotDOS = packageSlotsMap.get(packageId);
           Optional.ofNullable(appPackageSlotDOS).ifPresent(appPackageSlotItemList ->
                   appPackageSlotItemList.forEach(appPackageSlotItem -> {

                       if (alreadySetSlotIds.contains(appPackageSlotItem.getSlotId())) {
                           return;
                       }

                       StringBuilder builder = new StringBuilder();
                       builder.append(appPackageSlotItem.getAppId()).append("_").append(appPackageSlotItem.getSlotId());
                       appSlots.add(builder.toString());
                       alreadySetAppIds.add(appPackageSlotItem.getAppId());
                       alreadySetSlotIds.add(appPackageSlotItem.getSlotId());
                       slotTargetPackage.put(appPackageSlotItem.getSlotId(), packageId);
                   }));

       }));

       for (Long appId : targetApps) {

           if (alreadySetAppIds.contains(appId)) {
               continue;
           }

           if (appId.equals(-1L)) {
               appSlots.add("-1");
               continue;
           }

           StringBuilder builder = new StringBuilder();
           builder.append(appId).append("_-1");
           appSlots.add(builder.toString());
       }

       return appSlots;
   }

    /**
     * 
     * buildTargetAppIds:(查询定向媒体). <br/>
     *
     * @author chencheng
     * @param advertId
     * @param orientationId
     * @param rawOrientationId
     * @return
     * @since JDK 1.8
     */
    private Set<Long> buildTargetAppIds(Long advertId, Long orientationId, Long rawOrientationId, Map<Long, Long> appTargetPackage) {
    
        Set<Long> allAppIds = advertTargetAppService.selectAllTargetApps(advertId, orientationId, rawOrientationId, appTargetPackage);
    
        // 如果没有媒体id,就认为是不限,设置-1
        if (CollectionUtils.isEmpty(allAppIds)) {
            allAppIds = CommonConstants.defaultLongList;
        }
        return allAppIds;
    }

    /**
     * 
     * invalidAdvertTargetAppCache:(更新缓存). <br/>
     *
     * @author chencheng
     * @param advertId
     * @param advertPackageId
     * @since JDK 1.8
     */
    public void invalidAdvertTargetAppCache(Long advertId, long advertPackageId) {
        
        ADVERT_TARGET_APP_CACHE.refresh(Joiner.on("|").join(advertId,advertPackageId));
        
    }

    /**
     * 
     * selectByAdvertIdAndPackageId:(获取配置定向媒体-广告位信息). <br/>
     *
     * @author chencheng
     * @param advertId
     * @param advertPackageId
     * @return
     * @since JDK 1.8
     */
    public AdvertTargetAppVO selectByAdvertIdAndPackageId(Long advertId, Long advertPackageId) {
        
        try {
            return ADVERT_TARGET_APP_CACHE.get(Joiner.on("|").join(advertId,advertPackageId));
        } catch (ExecutionException e) {
            logger.error("get advert_target_app_cache error", e);
            return getAdvertTargetApp(advertId, advertPackageId);
        }
    }

    public Map<String, AdvertTargetAppVO> getAllCachedAdvertTargetAppVO(Collection<String> advertPackageIds) {
        try {
            return ADVERT_TARGET_APP_CACHE.getAll(advertPackageIds);
        } catch (ExecutionException e) {
            logger.error("get advert_target_app_cache error", e);
        }
        return Collections.emptyMap();
    }
    
    /**
     * 
     * getTargetApp:(定向媒体不限). <br/>
     *
     * @author chencheng
     * @param appId
     * @param slotId
     * @return
     * @since JDK 1.8
     */
    public String getTargetApp(Long appId, Long slotId) {
        
        return new StringBuilder().append(appId).append(APP_JOIN).append(slotId).toString();
    }

    /**
     * 
     * getTargetApp:(定向广告位不限). <br/>
     *
     * @author chencheng
     * @param appId
     * @return
     * @since JDK 1.8
     */
    public String getTargetApp(Long appId) {
        
        return new StringBuilder().append(appId).append(APP_JOIN).append(NOT_LIMIT_TYPE).toString();
    }
    
    /**
     * checkTargetApp:(广告是否定向到该媒体广告位). <br/>
     *
     * @param targetAppSlots 定向广告位
     * @param appId 媒体id
     * @param slotId 广告位id
     * @return true, if check target app
     * @author chencheng
     * @since JDK 1.8
     */
    public Boolean checkIfTargetSolt(Set<String> targetAppSlots, Long appId, Long slotId) {
        // 定向广告位为空是异常,广告位为空，无法判断定向
        if (targetAppSlots == null || targetAppSlots.size() == 0  || null == slotId) {
            return false;
        }
        // 不限 或定向媒体下所有广告位
        if (targetAppSlots.contains("-1") || targetAppSlots.contains(this.getTargetApp(appId)) ) {
            return true;
        }
        // 定向媒体-广告位
        if (targetAppSlots.contains(this.getTargetApp(appId, slotId))) {
            return true;
        }
        return false;
    }

    /**
     * checkTargetApp:(广告是否定向到该媒体). <br/>
     *
     * @param targetApps 定向媒体：不为空
     * @param appId 媒体id
     * @return true, if check target app
     * @author chencheng
     * @since JDK 1.8
     */
    public Boolean checkIfTargetApp(Set<Long> targetApps, Long appId) {
        // 为空，表示异常，不能判断定向
        if (targetApps == null || targetApps.size() == 0) {
            return false;
        }
        // 不限或定向媒体
        if (targetApps.contains(-1L) || targetApps.contains(appId)) {
            return true;
        }
        return false;
    }

    /**
     * checkTargetApp:(广告是否定向到该媒体并且定向到广告位). <br/>
     *
     * @param targetApps 定向媒体：不为空
     * @param targetAppSlots 定向广告位：可为空
     * @param appId 媒体id
     * @param slotId 广告位id
     * @return true, if check target app
     * @author chencheng
     * @since JDK 1.8
     */
    public Boolean checkTargetApp(Set<Long> targetApps, Set<String> targetAppSlots, Long appId, Long slotId) {
        
        return checkIfTargetApp(targetApps, appId) && checkIfTargetSolt(targetAppSlots, appId, slotId);
    }
    
}
