package cn.com.duiba.tuia.cache;

import cn.com.duiba.tuia.dao.advert.ImproveBiddingAdvertDao;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @since JDK 1.6
 */
@Component
public class AppImproveBiddingAdvertCache {

    @Resource
    private ExecutorService executorService;

    @Resource
    private ImproveBiddingAdvertDao improveBiddingAdvertDao;

    //媒体对应可提升竞价能力的广告缓存
    private final LoadingCache<Long, List<Long>> APP_IMPROVE_BIDDING_ADVERT_CACHE = CacheBuilder.newBuilder().initialCapacity(100).
            recordStats().refreshAfterWrite(1, TimeUnit.HOURS).expireAfterWrite(3, TimeUnit.HOURS).build(new CacheLoader<Long, List<Long>>() {
        @Override
        public List<Long> load(Long key) {

            List<Long> advertIds = getAppImproveBiddingAdvert(key);
            if (CollectionUtils.isEmpty(advertIds)) {
                return Collections.emptyList();
            }
            return advertIds;
        }

        @Override
        public ListenableFuture<List<Long>> reload(final Long key, List<Long> oldValue) {
            ListenableFutureTask<List<Long>> task = ListenableFutureTask.create(() -> load(key));
            executorService.submit(task);
            return task;
        }
    });

    private List<Long> getAppImproveBiddingAdvert(Long appId) {
        return improveBiddingAdvertDao.selectLatestAppImproveBiddingAdvert(appId);
    }

    public List<Long> getAppImproveBiddingAdvertCache(Long appId) {
        if (appId == null) {
            return Collections.emptyList();
        }
        //获取所有的广告对应相似行业数据
        List<Long> advertIds = APP_IMPROVE_BIDDING_ADVERT_CACHE.getUnchecked(appId);
        if (CollectionUtils.isEmpty(advertIds)) {
            return Collections.emptyList();
        }
        return advertIds;
    }
}