package cn.com.duiba.tuia.cache;

import cn.com.duiba.tuia.ssp.center.api.remote.media.RemoteMeituanService;
import cn.com.duiba.tuia.ssp.center.api.remote.media.dto.CanServeAdvertDto;
import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

@Service
@RefreshScope
public class AvertCreativeCacheService extends BaseCacheService {

    @Autowired
    private RemoteMeituanService remoteMeituanService;

    @Resource
    private ExecutorService executorService;

    public static final int PAGE_ITEM_SIZE = 200;

    private static final String SPLIT_CHAR = "|";

    private LoadingCache<String, String> advertCreativeList = CacheBuilder.newBuilder().initialCapacity(4000)
            .refreshAfterWrite(1, TimeUnit.DAYS)
            .build(new CacheLoader<String, String>() {
                @Override
                public String load(String key) {
                    return getCreativeInfo(key);
                }

                @Override
                public ListenableFutureTask<String> reload(final String key, String oldValue) {
                    ListenableFutureTask<String> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });

    /**
     * 初始化缓存
     */
    public void init() {
        List<CanServeAdvertDto> resultList = getList();

        for (CanServeAdvertDto canServeAdvertDto : resultList) {
            String key = Joiner.on("|").join(canServeAdvertDto.getAdvertId(), canServeAdvertDto.getMaterialId(), canServeAdvertDto.getLandingPage());
            advertCreativeList.put(key, String.valueOf(canServeAdvertDto.getMediaAdvertId()));
        }
    }

    /**
     * 调用接口获取列表数据
     * @return
     */
    private List<CanServeAdvertDto> getList() {
        int totalCount = remoteMeituanService.selectCanServeAdvertCount();
        int totalPage = totalCount % PAGE_ITEM_SIZE == 0 ? totalCount / PAGE_ITEM_SIZE : totalCount / PAGE_ITEM_SIZE + 1;
        List<CanServeAdvertDto> allList = Lists.newArrayList();
        for (int i = 0; i < totalPage; i++) {
            try {
                int currenPage = i + 1;
                List<CanServeAdvertDto> resultList = remoteMeituanService.selectCanServeAdvertsWithPage(currenPage , PAGE_ITEM_SIZE);
                if (CollectionUtils.isNotEmpty(resultList)) {
                    allList.addAll(resultList);
                }
            } catch (Exception e) {
                logger.warn("初始化缓存第{}页数据异常 ", (i+1), e);
            }
        }

        return allList;
    }

    public String getCreativeInfo(String key) {
        try {
            List<String> param = Splitter.on(SPLIT_CHAR).splitToList(key);
            String advertId = param.get(0);
            String materialId = param.get(1);
            String promoteUrl = param.get(2);

            String creativeId = remoteMeituanService.selectCreativeId(Long.valueOf(advertId), Long.valueOf(materialId), promoteUrl);
            return StringUtils.isNotBlank(creativeId) ? creativeId : "";
        } catch (Exception e) {
            logger.warn("getCreativeInfo error key={}", key, e);
            return "";
        }
    }

    /**
     * 根据key获取创意素材信息
     * @param key
     * @return
     */
    public String getAdvertCreativeCache(String key) {
        try {
            String value = "";
            value = advertCreativeList.get(key);
            if (StringUtils.isBlank(value)) {
                //更新单缓存
                updateAdvertCreativeCache(key);
            }
            return value;
        } catch (Exception e) {
            logger.warn("getAdvertCreativeCache error", e);
            return "";
        }
    }

    /**
     * 更新缓存
     * @param key
     */
    public void updateAdvertCreativeCache(String key) {
        try {
            advertCreativeList.refresh(key);
        } catch (Exception e) {
            logger.error("updateAdvertCreativeCache error", e);
        }
    }
}
