package cn.com.duiba.tuia.cache;

import cn.com.duiba.tuia.dao.account.AdvertiserFirstLaunchDAO;
import cn.com.duiba.tuia.dao.lowarpu.LowArpuWhiteDAO;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.collections.MapUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @since JDK 1.6
 */
@Component
public class LowArpuCacheService {

    private static final Logger logger = LoggerFactory.getLogger(LowArpuCacheService.class);

    private static final String ADVERTISER_FIRST_LAUNCH_KEY = "advertiser_first_launch";

    private static final String LOW_ARPU_WHITE_KEY = "low_arpu_white";

    @Resource
    private ExecutorService executorService;

    @Autowired
    private AdvertiserFirstLaunchDAO advertiserFirstLaunchDAO;

    @Autowired
    private LowArpuWhiteDAO lowArpuWhiteDAO;

    //广告主第一次发券时间缓存
    private final LoadingCache<String, Map<Long, Date>> ADVERTISER_FIRST_LAUNCH_CACHE = CacheBuilder.newBuilder().initialCapacity(1000).
            recordStats().refreshAfterWrite(30, TimeUnit.MINUTES).expireAfterWrite(2, TimeUnit.HOURS).build(new CacheLoader<String, Map<Long, Date>>() {
        @Override
        public Map<Long, Date> load(String key) {

            Map<Long, Date> firstLaunchMap = getAdvertiserFirstLaunch();
            if (MapUtils.isEmpty(firstLaunchMap)) {
                return Collections.emptyMap();
            }
            return firstLaunchMap;
        }

        @Override
        public ListenableFuture<Map<Long, Date>> reload(final String key, Map<Long, Date> oldValue) {
            ListenableFutureTask<Map<Long, Date>> task = ListenableFutureTask.create(() -> load(key));
            executorService.submit(task);
            return task;
        }
    });

    private Map<Long, Date> getAdvertiserFirstLaunch() {
        try {
            return advertiserFirstLaunchDAO.selectAdvertiserFirstLaunch();
        } catch (Exception e) {
            logger.error("getAdvertiserFirstLaunch exception", e);
            return Collections.emptyMap();
        }
    }

    public void invalidAdvertiserFirstLaunch() {
        ADVERTISER_FIRST_LAUNCH_CACHE.invalidateAll();
    }

    public Map<Long, Date> getAdvertiserFirstLaunchFromCache() {
        return ADVERTISER_FIRST_LAUNCH_CACHE.getUnchecked(ADVERTISER_FIRST_LAUNCH_KEY);

    }

    private final LoadingCache<String, Map<Integer, List<Long>>> LOW_ARPU_WHITE_CACHE = CacheBuilder.newBuilder().initialCapacity(100).
            recordStats().refreshAfterWrite(30, TimeUnit.MINUTES).expireAfterWrite(2, TimeUnit.HOURS).build(new CacheLoader<String, Map<Integer, List<Long>>>() {
        @Override
        public Map<Integer, List<Long>> load(String key) {

            Map<Integer, List<Long>> firstLaunchMap = getLowArpuWhiteMap();
            if (MapUtils.isEmpty(firstLaunchMap)) {
                return Collections.emptyMap();
            }
            return firstLaunchMap;
        }

        @Override
        public ListenableFuture<Map<Integer, List<Long>>> reload(final String key, Map<Integer, List<Long>> oldValue) {
            ListenableFutureTask<Map<Integer, List<Long>>> task = ListenableFutureTask.create(() -> load(key));
            executorService.submit(task);
            return task;
        }
    });

    private Map<Integer, List<Long>> getLowArpuWhiteMap() {
        try {
            return lowArpuWhiteDAO.selectLowArpuWhite();
        } catch (Exception e) {
            logger.error("getLowArpuWhiteMap exception", e);
            return Collections.emptyMap();
        }
    }

    public void invalidLowArpuWhite() {
        LOW_ARPU_WHITE_CACHE.invalidateAll();
    }

    public void refreshLowArpuWhite() {
        LOW_ARPU_WHITE_CACHE.refresh(LOW_ARPU_WHITE_KEY);
    }

    public Map<Integer, List<Long>> getLowArpuWhiteFromCache() {
        return LOW_ARPU_WHITE_CACHE.getUnchecked(LOW_ARPU_WHITE_KEY);
    }

}