package cn.com.duiba.tuia.cache;

import cn.com.duiba.tuia.core.api.utils.UrlBase64;
import cn.com.duiba.tuia.dao.promotetest.AdvertPromoteTestDAO;
import cn.com.duiba.tuia.dao.promotetest.MakeTagDAO;
import cn.com.duiba.tuia.domain.dataobject.AdvertPromoteTestDO;
import cn.com.duiba.tuia.domain.dataobject.MakeTagDO;
import cn.com.duiba.tuia.tool.StringTool;
import cn.com.duiba.tuia.utils.UrlParseUtils;
import cn.com.tuia.advert.utils.DomainUtil;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

@Service
public class MakeTagCacheService extends BaseCacheService {

    @Resource
    private ExecutorService executorService;
    @Resource
    private AdvertPromoteTestDAO advertPromoteTestDAO;
    @Resource
    private MakeTagDAO makeTagDAO;


    /**落地页标签缓存,key是落地页id*/
    private LoadingCache<String, AdvertPromoteTestDO> promotePromoteTagsCache = CacheBuilder.newBuilder()
            .initialCapacity(5000)
            .refreshAfterWrite(10, TimeUnit.MINUTES)
            .build(
                    new CacheLoader<String, AdvertPromoteTestDO>() {

                        @Override
                        public AdvertPromoteTestDO load(String key) {
                            return getPromotePromoteTagsCacheCache(key);
                        }

                        @Override
                        public ListenableFutureTask<AdvertPromoteTestDO> reload(String key, AdvertPromoteTestDO prevGraph) {
                            ListenableFutureTask<AdvertPromoteTestDO> task = ListenableFutureTask
                                    .create(() ->getPromotePromoteTagsCacheCache(key));
                            executorService.submit(task);
                            return task;
                        }

                    }
            );


    //AdvertPromoteTest缓存 key： advertId+"--"+urlType+"--"+url
    private LoadingCache<String, AdvertPromoteTestDO> advertPromoteTestCache = CacheBuilder.newBuilder()
            .initialCapacity(5000)
            .refreshAfterWrite(10, TimeUnit.MINUTES)
            .build(
                    new CacheLoader<String, AdvertPromoteTestDO>() {
                        public AdvertPromoteTestDO load(String key) {
                            return getAdvertPromoteTestByCache(key);
                        }

                        public ListenableFutureTask<AdvertPromoteTestDO> reload(String key, AdvertPromoteTestDO prevGraph) {
                            ListenableFutureTask<AdvertPromoteTestDO> task = ListenableFutureTask
                                    .create(() -> getAdvertPromoteTestByCache(key));
                            executorService.submit(task);
                            return task;
                        }
                    });

    //MakeTag缓存 key： sourceId+"--"+sourceType 有消息同步，不用设置过期时间来保证避免读取旧值
    private LoadingCache<String, MakeTagDO> makeTagCache = CacheBuilder.newBuilder()
            .initialCapacity(5000)
            .refreshAfterWrite(180, TimeUnit.MINUTES)
            .build(
                    new CacheLoader<String, MakeTagDO>() {
                        public MakeTagDO load(String key) {
                            return getMakeTagByCache(key);
                        }

                        public ListenableFutureTask<MakeTagDO> reload(String key, MakeTagDO prevGraph) {
                            ListenableFutureTask<MakeTagDO> task = ListenableFutureTask
                                    .create(() -> getMakeTagByCache(key));
                            executorService.submit(task);
                            return task;
                        }
                    });


    /**
     * 根据条件获取落地页审核表落地页标签信息
     *
     * @param advertId
     * @param url
     * @return
     */
    public AdvertPromoteTestDO getAdvertPromoteTestTagByKey(Long advertId, Integer urlType, String url) {
        try {
            AdvertPromoteTestDO dto = advertPromoteTestCache.getIfPresent(buildAdvertUrlKey(advertId, urlType, url));
            return dto;
        } catch (Exception e) {
            logger.error("获取url落地页标签error", e);
        }
        return null;
    }


    /**
     * 根据条件获取百奇或积木域名特有落地页标签
     *
     * @param sourceId
     * @param sourceType
     * @return
     */
    public List<String> getMakeTagByKey(Long sourceId, Integer sourceType) {
        try {
            MakeTagDO makeTag = makeTagCache.getIfPresent(buildMakeTagKey(sourceId, sourceType));
            return makeTag == null ? null : makeTag.getDomainPromoteTags();
        } catch (Exception e) {
            logger.error("获取百奇或积木域名特有落地页标签error", e);
        }
        return null;
    }

    /**
     * 根据消息刷新落地页审核表落地页标签信息
     *
     * @return
     */
    public void refreshAdvertPromoteTest(String key) {
        advertPromoteTestCache.refresh(key);
    }

    /**
     * 刷新根据消息百奇或积木域名特有落地页标签
     *
     * @return
     */
    public void refreshMakeTag(String key) {
        makeTagCache.refresh(key);
    }


    /**
     * 初始化缓存
     *
     * @param advertList
     */
    public void init(List<Long> advertList) {
        try {
            //初始化落地页审核表信息
            List<AdvertPromoteTestDO> results = advertPromoteTestDAO.getAdvertPromoteTestByAdvertIds(advertList);
            List<Long> sourceIds = Lists.newArrayList();
            for (AdvertPromoteTestDO dto : results) {
                Long sourceId = builderPromoteTags(dto);
                if (sourceId != null) {
                    sourceIds.add(sourceId);
                }
                //同一个广告下，同类型的落地页，同一个URL，在广告投放链路中，标签会进行合并，共同生效
                String key = buildAdvertUrlKey(dto.getAdvertId(), dto.getUrlType(), dto.getPromoteUrl());
                AdvertPromoteTestDO oldVal = advertPromoteTestCache.getIfPresent(key);
                Set<String> tagSet = new HashSet<>();
                if (oldVal != null) {
                    Optional.ofNullable(oldVal.getPromoteTags()).ifPresent(oldList -> {
                        tagSet.addAll(oldList);
                    });
                    Optional.ofNullable(dto.getPromoteTags()).ifPresent(newList -> {
                        tagSet.addAll(newList);
                    });
                    dto.setPromoteTags(Lists.newArrayList(tagSet));
                }
                advertPromoteTestCache.put(key, dto);
                //promotePromoteTagsCache初始化
                String  promotePromoteTagsCacheKey= buildAdvertUrlKey(dto.getId());
                promotePromoteTagsCache.put(promotePromoteTagsCacheKey,dto);
            }

            //初始化 百奇或积木落地页标签信息
            if (CollectionUtils.isNotEmpty(sourceIds)) {
                List<MakeTagDO> sourceList = makeTagDAO.listMakeTag(sourceIds, null);
                for (MakeTagDO makeTagDO : sourceList) {
                    builderMakeTags(makeTagDO);
                    makeTagCache.put(buildMakeTagKey(makeTagDO.getSourceId(), makeTagDO.getSourceType()), makeTagDO);
                }
            }
        } catch (Exception e) {
            logger.error("初始化缓存异常", e);
        }
    }

    /**
     * 判断是否是积木或百奇落地页，并设置特有的落地页标签
     *
     * @param url
     */
    public void builderPromoteTags(String url) {
        try {
            if (url == null) {
                return;
            }
            MakeTagDO makeTagSign = getMakeTagSign(url);
            if (makeTagSign != null) {
                MakeTagDO makeTag = makeTagDAO.getMakeTag(makeTagSign.getSourceId(), makeTagSign.getSourceType());
                if (makeTag == null) {
                    return;
                }
                builderMakeTags(makeTag);
                makeTagCache.put(buildMakeTagKey(makeTag.getSourceId(), makeTag.getSourceType()), makeTag);
            }
        } catch (Exception e) {
            logger.error("判断是否是积木或百奇落地页error", e);
        }

    }

    /**
     * 判断是否是积木或百奇落地页，并设置特有的落地页标签
     *
     * @param url
     */
    public MakeTagDO getMakeTagSign(String url) {
        MakeTagDO makeTag = null;
        try {
            if (url == null) {
                return null;
            }
            //判断是否是百奇域名
            if (DomainUtil.isBaiqiDomain(url)) {
                makeTag = new MakeTagDO();
                Map<String, String> urlParamMap = UrlParseUtils.uRLRequest(url);
                makeTag.setSourceId(Long.valueOf(urlParamMap.get("pid")));
                makeTag.setSourceType(2);
            }
            //判断是否是积木域名
            if (DomainUtil.isJimuDomain(url)) {
                makeTag = new MakeTagDO();
                Map<String, String> urlParamMap = UrlParseUtils.uRLRequest(url);
                makeTag.setSourceId(Long.valueOf(UrlBase64.urlBase64Decode(urlParamMap.get("id"))));
                makeTag.setSourceType(1);
            }
        } catch (Exception e) {
            logger.error("是否是积木或百奇落地页error", e);
        }
        return makeTag;
    }

    /**
     * 根据key值查询落地页URL及落地页标签信息
     *
     * @param key
     * @return
     */
    public AdvertPromoteTestDO getAdvertPromoteTestByCache(String key) {
        try {
            String[] keys = key.split("--");

            //同一个广告下，同类型的落地页，同一个URL，在广告投放链路中，标签会进行合并，共同生效
            List<AdvertPromoteTestDO> results = advertPromoteTestDAO
                    .getAdvertPromoteTestByAdvertIdAndUrl(Long.valueOf(keys[0]), Integer.valueOf(keys[1]), keys[2]);
            if (CollectionUtils.isEmpty(results)) {
                return new AdvertPromoteTestDO();
            }
            AdvertPromoteTestDO result = results.get(0);
            Set<String> tagSet = new HashSet<>();
            for (AdvertPromoteTestDO dto : results) {
                builderPromoteTags(dto);
                Optional.ofNullable(dto.getPromoteTags()).ifPresent(list -> {
                    tagSet.addAll(list);
                });
            }
            result.setPromoteTags(Lists.newArrayList(tagSet));

            return result;
        } catch (Exception e) {
            return new AdvertPromoteTestDO();
        }
    }

    /**
     * 根据key值查询落地页URL及落地页标签信息
     *
     * @param key
     * @return
     */
    public MakeTagDO getMakeTagByCache(String key) {
        try {
            String[] keys = key.split("--");
            MakeTagDO makeTag = makeTagDAO.getMakeTag(Long.valueOf(keys[0]), Integer.valueOf(keys[1]));
            if (makeTag == null) {
                return new MakeTagDO();
            }
            builderMakeTags(makeTag);
            return makeTag;
        } catch (Exception e) {
            return new MakeTagDO();
        }
    }

    /**
     * 组装落地页标签
     *
     * @param dto
     */
    private Long builderPromoteTags(AdvertPromoteTestDO dto) {
        try {
            if (dto == null) {
                return null;
            }

            //判断是否是百奇域名
            if (DomainUtil.isBaiqiDomain(dto.getPromoteUrl())) {
                Map<String, String> urlParamMap = UrlParseUtils.uRLRequest(dto.getPromoteUrl());
                dto.setSourceId(Long.valueOf(urlParamMap.get("pid")));
                dto.setSourceType(2);
            }
            //判断是否是积木域名
            if (DomainUtil.isJimuDomain(dto.getPromoteUrl())) {
                Map<String, String> urlParamMap = UrlParseUtils.uRLRequest(dto.getPromoteUrl());
                dto.setSourceId(Long.valueOf(UrlBase64.urlBase64Decode(urlParamMap.get("id"))));
                dto.setSourceType(1);
            }
            dto.setPromoteTags(builderTags(dto.getPromoteTag()));
            return dto.getSourceId();
        } catch (Exception e) {
            logger.error("判断是否是积木或百奇落地页error", e.getMessage());
        }

        return null;

    }

    /**
     * 组装落地页标签
     *
     * @param promoteTag
     * @return
     */
    private List<String> builderTags(String promoteTag) {
        List<String> tags = Lists.newArrayList();
        if (StringUtils.isNotBlank(promoteTag)) {
            tags.addAll(StringTool.getStringListByStr(promoteTag));
        }
        return tags;
    }

    /**
     * 组装百奇或积木域名特有落地页标签
     *
     * @param makeTag
     * @return
     */
    private void builderMakeTags(MakeTagDO makeTag) {
        List<String> tags = Lists.newArrayList();
        if (StringUtils.isNotBlank(makeTag.getAePromoteTag())) {
            tags.addAll(StringTool.getStringListByStr(makeTag.getAePromoteTag()));
        }
        if (StringUtils.isNotBlank(makeTag.getAutoPromoteTag())) {
            tags.addAll(StringTool.getStringListByStr(makeTag.getAutoPromoteTag()));
        }
        makeTag.setDomainPromoteTags(tags);
    }

    public String buildAdvertUrlKey(Long advertId, Integer urlType, String url) {
        return advertId + "--" + urlType + "--" + url;
    }


    public String buildAdvertUrlKey(Long promoteId) {
        return String.valueOf(promoteId);
    }

    /**
     * 组装百奇或积木域名 标记的key
     *
     * @param sourceId
     * @param sourceType
     * @return
     */
    public String buildMakeTagKey(Long sourceId, Integer sourceType) {
        return String.valueOf(sourceId) + "--" + String.valueOf(sourceType);
    }


    public AdvertPromoteTestDO getPromoteTestTagByIdKey( Long promoteId) {

        try {
            AdvertPromoteTestDO dto = promotePromoteTagsCache.getIfPresent(buildAdvertUrlKey( promoteId));
            return dto;
        } catch (Exception e) {
            logger.error("获取url落地页标签error", e);
        }
        return null;


    }


    /**
     * 根据落地页id查询落地页打标信息和url
     *
     * @param key
     * @return
     */
    private AdvertPromoteTestDO getPromotePromoteTagsCacheCache(String key) {
        try {

            AdvertPromoteTestDO result = advertPromoteTestDAO.getAdvertPromoteTestByAdvertIdAndId(Long.valueOf(key));
            if (result == null) {
                return new AdvertPromoteTestDO();
            }
            return result;
        } catch (Exception e) {
            logger.warn("缓存,根据key值查询新的落地页URL及落地页标签信息 出错,返回一个空的");
            return new AdvertPromoteTestDO();
        }

    }
    /**
     * 根据消息刷新落地页审核表落地页标签信息
     *
     * @return
     */
    public void refreshPromotePromoteTagsCacheCache(String key) {
        promotePromoteTagsCache.refresh(key);
    }



}
