/**
 * Project Name:engine-service<br>
 * File Name:MediaCacheService.java<br>
 * Package Name:cn.com.duiba.tuia.cache<br>
 * Date:2016年11月23日下午2:58:50<br>
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.<br>
 */

package cn.com.duiba.tuia.cache;

import cn.com.duiba.tuia.api.TuiaMediaClientService;
import cn.com.duiba.tuia.constants.AdvertReqLogExtKeyConstant;
import cn.com.duiba.tuia.constants.ErrorCode;
import cn.com.duiba.tuia.dao.app.SpecialWeightAppDAO;
import cn.com.duiba.tuia.dao.engine.AppDAO;
import cn.com.duiba.tuia.dao.resource_tags.ResourceTagsDAO;
import cn.com.duiba.tuia.dao.slot.SlotFlowStrategyDAO;
import cn.com.duiba.tuia.domain.dataobject.AppDO;
import cn.com.duiba.tuia.domain.dataobject.ResoureTagsDO;
import cn.com.duiba.tuia.domain.dataobject.SlotDO;
import cn.com.duiba.tuia.domain.dataobject.SlotFLowStrategyDO;
import cn.com.duiba.tuia.domain.model.*;
import cn.com.duiba.tuia.domain.vo.MaterialPromoteTestUrlsVO;
import cn.com.tuia.advert.enums.ResourceTagsTypeEnum;
import cn.com.duiba.tuia.exception.TuiaException;
import cn.com.duiba.tuia.service.AdvertPromoteTestService;
import cn.com.duiba.tuia.service.ShieldingStrategyService;
import cn.com.duiba.tuia.service.SlotService;
import cn.com.duiba.tuia.service.SlotWhiteListService;
import cn.com.duiba.tuia.ssp.center.api.dto.MediaTagDto;
import cn.com.duiba.tuia.ssp.center.api.dto.StrategyCacheDto;
import cn.com.duiba.tuia.ssp.center.api.dto.slot.SlotWhiteListStatusDto;
import cn.com.duiba.tuia.ssp.center.api.remote.RemoteMediaTagService;
import cn.com.duiba.tuia.strategy.StrategyBeans;
import cn.com.duiba.tuia.tool.SwitchesUtil;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.duibaboot.ext.autoconfigure.cat.annotation.CatTransaction;
import cn.com.tuia.advert.cache.CacheKeyTool;
import cn.com.tuia.advert.enums.StrategyTypeEnum;
import cn.com.tuia.advert.model.ObtainAdvertReq;
import cn.com.tuia.advert.model.ObtainAdvertRsp;
import com.alibaba.fastjson.JSON;
import com.google.common.base.Splitter;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import com.google.common.util.concurrent.RateLimiter;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.net.URLDecoder;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * ClassName: MediaCacheService <br/>
 * Function: 媒体缓存. <br/>
 * date: 2016年11月23日 下午2:58:50 <br/>
 *
 * @author leiliang
 * @version
 * @since JDK 1.6
 */
@Service
public class MediaCacheService extends BaseCacheService {

    /**
     * 媒体缓存刷新限流, 20秒执行一次
     */
    private static final RateLimiter APP_CACHE_REFRESH_RATE_LIMITER = RateLimiter.create(0.1/2);
    
    /** 默认流量策略广告白名单排序位数 */
    public static final int DEFULT_STRATEGY_WHITE_SORT = 0;

    /** 新媒体流量策略广告白名单排序位数*/
    public static final int FLOW_STRATEGY_WHITE_SORT = 1;

    /** 默认流量策略 */
    public static final int DEFULT_STRATEGY_WHITE = 1;
    
    /** 新媒体流量策略*/
    public static final int FLOW_STRATEGY_WHITE = 2;
    
    /** 广告位媒体白名单排序开关为开 */
    public static final int SWITCH_OPEN = 1;

    /** 广告位媒体白名单排序开关为关 */
    public static final int SWITCH_CLOSE = 0;

    /** 广告位媒体白名单同步到流量策略 */
    public static final int SYNC_FLOW_STRATEGY = 1;



    @Autowired
    private TuiaMediaClientService    tuiaMediaClientService;

    @Autowired
    private AppDAO             appDAO;
    
    @Autowired
    private ResourceTagsDAO    resourceTagsDAO;

    @Autowired
    private SpecialWeightAppDAO specialWeightAppDAO;

    @Resource
    private ExecutorService executorService;

    @Resource
    private RemoteMediaTagService remoteMediaTagService;

    @Autowired
    private SlotService slotService;

    @Autowired
    private SlotWhiteListService slotWhiteListService;

    @Autowired
    private SlotFlowStrategyDAO slotFlowStrategyDAO;

    @Autowired
    private AdvertPromoteTestService advertPromoteTestService;
    
    @Autowired
    private ServiceManager serviceManager;

    /**
     * appDetail增加参数isHandledSlot用来标示流量策略是否已转换为广告位
     *
     * @param appDetail
     * @param shieldingStrategy
     * @return
     * @throws TuiaException
     */
    public List<Long> getWhiteList(AppDetail appDetail,
                                    ShieldingStrategyService shieldingStrategy) throws TuiaException {
        List<Long> whiteList = new ArrayList<>();

        SlotDO slotDO = appDetail.getSlotDO();
        if (appDetail.isHandledSlot() && slotDO != null) {
            //已转换
            whiteList = shieldingStrategy.getSlotWhiteList(appDetail);
        }

        return whiteList;
    }

    /**
     * 广告位白名单缓存,slotId为key
     */
    private final LoadingCache<Long, List<Long>> SLOTWHITE_CACHE = CacheBuilder.newBuilder().initialCapacity(4000).maximumSize(5000)
            .refreshAfterWrite(5, TimeUnit.MINUTES).expireAfterWrite(1, TimeUnit.HOURS).build(new CacheLoader<Long, List<Long>>() {
        @Override
        public List<Long> load(Long key)  {
            List<Long> originalAdvertIds = Lists.newArrayList();
            try {
                originalAdvertIds = slotWhiteListService.selectList(key);
                // 是否根据arpu值自动排序 此处为按照arup值排序 去掉
//                if (checkWhiteSort(key, DEFULT_STRATEGY_WHITE_SORT)) {
//                    List<Long> orderAdvertIds = sortWhiteAdvertList(originalAdvertIds);
//                    return CollectionUtils.isEmpty(orderAdvertIds) ? originalAdvertIds : orderAdvertIds;
//                }
            }
            catch (Exception ex){logger.error(ex.getMessage(),ex);}
            return originalAdvertIds;
        }

        @Override
        public ListenableFuture<List<Long>> reload(final Long key, List<Long> oldValue) {
            ListenableFutureTask<List<Long>> task = ListenableFutureTask.create(() -> load(key));
            executorService.submit(task);
            return task;
        }
    });
    
    /**
     * 
     * sortWhiteAdvertList:(广告arpu值排序). <br/>
     *
     * @author chencheng
     * @param advertIds 广告id
     * @return 
     * @throws TuiaException 
     * @since JDK 1.8
     */
    private List<Long> sortWhiteAdvertList(List<Long> advertIds) throws TuiaException {
        
        // 有效广告排序
        Map<Long, Integer> validAdvertOrderLevelMap = serviceManager.queryValidAdvertOrderLevel();
        return validAdvertOrderLevelMap.entrySet().stream().filter(entry -> advertIds.contains(entry.getKey())).sorted(Comparator.comparing(Map.Entry::getValue)).map(Map.Entry::getKey).collect(Collectors.toList());   
    }
   
    /**
     * 
     * checkWhiteSort:(判断字段排序开关是否开启). <br/>
     *
     * @author chencheng
     * @param soltId 广告位id
     * @param strategyWhiteSortType 未运算的位数
     * @return
     * @throws TuiaException
     * @since JDK 1.8
     */
    private boolean checkWhiteSort(Long soltId, int strategyWhiteSortType) throws TuiaException {
        
        SlotDO slotDO = slotService.getSlotBySlotId(soltId);
        int sortTypeBit = Optional.ofNullable(slotDO).map(solt -> Optional.ofNullable(solt.getSortType()).orElse(0)).orElse(0);
        // 位运算获取开关值
        int whiteSortType = SwitchesUtil.switchChange(sortTypeBit, strategyWhiteSortType);
        return SWITCH_OPEN == whiteSortType;
    }

    /**
     * 广告位白名单缓存，使用strategyType_strategyId_slotId作为key
     * 有消息同步，不用设置过期时间来保证避免读取旧值
     */
    @SuppressWarnings("all")
    private final LoadingCache<String, List<Long>> SLOTWHITE_STRATEGY_CACHE = CacheBuilder.newBuilder().initialCapacity(4000).maximumSize(5000)
            .refreshAfterWrite(5, TimeUnit.MINUTES).build(new CacheLoader<String, List<Long>>() {
                @Override
                public List<Long> load(String key)  {
                    List<String> ids = Splitter.on("_").splitToList(key);
                    Long strategyId = Long.parseLong(ids.get(1));
                    Long slotId = Long.parseLong(ids.get(2));
                    List<Long> originalAdvertIds = Lists.newArrayList();
                    try {
                        originalAdvertIds = slotWhiteListService.selectList(ids.get(0), strategyId, slotId);
                        // 是否根据arpu值自动排序 老的 排序不再判断
//                        if (checkWhiteSort(slotId, FLOW_STRATEGY_WHITE_SORT)) {
//                            List<Long> orderAdvertIds = sortWhiteAdvertList(originalAdvertIds);
//                            return CollectionUtils.isEmpty(orderAdvertIds) ? originalAdvertIds : orderAdvertIds;
//                        }
                    }
                    catch (Exception ex){
                        logger.error(ex.getMessage(),ex);
                    }
                    return originalAdvertIds;
                }

                @Override
                public ListenableFuture<List<Long>> reload(final String key, List<Long> oldValue) {
                    ListenableFutureTask<List<Long>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });

    //有消息同步，不用设置过期时间来保证避免读取旧值
    private final LoadingCache<Long, AppDO> APP_CACHE = CacheBuilder.newBuilder().initialCapacity(2000).maximumSize(2500)
            .refreshAfterWrite(10, TimeUnit.MINUTES).build(new CacheLoader<Long, AppDO>() {
        @Override
        public AppDO load(Long appId) {
            try {
                AppDO appDO = appDAO.getAppByAppId(appId);
                if(appDO==null)return AppDO.getDefaultApp();
                //查询媒体屏蔽的落地页标签，流量策略的落地页标签
                ResoureTagsDO promoteUrlTagsDO = resourceTagsDAO.selectResoureTagsDOById(appId, ResourceTagsTypeEnum.APP_BANNED_URL_TAG.getCode());
                ResoureTagsDO shieldMaterialTagsDO = resourceTagsDAO.selectResoureTagsDOById(appId, ResourceTagsTypeEnum.APP_SHIELD_MATERIAL_TAG.getCode());
                ResoureTagsDO appBannedResourceTagsDO = resourceTagsDAO.selectResoureTagsDOById(appId, ResourceTagsTypeEnum.APP_BANNED_RESOURCE_TAG.getCode());
                String promoteUrlTagNums = Optional.ofNullable(promoteUrlTagsDO).map(ResoureTagsDO::getTagNums).orElse(null);
                String shieldMaterialTagNums = Optional.ofNullable(shieldMaterialTagsDO).map(ResoureTagsDO::getTagNums).orElse(null);
                String appBannedResourceTagNums = Optional.ofNullable(appBannedResourceTagsDO).map(ResoureTagsDO::getTagNums).orElse(null);

                appDO.setPromoteUrlTags(promoteUrlTagNums);
                appDO.setShieldMaterialTag(shieldMaterialTagNums);
                appDO.setAppBannedResourceTags(appBannedResourceTagNums);

                logger.info("load app cache, app=[{}]", JSON.toJSONString(appDO));

                // 媒体标签查询时间
                AppTagQueryTime queryTime = new AppTagQueryTime();
                queryTime.setAppId(appId);
                queryTime.setAppTagTime(appDO.getGmtModified());

                queryTime.setAppBannedUrlTagTime(Optional.ofNullable(promoteUrlTagsDO).map(ResoureTagsDO::getGmtModified).orElse(null));
                queryTime.setAppShieldMaterialTagTime(Optional.ofNullable(shieldMaterialTagsDO).map(ResoureTagsDO::getGmtModified).orElse(null));

                appDO.setAppTagQueryTime(queryTime);
                return appDO;
            }
            catch (Exception ex){
                logger.error("APP_CACHE error",ex);
                return AppDO.getDefaultApp();
            }
        }

        @Override
        public ListenableFuture<AppDO> reload(final Long key, AppDO oldValue) {
            ListenableFutureTask<AppDO> task = ListenableFutureTask.create(()->load(key));
            executorService.submit(task);
            return task;
        }
    });

    /**
     * 广告位的缓存 有消息同步，不用设置过期时间来保证避免读取旧值
     */
    private final LoadingCache<Long, Optional<SlotDO>> SLOT_CACHE = CacheBuilder.newBuilder().initialCapacity(1000)
            .refreshAfterWrite(10, TimeUnit.MINUTES).build(new CacheLoader<Long, Optional<SlotDO>>() {
        @Override
        public Optional<SlotDO> load(Long slotId) {
            try {
                SlotDO slotDO = slotService.getSlotBySlotId(slotId);
                // 查询广告位屏蔽的落地页标签，流量策略的落地页标签
                if(slotDO!=null){
                    ResoureTagsDO promoteUrlTagsDO = resourceTagsDAO.selectResoureTagsDOById(slotId, ResourceTagsTypeEnum.SLOT_BANNED_URL_TAG.getCode());
                    ResoureTagsDO flowPromoteUrlTagsDO = resourceTagsDAO.selectResoureTagsDOById(slotId, ResourceTagsTypeEnum.SLOT_FLOW_BANNED_URL_TAG.getCode());
                    ResoureTagsDO flowShieldMaterialTagsDO = resourceTagsDAO.selectResoureTagsDOById(slotId, ResourceTagsTypeEnum.SLOT_FLOW_SHIELD_MATERIAL_TAG.getCode());
                    ResoureTagsDO flowBannedResourceTagsDO = resourceTagsDAO.selectResoureTagsDOById(slotId, ResourceTagsTypeEnum.SLOT_FLOW_BANNED_RESOURCE_TAG.getCode());

                    String promoteUrlTagNums = Optional.ofNullable(promoteUrlTagsDO).map(ResoureTagsDO::getTagNums).orElse(null);
                    String flowPromoteUrlTagNums = Optional.ofNullable(flowPromoteUrlTagsDO).map(ResoureTagsDO::getTagNums).orElse(null);
                    String flowShieldMaterialTagNums = Optional.ofNullable(flowShieldMaterialTagsDO).map(ResoureTagsDO::getTagNums).orElse(null);
                    String flowBannendResourceTagNums = Optional.ofNullable(flowBannedResourceTagsDO).map(ResoureTagsDO::getTagNums).orElse(null);

                    slotDO.setPromoteUrlTags(promoteUrlTagNums);
                    slotDO.setFlowPromoteUrlTags(flowPromoteUrlTagNums);
                    slotDO.setFlowShieldMaterialTag(flowShieldMaterialTagNums);
                    slotDO.setFlowBannendResourceTags(flowBannendResourceTagNums);

                    // 广告位标签查询时间
                    SlotTagQueryTime queryTime = new SlotTagQueryTime();
                    queryTime.setSlotId(slotId);
                    queryTime.setSlotTagTime(slotDO.getGmtModified());

                    queryTime.setSlotBannedUrlTagTime(Optional.ofNullable(promoteUrlTagsDO).map(ResoureTagsDO::getGmtModified).orElse(null));
                    queryTime.setSlotFlowBannedUrlTagTime(Optional.ofNullable(flowPromoteUrlTagsDO).map(ResoureTagsDO::getGmtModified).orElse(null));
                    queryTime.setSlotFlowMaterialTagTime(Optional.ofNullable(flowShieldMaterialTagsDO).map(ResoureTagsDO::getGmtModified).orElse(null));

                    slotDO.setSlotTagQueryTime(queryTime);
                }

                logger.info("load slot cache, slot=[{}]", JSON.toJSONString(slotDO));
                return Optional.ofNullable(slotDO);
            }
            catch (Exception ex){
                logger.error("SLOT_CACHE error",ex);
                return Optional.ofNullable(null);
            }
        }

        @Override
        public ListenableFuture<Optional<SlotDO>> reload(final Long key, Optional<SlotDO> oldValue) {
            ListenableFutureTask<Optional<SlotDO>> task = ListenableFutureTask.create(()->load(key));
            executorService.submit(task);
            return task;
        }
    });

    /**
     * 根据兑吧应用id获取应用app信息,首先从缓存中查询，如果缓存中没有,则从DB查询，并更新到缓存中 <一句话功能描述>
     * 
     * @param appId
     * @return
     * @throws TuiaException
     */
    public AppDO getApp(Long appId) throws TuiaException {
        AppDO appDO=APP_CACHE.getUnchecked(appId);
        if ((appDO == null || appDO.getAppId() == AppDO.DEFAUT_APP_ID)
                && APP_CACHE_REFRESH_RATE_LIMITER.tryAcquire()) {
            APP_CACHE.refresh(appId);
            if (logConfig.getInfoEnable()) {
                logger.info("APP_CACHE_REFRESH_RATE_LIMITER appId={}", appId);
            }
        }
        return appDO;
    }
    /**
     * 更新媒体白名单缓存.
     *
     * @param appId the app id
     * @throws TuiaException the tuia exception
     */
    public void updateAppCache(Long appId) {
        APP_CACHE.refresh(appId);
    }

    /**
     * Gets the shield strategy vo.
     *
     * @param appId the app id
     * @param slotId the slot id
     * @return the shield strategy vo
     * @throws TuiaException the tuia exception
     */
    @SuppressWarnings("squid:S3776")
    @CatTransaction(type = "buildParameters", name= "getAppDetailCache")
    public  AppDetail getAppDetailCache(Long appId, Long slotId, ObtainAdvertRsp rsp, Boolean isProxy, String cityId, ObtainAdvertReq req) throws TuiaException {
        try {
            DBTimeProfile.enter("getAppDetailCache");
            //1.APP每天定时同步过来，这里包括媒体的应用管理 | 流量策略的行业，属性标签，落地页标签
            AppDO appDO = getApp(appId);
            if (appDO == null || appDO.getAppId() == AppDO.DEFAUT_APP_ID) {
                logger.error("the app=[{}] data is null", appId);
                throw new TuiaException(ErrorCode.E0500002);
            }
            //2.如果媒体定向到广告位的维度,则查询广告位的应用管理 | 流量策略的行业，属性标签，落地页标签
            SlotDO slotDO = null;
            if(null == slotId){
                logger.warn("slotId is null,the req is [{}]",req);
            }else if (null != appDO.getIsHandledSlot() && 1 == appDO.getIsHandledSlot()) {
                slotDO = getSlot(slotId);
            }
            //3.媒体广告位与媒体不符
            if(null != slotDO && !slotDO.getAppId().equals(appId)){
                throw new TuiaException(ErrorCode.E0500025);
            }
            //4.媒体标签集合
            List<String> appTagList = new ArrayList<>();
            AdvBannedTag advBannedTag = AdvBannedTag.build();
            //5.屏蔽链接
            Set<String> shieldUrls = Sets.newHashSet();
            //屏蔽的素材标签
            Set<String> shieldMaterialTags = Sets.newHashSet();

            Object objParam = null;
            CheckStrategyRet checkRet = new CheckStrategyRet();

            //5.1循环策略列表
            for(ShieldingStrategyService ss : StrategyBeans.shieldingStrategies){
                checkRet = ss.checkStrategy(appDO, slotDO, isProxy, cityId, slotId);
                //5.2不需要验证下一个策略赋值，返回
                if(!checkRet.isNextCheck()){
                    objParam = checkRet.getObject();
                    shieldUrls.addAll(urlDecorder(checkRet.getShieldUrl()));
                    appTagList.addAll(ss.getTags(appDO,slotDO,checkRet.isHandledSlot()));
                    ss.handleBannedTag(appDO, slotDO, checkRet.isHandledSlot(), advBannedTag);
                    shieldMaterialTags.addAll(ss.getShieldMaterialTags(appDO, slotDO));
                    rsp.setStrategyType(ss.getShieldingStrategyType());
                    break;
                }
            }

            if(logConfig.getInfoEnable()) {
                logger.info("orderId=[{}], appTag=[{}], shieldUrls=[{}], sheildMaterialTags=[{}]", req.getOrderId(), JSON.toJSONString(appTagList), JSON.toJSONString(shieldUrls), JSON.toJSONString(shieldMaterialTags));
            }

            ShieldStrategyVO shieldStrategyVO = new ShieldStrategyVO();
            //6.如果是走默认策略，才走下面的从媒体方查询的广告位的标签，否则按照我方的广告位流量策略的标签
            if (appId != null && slotId != null && 
                    (!checkRet.isHandledSlot() || checkRet.isHandledSlot() && StrategyTypeEnum.STRATEGY_TYEP_DEFAULT.getCode().equals(rsp.getStrategyType())))
               {
                Optional<StrategyCacheDto> strategyDto = tuiaMediaClientService.getCacheStrategyBySlotId(slotId);
                if (strategyDto.isPresent()) {
                    StrategyCacheDto strategyCacheDto = strategyDto.get();
                    // 查询出的结果不为空的情况
                    if(strategyCacheDto.getId() != -1L){
                        buildWithSoltStrategy(appId, shieldMaterialTags, appTagList, shieldUrls, strategyCacheDto, advBannedTag);

                        // 记录查询时间
                        if(slotDO != null){
                            SlotTagQueryTime queryTime = slotDO.getSlotTagQueryTime();
                            queryTime.setMediaTagTime(strategyCacheDto.getGmtModified());
                            slotDO.setSlotTagQueryTime(queryTime);
                        }
                    }
                }else {
                    throw new TuiaException(ErrorCode.E0500018);
                }
            }

            // 设置流量福袋，转化，流量类型
            configFlowStrategy(shieldStrategyVO, checkRet);

            //7.用于日志打印
            req.setAppTags(appTagList);
            //用于dsp 发券日志
            if (rsp.getLogExtMap() == null) {
                rsp.setLogExtMap(Maps.newHashMap());
            }
            rsp.getLogExtMap().put(AdvertReqLogExtKeyConstant.LAUNCH_LOG_APP_TAGS,appTagList);
            // 日志记录app和广告位禁用标签信息
            shieldStrategyVO.setAppBannedTags(appTagList);
            shieldStrategyVO.setAdvBannedTag(advBannedTag);
            shieldStrategyVO.setShieldUrls(shieldUrls);
            shieldStrategyVO.setShieldMaterialTags(shieldMaterialTags);
            return new AppDetail(shieldStrategyVO, appDO, slotDO, objParam, checkRet.isHandledSlot());
        } finally {
            DBTimeProfile.release();
        }
    }

    private void configFlowStrategy(ShieldStrategyVO shieldStrategyVO, CheckStrategyRet checkRet) {
        Integer flowType = checkRet.getFlowType();
        Boolean luckyBag = null;
        Boolean handledSlot = null;
        // 应用管理
        if(1 == flowType){

            // 福袋开关
            luckyBag = (Boolean) checkRet.getObject();

            handledSlot = checkRet.isHandledSlot();

            // 流量策略
        }else if(2 == flowType){

            FlowStrategyRet fs = (FlowStrategyRet) checkRet.getObject();

            // 福袋开关
            luckyBag = fs.isSendLuckybag();

            // 是否转化过广告位
            handledSlot = checkRet.isHandledSlot();
        }

        shieldStrategyVO.setFlowType(flowType);
        shieldStrategyVO.setLuckyBag(luckyBag);
        shieldStrategyVO.setHandledSlot(handledSlot);
    }

    /**
     * 根据slotId获取广告位的信息，放到缓存中，key:slotId
     * 只存落地页标签
     * @param slotId
     * @return
     */
    private SlotDO getSlot(Long slotId) throws TuiaException{
        Optional<SlotDO> slotDO = SLOT_CACHE.getUnchecked(slotId);
        return slotDO.orElse(null);
    }

    //有消息同步，不用设置过期时间来保证避免读取旧值
    private final LoadingCache<Long, List<MediaTagDto>> SLOT_TAGS_CACHE = CacheBuilder.newBuilder().initialCapacity(4000).maximumSize(5000)
            .refreshAfterWrite(10, TimeUnit.MINUTES).build(new CacheLoader<Long, List<MediaTagDto>>() {
                @Override
                public List<MediaTagDto> load(Long slotId) {
                    try{
                        DubboResult<List<MediaTagDto>> mediaTagDtos = remoteMediaTagService.getByIdsAndBlock(Collections.singletonList(slotId), 3);//stringRedisTemplate.opsForHash().get(slotidTagsKey, String.valueOf(slotId));

                        List<MediaTagDto> tags = mediaTagDtos.getResult();

                        return tags == null ? Collections.emptyList() : tags;
                    }catch(Exception e){
                        logger.error("getSlotidTags error slotId:{}", slotId, e);
                    }
                    return Collections.emptyList();
                }

                @Override
                public ListenableFuture<List<MediaTagDto>> reload(final Long key, List<MediaTagDto> oldValue) {
                    ListenableFutureTask<List<MediaTagDto>> task = ListenableFutureTask.create(()->load(key));
                    executorService.submit(task);
                    return task;
                }
            });
    /**
     * 获取广告位的标签 0:父标签 1：子标签
     * @param slotId
     * @return
     */
    public List<MediaTagDto> getSlotidTags(Long slotId) {
        if(slotId==null)return Lists.newArrayList();
        return SLOT_TAGS_CACHE.getUnchecked(slotId);
    }

    public void updateSlotTags(Long slotId) {
        SLOT_TAGS_CACHE.invalidate(slotId);
    }

    //有消息同步，不用设置过期时间来保证避免读取旧值
    private final LoadingCache<Long, List<String>> APPFLOW_TAG_CACHE = CacheBuilder.newBuilder().initialCapacity(4000).maximumSize(5000)
            .refreshAfterWrite(10, TimeUnit.MINUTES).build(new CacheLoader<Long, List<String>>() {
                @Override
                public List<String> load(Long appId) {
                    try{
                        MediaTagDto tags = remoteMediaTagService.queryMediaTagByAppIdType(appId,2); //stringRedisTemplate.opsForHash().get(appFlowTagsKey, String.valueOf(appId));
                        if(tags != null){
                            return Lists.newArrayList(tags.getPid().toString(),tags.getTagId().toString());
                        }
                    }catch(Exception e){
                        logger.error("getAppFlowTags error",e);
                    }
                    return Collections.emptyList();
                }
                @Override
                public ListenableFuture<List<String>> reload(final Long key, List<String> oldValue) {
                    ListenableFutureTask<List<String>> task = ListenableFutureTask.create(()->load(key));
                    executorService.submit(task);
                    return task;
                }
            });
    /**
     * 获取媒体流量标签 0:父标签 1：子标签
     * @param appId
     * @return
     */
    public List<String> getAppFlowTags(Long appId) {
        if(appId==null)return Lists.newArrayList();
       return APPFLOW_TAG_CACHE.getUnchecked(appId);
    }

    public void updateAppFlowTags(Long appId) {
        APPFLOW_TAG_CACHE.invalidate(appId);
    }

    //有消息同步，不用设置过期时间来保证避免读取旧值
    private final LoadingCache<Long, List<String>> APP_INDUSTRY_CACHE = CacheBuilder.newBuilder().initialCapacity(4000).maximumSize(5000)
            .refreshAfterWrite(10, TimeUnit.MINUTES).build(new CacheLoader<Long, List<String>>() {
                @Override
                public List<String> load(Long appId) {
                    try{
                        MediaTagDto tags = remoteMediaTagService.queryMediaTagByAppIdType(appId,1); //stringRedisTemplate.opsForHash().get(appIndustryTagsKey, String.valueOf(appId));
                        if(tags != null){
                            return Lists.newArrayList(tags.getPid().toString(),tags.getTagId().toString());
                        }
                    }catch(Exception e){
                        logger.error("getAppIndustryTags error",e);
                    }
                    return Collections.emptyList();
                }

                @Override
                public ListenableFuture<List<String>> reload(final Long key, List<String> oldValue) {
                    ListenableFutureTask<List<String>> task = ListenableFutureTask.create(()->load(key));
                    executorService.submit(task);
                    return task;
                }
            });
    /**
     * 获取媒体行业标签 0:父标签 1：子标签
     * @param appId
     * @return
     */
    public List<String> getAppIndustryTags(Long appId) {
        if(appId==null)return Lists.newArrayList();
     return APP_INDUSTRY_CACHE.getUnchecked(appId);
    }

    public void updateAppIndustryTags(Long appId) {
        APP_INDUSTRY_CACHE.invalidate(appId);
    }

    //有消息同步，不用设置过期时间来保证避免读取旧值
    private final LoadingCache<Long, Long> APP_EMO_BLOCK_CACHE = CacheBuilder.newBuilder().initialCapacity(4000).maximumSize(5000)
            .refreshAfterWrite(10, TimeUnit.MINUTES).build(new CacheLoader<Long, Long>() {
                @Override
                public Long load(Long appId) {
                    try{
                        MediaTagDto tags = remoteMediaTagService.queryMediaTagByAppIdType(appId, MediaTagDto.EMO_BLOCK); //stringRedisTemplate.opsForHash().get(appIndustryTagsKey, String.valueOf(appId));
                        if(tags != null){
                            return tags.getTagId();
                        }
                    }catch(Exception e){
                        logger.error("getAppIndustryTags error",e);
                    }
                    return -1L;
                }

                @Override
                public ListenableFuture<Long> reload(final Long key, Long oldValue) {
                    ListenableFutureTask<Long> task = ListenableFutureTask.create(()->load(key));
                    executorService.submit(task);
                    return task;
                }
            });
    /**
     * 获取媒体是否网赚标签
     * @param appId
     * @return
     */
    public Long getAppEmoBlockTag(Long appId) {
        if (appId == null) {
            return null;
        }
        return APP_EMO_BLOCK_CACHE.getUnchecked(appId);
    }

    public void updateAppEmoBlockTags(Long appId) {
        APP_EMO_BLOCK_CACHE.invalidate(appId);
    }

    /**
     * 添加广告位策略
     * @param appId
     * @param appTagList
     * @param shieldUrls
     * @param strategyDto
     * @param advBannedTag
     * @return
     * @throws TuiaException
     */
    private void buildWithSoltStrategy(Long appId, Set<String> shieldMaterialTags, List<String> appTagList,
                                       Set<String> shieldUrls, StrategyCacheDto strategyDto, AdvBannedTag advBannedTag) throws TuiaException {
        //媒体是否匹配
        if(strategyDto.getAppId() != null && !strategyDto.getAppId().equals(appId)){
            throw new TuiaException(ErrorCode.E0500025);
        }
        shieldUrls.addAll(urlDecorder(strategyDto.getShieldUrls()));
        
        if (strategyDto.getAdvertTagNums() != null) {
            appTagList.addAll(strategyDto.getAdvertTagNums());
            advBannedTag.getAttributeTags().addAll(strategyDto.getAdvertTagNums());
        }
        if (strategyDto.getShieldIndustries() != null) {
            appTagList.addAll(strategyDto.getShieldIndustries());
            advBannedTag.getIndustryTags().addAll(strategyDto.getShieldIndustries());
        }
        if (CollectionUtils.isNotEmpty(strategyDto.getPromoteTagNums())) {
            appTagList.addAll(strategyDto.getPromoteTagNums());
            advBannedTag.getPromoteUrlTags().addAll(strategyDto.getPromoteTagNums());
        }
        if (CollectionUtils.isNotEmpty(strategyDto.getResourceTagNums())) {
            appTagList.addAll(strategyDto.getResourceTagNums());
            advBannedTag.getResourceTags().addAll(strategyDto.getResourceTagNums());
        }
        //合并媒体管理后台屏蔽的素材标签
        if (CollectionUtils.isNotEmpty(strategyDto.getShieldMaterialTags())) {
            shieldMaterialTags.addAll(strategyDto.getShieldMaterialTags());
        }
    }

    private List<String> urlDecorder(List<String> urls){
        List<String> urlList = Lists.newArrayList();
        if(null == urls || urls.isEmpty()){
            return  urlList;
        }
        for(String url : urls){
            try{
                urlList.add(URLDecoder.decode(url, "utf-8"));
            }catch(Exception e){
                logger.error("urlDecorder error",e);
            }
        }
        return urlList;
    }

    public Boolean isSpecialWeightApp(String appId,Set<String> specialWeightAppIdList) {
        try{

            if(specialWeightAppIdList.contains(appId)){
                return true;
            }
        }catch(Exception e){
            logger.error("getAppIndustryTags error",e);
        }
        return false;
    }

    //有消息同步，不用设置过期时间来保证避免读取旧值
    private final LoadingCache<Long, Set<String>> SPECIAL_WEIGHT_APP_CACHE = CacheBuilder.newBuilder().initialCapacity(1000).
            recordStats().refreshAfterWrite(15, TimeUnit.MINUTES).build(new CacheLoader<Long, Set<String>>() {
        @Override
        public Set<String> load(Long key)  {
            List<String> specialWeightAppIdList=Lists.newArrayList();
            try {
                specialWeightAppIdList = specialWeightAppDAO.getAllAppId();
            }
            catch (Exception ex){logger.error(ex.getMessage(),ex);}
            if(null == specialWeightAppIdList || specialWeightAppIdList.isEmpty()){
                specialWeightAppIdList = Lists.newArrayList();
                specialWeightAppIdList.add("-1");
            }
            return Sets.newHashSet(specialWeightAppIdList);
        }

        @Override
        public ListenableFuture<Set<String>> reload(final Long key, Set<String> oldValue) {
            ListenableFutureTask<Set<String>> task = ListenableFutureTask.create(() -> load(key));
            executorService.submit(task);
            return task;
        }
    });

    public Set<String> getSpecialWeightAppIdList() {
        return SPECIAL_WEIGHT_APP_CACHE.getUnchecked(0L);
    }
    public void updateSpecialWeightAppIdList() {
          SPECIAL_WEIGHT_APP_CACHE.refresh(0L);
    }

    /**
     * 广告位白名单缓存，默认策略的
     * @param slotId
     * @return
     */
    public List<Long> getSlotWhiteList(Long slotId) {
        return SLOTWHITE_CACHE.getUnchecked(slotId);
    }

    /**
     * 广告位白名单缓存，新流量策略的
     * @param strategyTypeFlow
     * @param strategyId
     * @param slotId
     * @return
     */
    public List<Long> getSlotWhiteList(StrategyTypeEnum strategyTypeFlow, Long strategyId, Long slotId) {
        return SLOTWHITE_STRATEGY_CACHE.getUnchecked(CacheKeyTool.getCacheKey(strategyTypeFlow.getCode(), strategyId, slotId));
    }

    /**
     * 刷新广告位缓存
     * @param slotId
     */
    public void updateSlotCache(Long slotId) {
        SLOT_CACHE.refresh(slotId);
    }

    /**
     * 刷新广告位白名单缓存
     * @param slotId
     */
    public void updateSlotWhiteList(Long slotId) {
        SLOTWHITE_CACHE.refresh(slotId);
    }

    /**
     * 刷新广告位白名单，带有流量策略的白名单，新策略
     * @param key
     */
    public void updateSlotWhiteStrategy(String key) {
        SLOTWHITE_STRATEGY_CACHE.refresh(key);
    }

    /**
     * 广告位策略缓存 有消息同步，不用设置过期时间来保证避免读取旧值
     */
    private final LoadingCache<Long, Optional<SlotFLowStrategyDO>> SLOT_FLOW_STRATEGY_CACHE = CacheBuilder.newBuilder().initialCapacity(300).
                    recordStats().refreshAfterWrite(1, TimeUnit.MINUTES).build(new CacheLoader<Long, Optional<SlotFLowStrategyDO>>() {
        @Override
        public Optional<SlotFLowStrategyDO> load(Long slotId) throws Exception {
            SlotFLowStrategyDO flow = slotFlowStrategyDAO.findSlotFlowStrategyBySlotId(slotId);
            return Optional.ofNullable(flow);
        }

        @Override
        public ListenableFuture<Optional<SlotFLowStrategyDO>> reload(final Long key, Optional<SlotFLowStrategyDO> oldValue) throws Exception {
            ListenableFutureTask<Optional<SlotFLowStrategyDO>> task = ListenableFutureTask.create(new Callable<Optional<SlotFLowStrategyDO>>() {
                public Optional<SlotFLowStrategyDO> call() throws Exception {
                    return load(key);
                }
            });
            executorService.submit(task);
            return task;
        }
    });

    /**
     * 获取广告位流量策略缓存
     * @param slotId
     * @return
     */
    public Optional<SlotFLowStrategyDO> getSlotFlowStrategyBySlotId(Long slotId) {
        return SLOT_FLOW_STRATEGY_CACHE.getUnchecked(slotId);
    }

    /**
     * 刷新广告位流量策略缓存
     * @param slotId
     */
    public void updateSlotFlowStrategyCache(Long slotId) {
        SLOT_FLOW_STRATEGY_CACHE.refresh(slotId);
    }

    /**
     * 获取素材对应的测试落地页链接和分配流量占比
     * @param materialId
     * @return
     * @throws TuiaException
     */
//    public MaterialPromoteTestUrlsVO getMaterialPromoteTest(Long materialId) throws TuiaException{
//        Optional<MaterialPromoteTestUrlsVO> promoteTestUrlsVO = MATERIAL_PROMOTE_TEST_CACHE.getUnchecked(materialId);
//        return promoteTestUrlsVO.orElse(null);
//    }

    //有消息同步，不用设置过期时间来保证避免读取旧值
//    private final LoadingCache<Long, Optional<MaterialPromoteTestUrlsVO>> MATERIAL_PROMOTE_TEST_CACHE = CacheBuilder.newBuilder().initialCapacity(1000)
//            .refreshAfterWrite(1, TimeUnit.HOURS).build(new CacheLoader<Long, Optional<MaterialPromoteTestUrlsVO>>() {
//        @Override
//        public Optional<MaterialPromoteTestUrlsVO> load(Long materialId) {
//            try {
//                MaterialPromoteTestUrlsVO promoteTestUrlsVO = advertPromoteTestService.findPromoteUrlsByMaterialId(materialId);
//
//                return Optional.ofNullable(promoteTestUrlsVO);
//            }
//            catch (Exception ex){
//                logger.error(ex.getMessage(),ex);
//                return Optional.ofNullable(null);
//            }
//        }
//
//        @Override
//        public ListenableFuture<Optional<MaterialPromoteTestUrlsVO>> reload(final Long key, Optional<MaterialPromoteTestUrlsVO> oldValue) {
//            ListenableFutureTask<Optional<MaterialPromoteTestUrlsVO>> task = ListenableFutureTask.create(()->load(key));
//            executorService.submit(task);
//            return task;
//        }
//    });

    /**
     * 刷新缓存
     * @param key
     */
    public void updateMaterialPromoteTest(Long key) {

        logger.info("缓存下线，请查看消息来源");
        //        MATERIAL_PROMOTE_TEST_CACHE.refresh(key);
    }

    /**
     * 设置媒体新增白名单缓存
     * @param key
     */
//    private final LoadingCache<String, Boolean> MEDIA_SLOT_CACHE = CacheBuilder.newBuilder().maximumSize(6000)
//            .refreshAfterWrite(5, TimeUnit.MINUTES).build(new CacheLoader<String, Boolean>() {
//                @Override
//                public Boolean load(String key) {
//                    try {
//                        return slotWhiteListService.isSlotAdvertInWhiteList(key);
//                    }catch (Exception e){
//                        logger.error("MEDIA_SLOT_CACHE key:{} exception",key,e);
//                        return false;
//                    }
//                }
//
//                @Override
//                public ListenableFuture<Boolean> reload(final String key, Boolean oldValue) {
//                    ListenableFutureTask<Boolean> task = ListenableFutureTask.create(()->load(key));
//                    executorService.submit(task);
//                    return task;
//                }
//            });


//    public Boolean isExistMediaSlotWhite(Long slotId,Long advertId){
//
//        String key = slotId+"-"+advertId;
//
//        Boolean result =  MEDIA_SLOT_CACHE.getUnchecked(key);
//
//        return result;
//    }


    /**
     * 判断当前广告是否开启 白名单开关，且满足对于的范围
     * @param slotId
     * @param strategyType 1 -应用管理，2-流量策略
     * @return
     */
    public Boolean isOpenSlotWhiteSwitch(Long slotId,Integer strategyType){

        try {

            if (slotId == null) {
                return false;
            }

            SlotWhiteListStatusDto slotWhiteListStatusDto = slotWhiteListService.isOpenSwitch(slotId);

            //未开启白名单开关
            if (slotWhiteListStatusDto.getOpenWihteList() == SWITCH_CLOSE) {
                return false;
            }

            if (strategyType == DEFULT_STRATEGY_WHITE) {
                //活动开启白名单默认是对应用管理生效
                return true;
            } else if (strategyType == FLOW_STRATEGY_WHITE && slotWhiteListStatusDto.getIsSyncWhiteList() == SYNC_FLOW_STRATEGY) {
                //当前请求未流量策略 且 活动白名单同步流量策略
                return true;
            }

            return false;
        }catch (Exception e){
            logger.info("isOpenSlotWhiteSwitch exception:",e);
            return false;
        }
    }
}
