/**
 * Project Name:engine-service File Name:WelfareFlowAdvertCacheService.java Package Name:cn.com.duiba.tuia.cache
 * Date:2018年7月11日下午4:50:44 Copyright (c) 2018, duiba.com.cn All Rights Reserved.
 */

package cn.com.duiba.tuia.cache;

import java.util.*;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import cn.com.duiba.tuia.domain.dataobject.NewAppTestPriorityDO;
import cn.com.duiba.tuia.domain.model.AdvertFilter;
import cn.com.duiba.tuia.enums.CatGroupEnum;
import cn.com.duiba.tuia.exception.TuiaRuntimeException;
import cn.com.duiba.tuia.ssp.center.api.constant.NewMediaTestEnum;
import cn.com.duiba.tuia.ssp.center.api.dto.NewMediaTestDto;
import cn.com.duiba.tuia.ssp.center.api.remote.RemoteNewMediaTestService;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.tuia.advert.constants.SystemConfigKeyConstant;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSON;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;

import cn.com.duiba.tuia.constants.ErrorCode;
import cn.com.duiba.tuia.dao.advertNewtrade.AdvertNewTradeDAO;
import cn.com.duiba.tuia.dao.engine.NewAppAdvertTradeDAO;
import cn.com.duiba.tuia.dao.engine.NewAppDAO;
import cn.com.duiba.tuia.dao.engine.NewAppTestToAdvertDAO;
import cn.com.duiba.tuia.domain.dataobject.AdvertNewTradeDO;
import cn.com.duiba.tuia.domain.dataobject.NewAppAdvertTradeDO;
import cn.com.duiba.tuia.domain.dataobject.NewAppTestToAdvertDO;
import cn.com.duiba.tuia.domain.model.FilterResult;
import cn.com.duiba.tuia.domain.model.FilterResultTypes;
import cn.com.duiba.tuia.domain.model.UpdateNewAppTestCacheMsg;
import cn.com.duiba.tuia.domain.vo.AdvertFilterVO;
import cn.com.duiba.tuia.domain.vo.AdvertPriceVO;
import cn.com.duiba.tuia.domain.vo.AdvertVO;
import cn.com.duiba.tuia.exception.TuiaException;
import cn.com.duiba.tuia.service.AdvertExposeService;
import cn.com.duiba.tuia.service.AdvertRealDataService;
import cn.com.duiba.tuia.ssp.center.api.dto.MediaTagDto;
import cn.com.duiba.tuia.ssp.center.api.remote.RemoteMediaTagService;
import cn.com.duiba.tuia.tool.CatUtil;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.tuia.advert.cache.CacheKeyTool;
import cn.com.tuia.advert.cache.RedisCommonKeys;
import cn.com.tuia.advert.cache.RedisPreKey;
import cn.com.tuia.advert.enums.ChargeTypeEnum;
import cn.com.tuia.advert.model.ObtainAdvertReq;
import cn.com.tuia.advert.model.ObtainAdvertRsp;
import static java.util.Comparator.comparing;
import static java.util.stream.Collectors.toList;

/**
 * ClassName:WelfareFlowAdvertCacheService <br/>
 * Function: 福利流量广告发券缓存. <br/>
 * Date: 2018年7月11日 下午4:50:44 <br/>
 * 
 * @author chencheng
 * @version
 * @since JDK 1.8
 * @see
 */
@Component
public class NewAppTestCacheService extends BaseCacheService {

    @Resource
    private ExecutorService       executorService;
    @Autowired
    private AdvertRealDataService advertRealDataService;

    @Autowired
    private NewAppDAO newAppDAO;
    
    @Autowired
    private NewAppAdvertTradeDAO newAppAdvertTradeDAO;
    
    @Autowired
    private AdvertNewTradeDAO advertNewTradeDAO;
    
    @Autowired
    private ServiceManager serviceManager;
    
    @Autowired
    private AdvertNewTradeCacheService advertNewTradeCacheService;
    
    @Autowired
    private AdvertMapCacheManager advertMapCacheManager;
    
    @Autowired
    private AdvertExposeService advertExposeService;

    @Resource
    private RemoteMediaTagService remoteMediaTagService;

    @Autowired
    private NewAppTestToAdvertDAO newAppTestToAdvertDAO;

    @Autowired
    private RemoteNewMediaTestService remoteNewMediaTestService;

    @Autowired
    private RefreshCacheMqService refreshCacheMqService;

    /** 本地缓存, 可测试新媒体缓存key*/
    private static final String NEW_APP_IDS_TEST_KEY = "NEW_APP_IDS_TEST_KEY";
    
    /** 所有的可测试广告缓存key*/
    private static final String ALL_TEST_ADVERT_KEY = "ALL_TEST_ADVERT_KEY";

    /**
     * 新媒体id缓存
     */
    private final LoadingCache<String, Optional<Set<NewAppTestPriorityDO>>> newAppIdsCache = CacheBuilder.newBuilder()
            .initialCapacity(1000).refreshAfterWrite(5,TimeUnit.MINUTES).expireAfterWrite(1, TimeUnit.HOURS)
            .build(new CacheLoader<String, Optional<Set<NewAppTestPriorityDO>>>() {
                @Override
                public Optional<Set<NewAppTestPriorityDO>> load(String key) throws Exception {
                    return Optional.ofNullable(getTestNewAppId());
                }
                
                /**
                 * 
                 * getTestNewAppId:(获取可测试的媒体). <br/>
                 *
                 * @author chencheng
                 * @return
                 * @since JDK 1.8
                 */
                private Set<NewAppTestPriorityDO> getTestNewAppId() {
                    // 手动添加表中的待测试的新媒体
                    // 查询出来之后，catch  NewMediaTestEnum
                    List<NewMediaTestDto> newMediaTestDtos = new ArrayList<>();
                    try{
                        newMediaTestDtos = remoteNewMediaTestService.getAllTestingMedia();
                    }catch (Exception e){
                        logger.error("RemoteNewMediaTestService.getAllTestingMedia is error", e);
                    }

                    if (CollectionUtils.isEmpty(newMediaTestDtos)) {
                        return Sets.newHashSet();
                    }

                    // 过滤待测试媒体且是参与测试的媒体
                    newMediaTestDtos = newMediaTestDtos.stream()
                            .filter(dto -> dto.getTesStatus().equals(NewMediaTestEnum.StatusEnum.TESTING.getValue())
                                    && dto.getAppMark().equals(NewMediaTestEnum.AppMarkEnum.JOIN_TEST.getValue())).collect(Collectors.toList());

                    if (CollectionUtils.isEmpty(newMediaTestDtos)) {
                        return Sets.newHashSet();
                    }

                    //List<Long> appIds = newAppDAO.getAllNewAppId(); 这次版本不取回流表的了
                    // 测试周期内已经达到发券量的媒体, 因为已经手动添加,所以手动更改表中数据,所以不需要缓存了. 因为有可能再次添加.
                   /* List<Long> notTestAppIds = getRedisSetLongValue(RedisPreKey.KEY_NEW_APP_CANNOT_TEST);
                    if (CollectionUtils.isEmpty(notTestAppIds)) {
                        return appIds;
                    }*/
                    return newMediaTestDtos.stream()
                            .map(data -> new NewAppTestPriorityDO(data.getAppId(), data.getPriority())).collect(Collectors.toSet());
                }
                
                @Override
                public ListenableFuture<Optional<Set<NewAppTestPriorityDO>>> reload(String key, Optional<Set<NewAppTestPriorityDO>> oldValue) throws Exception {
                    ListenableFutureTask<Optional<Set<NewAppTestPriorityDO>>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });
    
    /**
     * 数据回流，可测试广告缓存
     */
    private final LoadingCache<String, Optional<List<Long>>> allTestAdvertCache = CacheBuilder.newBuilder()
            .initialCapacity(1000).expireAfterWrite(6,TimeUnit.MINUTES)
            .build(new CacheLoader<String, Optional<List<Long>>>() {
                @Override
                public Optional<List<Long>> load(String key) throws Exception {
                    String value = stringRedisTemplate.opsForValue().get(CacheKeyTool.getCacheKey(RedisCommonKeys.KC134));
                    if (StringUtils.isNotBlank(value)) {
                        return Optional.ofNullable(JSON.parseArray(value, Long.class));
                    }
                    return Optional.ofNullable(Lists.newArrayList());
                }
                
                
                @Override
                public ListenableFuture<Optional<List<Long>>> reload(String key, Optional<List<Long>> oldValue) throws Exception {
                    ListenableFutureTask<Optional<List<Long>>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });
    
    
    /**
     * 测试广告在已投媒体的对应
     */
    private final LoadingCache<String, Optional<Set<Long>>> appTestToAdvertCache = CacheBuilder.newBuilder()
            .initialCapacity(1000).expireAfterWrite(10, TimeUnit.MINUTES)
            .build(new CacheLoader<String, Optional<Set<Long>>>() {
                @Override
                public Optional<Set<Long>> load(String key) throws Exception {
                    List<NewAppTestToAdvertDO> appTestToAdvert = getAllAppTestToAdvert(Collections.singletonList(key));
                    if (CollectionUtils.isEmpty(appTestToAdvert)) {
                        Map<String, Optional<Set<Long>>> map = new HashMap();
                        return map.put(key, Optional.ofNullable(Sets.newHashSet()));
                    }

                    return Optional.ofNullable(appTestToAdvert.stream().map(ad -> ad.getAppId()).collect(Collectors.toSet()));
                }

                @Override
                public Map<String, Optional<Set<Long>>> loadAll(Iterable<? extends String> keys) {
                    List<String> ks = Lists.newArrayList(keys);

                    //获取广告->已投媒体关系集合
                    List<NewAppTestToAdvertDO> appTestToAdvert = getAllAppTestToAdvert(ks);

                    Map<String, Optional<Set<Long>>> map = new HashMap<>();

                    if (CollectionUtils.isEmpty(appTestToAdvert)) {
                        ks.stream().forEach(key -> {
                            map.put(key, Optional.ofNullable(Sets.newHashSet()));
                        });
                        return map;
                    }

                    // advertId->list
                    Map<Long, List<NewAppTestToAdvertDO>> listMap = appTestToAdvert.stream().collect(Collectors.groupingBy(NewAppTestToAdvertDO::getAdvertId));

                    ks.forEach(key -> {
                        // 取key
                        Long advertId = Long.parseLong(key.split("_")[1]);

                        List<NewAppTestToAdvertDO> list = Optional.ofNullable(listMap.get(advertId)).orElse(Lists.newArrayList());
                        if (CollectionUtils.isEmpty(list)) {
                            map.put(key, Optional.ofNullable(Sets.newHashSet()));
                        } else {
                            map.put(key, Optional.ofNullable(list.stream().map(ad -> ad.getAppId()).collect(Collectors.toSet())));
                        }

                    });

                    return map;

                }

                @Override
                public ListenableFuture<Optional<Set<Long>>> reload(String key, Optional<Set<Long>> oldValue) throws Exception {
                    ListenableFutureTask<Optional<Set<Long>>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });
    
    private List<NewAppTestToAdvertDO> getAllAppTestToAdvert(List<String> cacheKeys) {

        if(CollectionUtils.isEmpty(cacheKeys)){
            return Lists.newArrayList();
        }

        List<Long> advertIds = new ArrayList<>();
        cacheKeys.forEach(key -> {
            // cacheKey:当前日期_advertId, eg:2019-08-08_12456
            Long advertId = Long.parseLong(key.split("_")[1]);
            advertIds.add(advertId);
        });

        // 获取广告已投媒体
        List<NewAppTestToAdvertDO> list = newAppTestToAdvertDAO.selectCurDateNewAppTest(advertIds);
        if (CollectionUtils.isEmpty(list)) {
            return Lists.newArrayList();
        }
        // 媒体对应的广告id
        return list;
    }

    /**
     * 新媒体不可测试行业缓存
     */
    private final LoadingCache<Long, Optional<List<String>>> newAppNotTestTradeCache = CacheBuilder.newBuilder()
            .initialCapacity(1000).expireAfterWrite(30,TimeUnit.MINUTES)
            .build(new CacheLoader<Long, Optional<List<String>>>() {
                @Override
                public Optional<List<String>> load(Long key) throws Exception {
                    return Optional.ofNullable(getRedisNewAppNotTestTrade(key));
                }

                @Override
                public ListenableFuture<Optional<List<String>>> reload(Long key, Optional<List<String>> oldValue) throws Exception {
                    ListenableFutureTask<Optional<List<String>>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });
    
    /**
     * 
     * getnewAppNotTestTrade:(从redis获取媒体不再测试广告行业缓存). <br/>
     *
     * @author chencheng
     * @param appId
     * @return
     * @since JDK 1.8
     */
    private List<String> getRedisNewAppNotTestTrade(Long appId) {
        // 测试周期内不在媒体上测试的行业
        List<String> notTestTrade = getRedisSetStringValue(CacheKeyTool.getCacheKey(RedisPreKey.KEY_NEW_APP_CANNOT_TEST_TRADE, appId));
        // 今日不再测试的行业
        String value = stringRedisTemplate.opsForValue().get(CacheKeyTool.getCacheKey(RedisPreKey.KEY_NEW_APP_TODAY_CANNOT_TEST_TRADE, appId));
        if (StringUtils.isNotBlank(value)) {
            List<String> todayNotTestTrade = JSON.parseArray(value, String.class);
            notTestTrade.addAll(todayNotTestTrade);
        }
        return notTestTrade;
    }


    /**
     * 新媒体不可测试广告缓存
     */
    private final LoadingCache<Long, Optional<List<Long>>> newAppNotTestAdvertCache = CacheBuilder.newBuilder()
            .initialCapacity(1000).expireAfterWrite(1,TimeUnit.HOURS)
            .build(new CacheLoader<Long, Optional<List<Long>>>() {
                @Override
                public Optional<List<Long>> load(Long key) throws Exception {
                    String appKey = CacheKeyTool.getCacheKey(RedisPreKey.KEY_NEW_APP_CANNOT_TEST_ADVERT, key);
                    return Optional.ofNullable(getRedisSetLongValue(appKey));
                }
                
                @Override
                public ListenableFuture<Optional<List<Long>>> reload(Long key, Optional<List<Long>> oldValue) throws Exception {
                    ListenableFutureTask<Optional<List<Long>>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });
    
    /**
     * 新媒体投放的广告行业优先排序
     */
    private final LoadingCache<String, Optional<List<NewAppAdvertTradeDO>>> newAppAdvertTradeCache = CacheBuilder.newBuilder()
            .initialCapacity(1000).expireAfterWrite(2,TimeUnit.HOURS)
            .build(new CacheLoader<String, Optional<List<NewAppAdvertTradeDO>>>() {
                @Override
                public Optional<List<NewAppAdvertTradeDO>> load(String key) throws Exception {
                    return Optional.ofNullable(newAppAdvertTradeDAO.getAppAdvertTrade(key));
                }
                
                @Override
                public ListenableFuture<Optional<List<NewAppAdvertTradeDO>>> reload(String key, Optional<List<NewAppAdvertTradeDO>> oldValue) throws Exception {
                    ListenableFutureTask<Optional<List<NewAppAdvertTradeDO>>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });

    
    /**
     * 
     * newAppIdsCacheRefresh:(可测试的新媒体缓存更新). <br/>
     *
     * @author chencheng
     * @since JDK 1.8
     */
    public void allTestAdvertCacheRefresh(){
        allTestAdvertCache.refresh(ALL_TEST_ADVERT_KEY);
    }
    

    
    /**
     * 
     * newAppIdsCacheRefresh:(可测试的新媒体缓存更新). <br/>
     *
     * @author chencheng
     * @since JDK 1.8
     */
    public void newAppIdsCacheRefresh(){
        newAppIdsCache.refresh(NEW_APP_IDS_TEST_KEY);
    }
    
    /**
     * 
     * appTestToAdvertCacheRefresh:(所有测试媒体对应的已投放广告关系缓存更新). <br/>
     *
     * @author chencheng
     * @param advertId
     * @since JDK 1.8
     */
    public void appTestToAdvertCacheRefresh(Long advertId){
        String cacheKey = DateUtils.getDayStr(new Date())+"_"+advertId;
        appTestToAdvertCache.refresh(cacheKey);
    }
    
    
    /**
     * 
     * newAppNotTestTradeCacheRefresh:(新媒体上不能继续测试的行业缓存更新). <br/>
     *
     * @author chencheng
     * @param appId
     * @since JDK 1.8
     */
    public void newAppNotTestTradeCacheRefresh(Long appId){
        newAppNotTestTradeCache.refresh(appId);
    }
    

    /**
     * 
     * newAppNotTestAdvertCacheRefresh:(新媒体不可测试广告缓存). <br/>
     *
     * @author chencheng
     * @since JDK 1.8
     */
    public void newAppNotTestAdvertCacheRefresh(Long appId){
        newAppNotTestAdvertCache.refresh(appId);
    }
    
    /**
     * 
     * getNewAppIds:(获取可测试的新媒体). <br/>
     *
     * @author chencheng
     * @return
     * @since JDK 1.8
     */
    public List<Long> getNewAppIds() {
        try {

            Set<NewAppTestPriorityDO> newAppList = newAppIdsCache.get(NEW_APP_IDS_TEST_KEY).orElse(Sets.newHashSet());

            List<Long> newAppIds = newAppList.stream().map(NewAppTestPriorityDO::getAppId).collect(Collectors.toList());
            return newAppIds;
        } catch (ExecutionException e) {
            logger.error("getNewAppIds error", e);
            return Lists.newArrayList();
        }
    }

    /**
     * 获取所有测试媒体的优先级
     * @return
     */
    private Map<Long, Integer> getNewAppPriority() {

        try {
            Set<NewAppTestPriorityDO> newAppList = newAppIdsCache.get(NEW_APP_IDS_TEST_KEY).orElse(Sets.newHashSet());

            Map<Long, Integer> map = new HashMap<>(newAppList.size());

            if (CollectionUtils.isEmpty(newAppList)){
                return map;
            }

            newAppList.forEach(dto -> map.put(dto.getAppId(), Optional.ofNullable(dto.getPriority()).orElse(0)));

            return map;
        } catch (ExecutionException e) {
            logger.error("getNewAppPriority error", e);
            return Maps.newHashMap();
        }


    }

    /**
     * 获取所有测试媒体的优先级,给其他类用的
     * @return
     */
    public Map<Long, Integer> getNewAppPriorityByPub(){

        return getNewAppPriority();
    }
    

    /**
     * 
     * getAllTestAdvert:(获取所有的新媒体可测试的广告). <br/>
     *
     * @author chencheng
     * @return
     * @since JDK 1.8
     */
    public List<Long> getAllTestAdvert() {
        try {
            return allTestAdvertCache.get(ALL_TEST_ADVERT_KEY).orElse(Lists.newArrayList());
        } catch (ExecutionException e) {
            logger.error("getAllTestAdvert error", e);
            return Lists.newArrayList();
        }
    }
   
    
    /**
     * 
     * getAllTestAdvert:(获取广告已投媒体). <br/>
     *
     * @author chencheng
     * @return
     * @since JDK 1.8
     */
    public Set<Long> getAppTestToAdvert(Long advertId) {
        try {

            String cacheKey = DateUtils.getDayStr(new Date())+"_"+advertId;

            Set<Long> listAppTestToAdvert = appTestToAdvertCache.get(cacheKey).orElse(Sets.newHashSet());

            if (CollectionUtils.isEmpty(listAppTestToAdvert)) {
                return Sets.newHashSet();
            }
            return listAppTestToAdvert;
        } catch (ExecutionException e) {
            logger.error("getAllTestAdvert error", e);
            return Sets.newHashSet();
        }
    }
    
    
    /**
     * 
     * getAdvertToAppMap:(获取广告所投媒体数). <br/>
     *
     * @author chencheng
     * @param advertIds
     * @return
     * @since JDK 1.8
     */
    public Map<String, Optional<Set<Long>>> getAdvertToAppMap(List<Long> advertIds){
        try {

            List<String> cacheKeys = new ArrayList<>();

            advertIds.forEach(advertId ->{
                String cacheKey = DateUtils.getDayStr(new Date())+"_"+advertId;
                cacheKeys.add(cacheKey);
            });

            Map<String, Optional<Set<Long>>> appTestToAdvertMap = appTestToAdvertCache.getAll(cacheKeys);
            return  appTestToAdvertMap;
        } catch (Exception e) {
            logger.error("getOtherAppAdvertIds error", e);
            return Maps.newHashMap();
        }
    }

    
    /**
     * 
     * getNewAppNotTestAdvertIds:(获取新媒体上不能继续测试的行业). <br/>
     *
     * @author chencheng
     * @param appId
     * @return
     * @since JDK 1.8
     */
    public List<String> getNewAppNotTestTradeIds(Long appId){
        try {
            return newAppNotTestTradeCache.get(appId).orElse(Lists.newArrayList());
        } catch (Exception e) {
            logger.error("getNewAppNotTestTradeIds error", e);
            return Lists.newArrayList();
        }
    }
    

    
    /**
     * 
     * getNewAppNotTestAdvertIds:(新媒体当日不可投的广告). <br/>
     *
     * @author chencheng
     * @param appId
     * @return
     * @since JDK 1.8
     */
    public List<Long> getNewAppNotTestAdvertIds(Long appId){
        try {
            return newAppNotTestAdvertCache.get(appId).orElse(Lists.newArrayList());
        } catch (Exception e) {
            logger.error("getNewAppNotTestAdvertIds error", e);
            return Lists.newArrayList();
        }
    }
    
    
    /**
     * 
     * getAppTestToAdvert:(新媒体投放的广告行业). <br/>
     *
     * @author chencheng
     * @param appTradeName
     * @return
     * @since JDK 1.8
     */
    public List<NewAppAdvertTradeDO> getNewAppAdvertTrade(String appTradeName) {
        try {
            return newAppAdvertTradeCache.get(appTradeName).orElse(Lists.newArrayList());
        } catch (ExecutionException e) {
            logger.error("getNewAppAdvertTrade error", e);
            return Lists.newArrayList();
        }
    }
    
    
    /**
     * 
     * getRedisSetLongValue:(从redis缓存获取set). <br/>
     *
     * @author chencheng
     * @param key
     * @return
     * @since JDK 1.8
     */
    private List<String> getRedisSetStringValue(String key) {
        Set<String> value = stringRedisTemplate.opsForSet().members(key);
        if (null == value) {
            return Lists.newArrayList();
        }
        return new ArrayList<>(value);
    }
    
    /**
     * 
     * getRedisSetLongValue:(从redis缓存获取set). <br/>
     *
     * @author chencheng
     * @param key
     * @return
     * @since JDK 1.8
     */
    private List<Long> getRedisSetLongValue(String key) {
        Set<String> value = stringRedisTemplate.opsForSet().members(key);
        if (null == value) {
            return Lists.newArrayList();
        }
        return value.stream().map(s-> Long.parseLong(s)).collect(Collectors.toList());
    }

    
    /**
     * 
     * ifTestNewApp:(是否属于可测试新媒体). <br/>
     *
     * @author chencheng
     * @param appId
     * @return
     * @since JDK 1.8
     */
    public boolean ifTestNewApp(Long appId){
        List<Long> newAppIds = getNewAppIds();
        if (newAppIds.contains(appId)) {
            return true;
        }
        return false;
    };
    
    /**
     * 
     * doNewAppTestAdvert:(新媒体试投). <br/>
     *
     * @author chencheng
     * @param req
     * @param rsp
     * @param filterResult
     * @param validAdverts
     * @return
     * @since JDK 1.8
     */
    public boolean doNewAppTestAdvert(ObtainAdvertReq req, ObtainAdvertRsp rsp, FilterResult filterResult,
                                      Map<Long, AdvertFilterVO> validAdverts, AdvertFilter advertFilter) {
        try {
            DBTimeProfile.enter("doNewAppTestAdvert");
            Long appId = req.getAppId();
            // 1.是否属于可测试新媒体
            if (!ifTestNewApp(appId)) {
                return false;
            }
            filterResult.setNewAppTestType(NewAppAdvertTradeDO.APP_TYPE_NEW_TEST.toString());
            // 2.是否有测试广告,并排序
            List<AdvertFilterVO> listTestAdvert = testNewAppAdvert(appId, validAdverts);
            if (CollectionUtils.isEmpty(listTestAdvert)) {
                filterResult.setNewAppTestType(NewAppAdvertTradeDO.APP_TYPE_NEW_NOT_TEST.toString());
                return false;
            }
            // 3.出券
            doObtainNewAppTestAdvert(req, rsp, listTestAdvert, filterResult, advertFilter);
            if (!rsp.isResult()) {
                filterResult.setNewAppTestType(NewAppAdvertTradeDO.APP_TYPE_NEW_NOT_TEST.toString());
            }else {
                CatUtil.catLog(CatGroupEnum.CAT_102021.getCode());
            }
            return rsp.isResult();
        } catch (Exception e) {
            logger.error("doNewAppTestAdvert error, appId=[{}]", req.getAppId(), e);
            filterResult.setNewAppTestType(NewAppAdvertTradeDO.APP_TYPE_NEW_NOT_TEST.toString());
            return false;
        }finally {
            DBTimeProfile.release();
        }
    }
    
   
    /**
     * 
     * testNewAppAdvert:(获取媒体可投的广告). <br/>
     *
     * @author chencheng
     * @param appId 媒体id
     * @param validAdverts 有效广告
     * @return
     * @since JDK 1.8
     */
    @SuppressWarnings("squid:S3776")
    public List<AdvertFilterVO> testNewAppAdvert(Long appId, Map<Long, AdvertFilterVO> validAdverts){
        try {
            DBTimeProfile.enter("testNewAppAdvert");
            // 获取所有的新媒体可测试的广告
            List<Long> allTestAdvertids = getAllTestAdvert();
            if (CollectionUtils.isEmpty(allTestAdvertids)) {
                return Lists.newArrayList();
            }
            // 在其他媒体的已发券的广告
            Map<String, Optional<Set<Long>>> advertToAppMap = getAdvertToAppMap(allTestAdvertids);

            // 当日媒体不可发的广告
            List<Long> notTestAdvertIds = getNewAppNotTestAdvertIds(appId);

            // 获取广告可测媒体个数
            Integer adLimitAppSize = serviceManager.getIntValue(SystemConfigKeyConstant.NEW_APP_TEST_ADVERT_LIMIT_APP_SIZE);

            // 获取媒体的优先级
            Map<Long, Integer> newAppPriorityMap = this.getNewAppPriority();

            // 剔除不可发的行业，剔除无效的广告，按照广告行业分组,分组内按照广告最低出价，从高到低排序
            List<String> listNewAppNotTestTrade = getNewAppNotTestTradeIds(appId);
            Map<String, List<AdvertFilterVO>> tradeAdvertMap = allTestAdvertids.stream()
                .filter(id -> this.filterAppByAppLimit(appId, id, adLimitAppSize, advertToAppMap, newAppPriorityMap))
                .filter(id -> !notTestAdvertIds.contains(id))
                .map(id -> advertNewTradeCacheService.getAdvertNewTradeName(id))
                .filter(dto -> dto != null)
                .filter(dto -> !listNewAppNotTestTrade.contains(dto.getNewTradeName()))
                .filter(dto -> validAdverts.get(dto.getAdvertId()) != null)
                .collect(Collectors.groupingBy(AdvertNewTradeDO::getNewTradeName,
                                               Collectors.collectingAndThen(Collectors.toList(),
                                                                            list1 -> orderyAdvertTrade(list1, validAdverts))));
            if (tradeAdvertMap.isEmpty()) {
                return sortAdvertMinFee(tradeAdvertMap);
            }
            // 获取媒体行业标签
            MediaTagDto mediaTagDto = remoteMediaTagService.queryMediaTagByAppIdType(appId, MediaTagDto.INDUSTRY_BLOCK);
            if (mediaTagDto == null || mediaTagDto.getTitle() == null) {
                return sortAdvertMinFee(tradeAdvertMap);
            }
            // 按照媒体行业排序规则查询
            List<NewAppAdvertTradeDO> listTrade = getNewAppAdvertTrade(mediaTagDto.getTitle());
            List<AdvertFilterVO> listAdvertFilterVO = Lists.newArrayList();
            if (CollectionUtils.isEmpty(listTrade)) {
                return sortAdvertMinFee(tradeAdvertMap);
            }
            // 先按照已有的排序
            listTrade.forEach(trade -> {
                if(StringUtils.isNotBlank(trade.getAdvertTrade())){
                    List<AdvertFilterVO>list = tradeAdvertMap.get(trade.getAdvertTrade());
                    if(list != null){
                        listAdvertFilterVO.addAll(list);
                    }
                }
            });
            // 其他行业按照价格排序
            Map<String, List<AdvertFilterVO>> tradeOtherAdvertMap = tradeAdvertMap.entrySet().stream()
            .filter(entity -> null != entity && !listTrade.contains(entity.getKey()) && null != entity.getValue())
                    .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue, (oldVal, newVal) -> newVal));
            listAdvertFilterVO.addAll(sortAdvertMinFee(tradeOtherAdvertMap));
            return listAdvertFilterVO;
        }catch (Exception e){
            logger.error("NewAppTestCacheService.testNewAppAdvert is error", e);
            return new ArrayList<>();
        } finally {
            DBTimeProfile.release();
        }
    
    }


    /**
     * 是否可以在请求的媒体上投放广告
     *
     * @param reqAppId 请求的媒体ID
     * @param advertId 请求的广告ID
     * @param adLimitAppSize 限制可以投放的媒体数
     * @param advertToAppMap 广告->已投放媒体
     * @param newAppPriorityMap 媒体的优先级
     * @return true-保留，false-过滤
     */
    private boolean filterAppByAppLimit(Long reqAppId, Long advertId, Integer adLimitAppSize, Map<String, Optional<Set<Long>>> advertToAppMap, Map<Long, Integer> newAppPriorityMap){

        String cacheKey = DateUtils.getDayStr(new Date())+"_"+advertId;

        // 获取该广告已经投放媒体
        Set<Long> appSet = advertToAppMap.get(cacheKey).orElse(Sets.newHashSet());
        // 如果小于可测媒体个数或者已经投放,则保留
        if(appSet.contains(reqAppId) || appSet.size() < adLimitAppSize){
            return true;
        }

        Integer reqPriority =  newAppPriorityMap.get(reqAppId);

        // 如果已经超过了限制媒体个数， 且请求的媒体是普通测试媒体，则直接过滤
        if (NewMediaTestEnum.PriorityEnum.NORMAL.getValue().equals(reqPriority)) {
            return false;
        }

        // 请求媒体是优先测试媒体，则：
        // 遍历已投媒体是否存在普通媒体
        // 如果已经投的媒体都是优先级媒体，则直接过滤,不是，则保留

        return appSet.stream().anyMatch(appId -> NewMediaTestEnum.PriorityEnum.NORMAL.getValue().equals(newAppPriorityMap.get(appId)));

    }

    /**
     * 
     * sortAdvertMinFee:(所有的配置按照最低价，从高到低排序). <br/>
     *
     * @author chencheng
     * @param tradeAdvertMap
     * @return
     * @since JDK 1.8
     */
    private List<AdvertFilterVO> sortAdvertMinFee(Map<String, List<AdvertFilterVO>> tradeAdvertMap) {
        if (tradeAdvertMap.isEmpty()) {
            return Lists.newArrayList();
        }
        return tradeAdvertMap.entrySet().stream().map(map -> map.getValue()).flatMap(List::stream)
        .filter(dto -> dto.getMinFee10PriceVO()!= null)
        .sorted(comparing(AdvertFilterVO::getMinFee10Price).reversed())
        .collect(toList());
    }

    /**
     * 
     * orderyAdvertTrade:(广告最低出价，从高到低排序). <br/>
     *
     * @author chencheng
     * @param list
     * @param validAdverts
     * @return
     * @since JDK 1.8
     */
    private List<AdvertFilterVO> orderyAdvertTrade(List<AdvertNewTradeDO> list, Map<Long, AdvertFilterVO> validAdverts) {
        try {
            Integer appSize = Integer.valueOf(serviceManager.getStrValue(SystemConfigKeyConstant.NEW_APP_TEST_ORIENTATION_APP_SIZE_KEY));
            return list.stream().map(dto -> {
                AdvertFilterVO advertFilterVO = validAdverts.get(dto.getAdvertId());
                advertFilterVO.setNewTradeName(dto.getNewTradeName());
                return advertFilterVO;
            }).filter(dto -> dto.setMinFee10PriceVO(dto.getTargetedAppMap(), appSize) && dto.getMinFee10PriceVO() != null)
               .sorted(comparing(AdvertFilterVO::getMinFee10Price).reversed())
               .collect(Collectors.toList());
        } catch (Exception ex) {
            logger.error("[OrderAdvertTrade] error list:{}", JSON.toJSONString(list), ex);
            throw new TuiaRuntimeException();
        }
    }
   
    
    /**
     * 
     * doObtainNewAppTestAdvert:(新媒体试投发券). <br/>
     *
     * @author chencheng
     * @param req
     * @param rsp
     * @param advertFilterVOList
     * @param filterResult
     * @since JDK 1.8
     */
    @SuppressWarnings("squid:S3776")
    private void doObtainNewAppTestAdvert(ObtainAdvertReq req, ObtainAdvertRsp rsp,
                                          List<AdvertFilterVO> advertFilterVOList,
                                          FilterResult filterResult, AdvertFilter advertFilter) {
        try {
            DBTimeProfile.enter("doObtainNewAppTestAdvert");
            // 获取"有效"状态广告id列表 移除被重复曝光过滤掉的广告
            advertFilterVOList = advertFilterVOList.stream().filter(x -> !filterResult.getMultipleExposureIds().contains(x.getAdvertId())).collect(toList());

            for (AdvertFilterVO advertFilterVO : advertFilterVOList) {
                Long advertId = advertFilterVO.getAdvertId();
                AdvertVO advertVO = advertMapCacheManager.getAdvertCache(advertId);
                if (advertVO == null) {
                    logger.info("serviceManager.getAdvert error, advertVO is null the advertId = [{}]", advertId);
                    filterResult.addFilterId(ErrorCode.E0500008.getErrorCode(), null, advertId);
                    continue;
                }
                //获取之前得到的价格最低配置
                AdvertPriceVO advertPriceVO = advertFilterVO.getMinFee10PriceVO();
                if (advertPriceVO == null) {
                    filterResult.addFilterId(ErrorCode.E0500030.getErrorCode(), null, advertId);
                    continue;
                }
                // 配置超过最高出价，以最高出价来出
                filterResult.setNewAppTestBeforeFee(0L);
                if (advertPriceVO.getFee().longValue() > AdvertFilterVO.NEW_APP_TEST_MAX_FEE) {
                    filterResult.setNewAppTestBeforeFee(advertPriceVO.getFee());
                    advertPriceVO.setFee(AdvertFilterVO.NEW_APP_TEST_MAX_FEE);
                }
                try {
                    DBTimeProfile.enter("finishBiz");

                    filterResult.setNewTradeName(advertFilterVO.getNewTradeName());
                    filterResult.setType(FilterResultTypes.NEW_APP_TEST_ADVERT_TYPE);
                    advertExposeService.finishBiz(req, rsp, advertId, advertVO, filterResult, advertPriceVO, advertPriceVO.getFee(), advertFilter);
                } catch (TuiaException ex) {
                    logger.warn("finishBiz happen error, duibaOrderId={}, type={}", req.getOrderId(), filterResult.getType(), ex);
                    // 过滤掉之后需要添加到过滤结果中
                    filterResult.addFilterId(ex.getResultCode(), null, advertId);
                    if(ex.getResultCode().equals(ErrorCode.E0500006.getErrorCode())){break;}
                    else { continue;}
                } finally {
                    DBTimeProfile.release();
                }

                if (rsp.isResult()) {
                    filterResult.setResultCode(ErrorCode.E0000000.getErrorCode());
                    filterResult.setSuccessId(advertId);
                    filterResult.setAdvertOrderId(rsp.getAdvertOrderId());
                    //出价
                    filterResult.setFee(advertPriceVO.getFee());
                    //计费方式
                    filterResult.setChargeType(ChargeTypeEnum.getByCode(advertPriceVO.getChargeType()).getDesc());
                    //包的类型 1-人工生成的包 2-智能生成的策略包
                    filterResult.setPackageType(advertPriceVO.getPackageType());
                    //记录活动类型
                    filterResult.setActivitySceneType(advertPriceVO.getActivityType());
                    // 中券成功的情况需要记录过滤结果
                    filterResult.setPlanId(advertPriceVO.getAdvertOrientationPackageId());
                    break;
                }
            }
        } finally {
            DBTimeProfile.release();
        }
        
    }
    
    /**
     * 
     * checkAdvertTestToApp:(检查是否更新媒体与投放广告的关联关系). <br/>
     *
     * @author chencheng
     * @param appId
     * @param advertId
     * @since JDK 1.8
     */
    public Boolean checkAdvertTestToApp(Long appId, Long advertId){
       Set<Long> appIds = getAppTestToAdvert(advertId);
        return appIds.contains(appId);
    }

    /**
     * 获取广告已经投放的媒体
     * @param advertId
     * @return
     */
    public Set<Long> getAdvertTestToApps(Long advertId){
        return getAppTestToAdvert(advertId);
    }

    /**
     * 
     * sendUpdateNewAppTestAdvertCacheMsg:(发送更新媒体的投放广告的关系消息). <br/>
     * @author chencheng
     * @param advertId
     * @since JDK 1.8
     */
    public void sendUpdateNewAppTestAdvertCacheMsg(Long appId, Long advertId){
        refreshCacheMqService.updateNewAppTestCacheMsg(appId, advertId, UpdateNewAppTestCacheMsg.TYPE_NEW_APP_TEST_ALL_ADVERT_CACHE_OTHER);
    }
    
    /**
     * 
     * initAdvertTestToApp:(所有媒体的投放广告的关系初始化). <br/>
     *
     * @author chencheng
     * @since JDK 1.8
     */
    public void initAdvertTestToApp() {
        
        List<Long> newAppIds = getNewAppIds();
        if (CollectionUtils.isEmpty(newAppIds)) {
            return;
        }
        // 广告->媒体list
        List<NewAppTestToAdvertDO> newAppTestToAdvertDOS = newAppTestToAdvertDAO.selectCurDateNewAppTest(Lists.newArrayList());

        if(CollectionUtils.isEmpty(newAppTestToAdvertDOS)){
            return;
        }

        Map<Long, List<NewAppTestToAdvertDO>> listMap = newAppTestToAdvertDOS.stream().collect(Collectors.groupingBy(NewAppTestToAdvertDO::getAdvertId));
        listMap.keySet().stream().forEach(advertId -> {
            String cacheKey = DateUtils.getDayStr(new Date())+"_"+advertId;
            appTestToAdvertCache.put(cacheKey, Optional.ofNullable(listMap.get(advertId).stream().map(NewAppTestToAdvertDO::getAppId).collect(Collectors.toSet())));
        });

    }
   
}
