/**
 *
 */
package cn.com.duiba.tuia.cache;

import cn.com.duiba.tuia.constants.ErrorCode;
import cn.com.duiba.tuia.constants.TuiaServiceConfig;
import cn.com.duiba.tuia.core.api.enums.advert.AdvertPkgPeriodTypeEnum;
import cn.com.duiba.tuia.core.api.enums.permisson.DataPermissonSourceTypeEnum;
import cn.com.duiba.tuia.dao.account.AccountDAO;
import cn.com.duiba.tuia.dao.account.SpecialAccountAppDAO;
import cn.com.duiba.tuia.dao.advert.*;
import cn.com.duiba.tuia.dao.advert_tag.AdvertTagDAO;
import cn.com.duiba.tuia.dao.apppackage.AppPackageDAO;
import cn.com.duiba.tuia.dao.engine.AdvertOrderDAO;
import cn.com.duiba.tuia.dao.engine.PromoteUrlJsDAO;
import cn.com.duiba.tuia.dao.engine.SystemConfigDAO;
import cn.com.duiba.tuia.dao.material.AdvertMaterialRealtionService;
import cn.com.duiba.tuia.dao.permisson.DataPermissonDAO;
import cn.com.duiba.tuia.dao.promotetest.AdvertPromoteTestDAO;
import cn.com.duiba.tuia.dao.resource_tags.ResourceTagsDAO;
import cn.com.duiba.tuia.dao.shunt_config.ShuntConfigDAO;
import cn.com.duiba.tuia.domain.dataobject.*;
import cn.com.duiba.tuia.domain.model.*;
import cn.com.duiba.tuia.domain.vo.AdvertVO;
import cn.com.duiba.tuia.domain.vo.GlobalConfigFlowVO;
import cn.com.duiba.tuia.enums.AccountSourceTypeEnum;
import cn.com.duiba.tuia.enums.AdvertSortTypeEnum;
import cn.com.tuia.advert.enums.ResourceTagsTypeEnum;
import cn.com.duiba.tuia.exception.TuiaException;
import cn.com.duiba.tuia.service.*;
import cn.com.duiba.tuia.ssp.center.api.dto.advertmonitor.ActivityAdvert4MonitorDto;
import cn.com.duiba.tuia.ssp.center.api.remote.RemoteActivityBackendService;
import cn.com.duiba.tuia.tool.StringTool;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.tuia.advert.cache.CacheKeyTool;
import cn.com.tuia.advert.cache.RedisCommonKeys;
import cn.com.tuia.advert.constants.CommonConstant;
import cn.com.tuia.advert.constants.SystemConfigKeyConstant;
import cn.com.tuia.advert.enums.AdxSceneEnum;
import cn.com.tuia.advert.enums.CurrentMainStatusEnum;
import cn.com.tuia.advert.model.Period;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Splitter;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import java.util.stream.Collectors;


/**
 * @author sunjiangrong
 */
@Component
public class ServiceManager extends BaseCacheService {


    private static final List<Period> NOT_LIMIT_PERIOD = Lists.newArrayList(new Period("00:00","24:00"));

    private static final List<Period> EMPTY_PERIOD = Lists.newArrayList(new Period("00:00","00:00"));

    // 正常订单表
    private static final String ORDER_TABLE = "tuia_order_info";

    // 补打日志订单表
    private static final String ORDER_RELOG_TABLE = "tuia_relog_log_info";

    @Resource
    private AdvertDAO advertDAO;

    @Autowired
    private AdvertExpandDAO advertExpandDAO;

    @Resource
    private AdvertOrderDAO advertOrderDAO;

    @Resource
    private SystemConfigDAO systemConfigDAO;

    @Resource
    private ShuntConfigDAO shuntConfigDAO;

    @Autowired
    private SpecialAccountAppDAO specialAccountAppDAO;

    @Autowired
    private AppPackageDAO appPackageDAO;

    @Autowired
    private AdvertMaterialRealtionService advertMaterialRealtionService;

    /**
     * The advert tag dao.
     */
    @Resource
    private AdvertTagDAO advertTagDAO;

    /**
     * The account dao.
     */
    @Resource
    private AccountDAO accountDAO;

    @Resource
    private ExecutorService executorService;
    @Resource
    private RemoteActivityBackendService          remoteActivityBackendService;

    @Value("${engine.click.limit}")
    private int CLICK_LIMIT;

    @Value("${engine.click.limit.time}")
    private int CLICK_LIMIT_TIME;

    @Autowired
    private AdvertPeriodService advertPeriodService;

    @Resource
    private ResourceTagsDAO resourceTagsDAO;

    @Resource
    private DataPermissonDAO dataPermissonDAO;
    
    @Resource
    private ValidAdvertService validAdvertService;

    @Autowired
    private AdvertCouponBackDAO advertCouponBackDAO;

    @Autowired
    private GlobalConfigFlowService globalConfigFlowService;
    
    @Autowired
    private AdvertPromoteTestDAO advertPromoteTestDAO;

    @Autowired
    private AdvertSdkDao advertSdkDao;

    @Autowired
    private PromoteUrlJsDAO promoteUrlJsDAO;

    @Resource
    private MakeTagCacheService makeTagCacheService;

    private static final String VALID_ADVERT_IDS_CACHE_KEY = "validAdvertIds";

    //全局流量配置类型为微信加粉
    private static final String WEI_XIN_JIA_FEN="04.01.0001";
    //全局流量类型为权重
    private static final Integer IS_ADVERT_WEIGHT=1;

    /** 托管底价 */
    private static final String OCPC_CLICK_MIN_PRICE = "ocpc.click.min.price";
    /** 托管底价白名单开关状态 */
//    private static final String OCPC_BASEPRICE_CONTROL_ENABLESTATE = "tuia.ocpc.baseprice.control.enablestate";
    /** 托管底价白名单开启状态 */
    public static final String OCPC_BASEPRICE_CONTROL_OPEN = "1";
    /** 托管底价白名单关闭状态 */
    public static final String OCPC_BASEPRICE_CONTROL_CLOSE = "0";

    public static final Long DEFAULT_BASE_PRICE = 15L;
    //未接入js
    public static final String NO_JS = Boolean.toString(false);
    //接入js
    public static final String HAS_JS = Boolean.toString(true);

    private final Pattern patternReplace = Pattern.compile(":");

    private final LoadingCache<String, Optional<String>> localCache = CacheBuilder.newBuilder()
            .initialCapacity(1000).refreshAfterWrite(5, TimeUnit.MINUTES).expireAfterWrite(2,TimeUnit.HOURS)
            .build(new CacheLoader<String, Optional<String>>() {
                @Override
                public Optional<String> load(String key) throws Exception {
                    return Optional.ofNullable(getStrValueInner(key));
                }
                
                @Override
                public ListenableFuture<Optional<String>> reload(String key, Optional<String> oldValue) throws Exception {
                    ListenableFutureTask<Optional<String>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });
    
    /**
     * 有效广告对应排序map本地缓存 有消息同步，不用设置过期时间来保证避免读取旧值
     */
    private final LoadingCache<String, Map<Long, Integer>> validAdvertIdsSortCache = CacheBuilder.newBuilder().initialCapacity(1).
            recordStats().refreshAfterWrite(1, TimeUnit.MINUTES).build(new CacheLoader<String, Map<Long, Integer>>() {
        @Override
        public Map<Long, Integer> load(String key) throws Exception {
            return getValidAdvertOrderMap();
        }

        @Override
        public ListenableFuture<Map<Long, Integer>> reload(String key, Map<Long, Integer> oldValue) throws Exception {
            ListenableFutureTask<Map<Long, Integer>> task = ListenableFutureTask.create(() -> load(key));
            executorService.submit(task);
            return task;
        }

    });

    public void deleteValidAdvertIdsSortCache() {
        //清除缓存
        validAdvertIdsSortCache.invalidate(VALID_ADVERT_IDS_CACHE_KEY);
    }
    
    /**
     * 获取当前状态有效的广告ID列表。
     * 付费广告缓存不会删除，只会更新。假如付费券排序缓存没有，查询定时任务更新的数据库里的值。这里不再花费时间重新构建
     *
     * @return
     * @throws TuiaException
     */
    public List<Long> queryValidAdvertIds() throws TuiaException {
        String value = stringRedisTemplate.opsForValue().get(CacheKeyTool.getCacheKey(RedisCommonKeys.KC101));
        // 如果redis缓存不为空，返回
        if (StringUtils.isNotBlank(value)) {
            return JSON.parseArray(value, Long.class);
        }
        List<Long> advertIds = Lists.newArrayList();
        advertIds.addAll(getValidPayAdvertList());
        advertIds.addAll(getValidFreeAdvertList());
        if (CollectionUtils.isNotEmpty(advertIds)) {
            stringRedisTemplate.opsForValue().set(CacheKeyTool.getCacheKey(RedisCommonKeys.KC101), JSON.toJSONString(advertIds));
        }
        return advertIds;
    }

    /**
     *
     * getValidAdvertOrderMap:(获取有效广告对应的排序). <br/>
     *
     * @author chencheng
     * @return
     * @throws TuiaException
     * @since JDK 1.8
     */
    private Map<Long, Integer> getValidAdvertOrderMap() throws TuiaException {

        List<Long> validAdvertIds = this.queryValidAdvertIds();
        Map<Long, Integer>  validAdvertOrderLevelMap = Maps.newHashMap();
        for (int i = 0; i < validAdvertIds.size(); i++) {
            validAdvertOrderLevelMap.put(validAdvertIds.get(i), i);
        }
        return validAdvertOrderLevelMap;
    }

    /**
     *
     * queryValidAdvertOrderLevel:(获取广告对应的排序map). <br/>
     *
     * @author chencheng
     * @return
     * @throws TuiaException
     * @since JDK 1.8
     */
    public Map<Long, Integer> queryValidAdvertOrderLevel() throws TuiaException{
        try {
            return validAdvertIdsSortCache.get(VALID_ADVERT_IDS_CACHE_KEY);
        } catch (Exception e) {
            logger.info("查询有效广告列表本地缓存错误", e);
            return getValidAdvertOrderMap();
        }

    }

    /**
     * 
     * getValidPayAdvertList:(付费有效广告排序缓存.缓存设置为不失效). <br/>
     *
     * @author chencheng
     * @return
     * @since JDK 1.8
     */
    private List<Long> getValidPayAdvertList() {
        String value = stringRedisTemplate.opsForValue().get(CacheKeyTool.getCacheKey(RedisCommonKeys.KC124));
        if (StringUtils.isNotBlank(value)) {
            return JSON.parseArray(value, Long.class);
        }
        try {
            List<Long> validPayAdvertIds = validAdvertService.selectValidPayAdvertFromDB();
            if (CollectionUtils.isNotEmpty(validPayAdvertIds)) {
                stringRedisTemplate.opsForValue().set(CacheKeyTool.getCacheKey(RedisCommonKeys.KC124), JSON.toJSONString(validPayAdvertIds));
            }
            // 这次用数据库里的旧数据，发个消息更新数据
            updateValidAdvertIdsCache(AdvertSortTypeEnum.PAY.getCode());
            return validPayAdvertIds;
        } catch (Exception e) {
            logger.error("get valid pay a dvert list error", e);
            return Lists.newArrayList();
        }
    }


    /**
     *
     * getValidPayAdvertList:(免费有效广告排序缓存.缓存设置为不失效). <br/>
     *
     * @author chencheng
     * @return
     * @throws TuiaException
     * @since JDK 1.8
     */
    private List<Long> getValidFreeAdvertList() throws TuiaException {
        String value = stringRedisTemplate.opsForValue().get(CacheKeyTool.getCacheKey(RedisCommonKeys.KC125));
        if (StringUtils.isNotBlank(value)) {
            return JSON.parseArray(value, Long.class);
        }
        //缓存重新构建后的数据
        return updateValidFreeAdvertIds();
    }


    /**
     *
     * updateValidPayAdvertIds:(免费有效广告排序重新构建). <br/>
     *
     * @author chencheng
     * @return
     * @throws TuiaException
     * @since JDK 1.8
     */
    private List<Long> updateValidFreeAdvertIds() throws TuiaException {
        List<Long> validFreeAdvertIds = validAdvertService.getValidFreeAdvertAgain();
        if (CollectionUtils.isNotEmpty(validFreeAdvertIds)) {
            stringRedisTemplate.opsForValue().set(CacheKeyTool.getCacheKey(RedisCommonKeys.KC125), JSON.toJSONString(validFreeAdvertIds));
        }
        return validFreeAdvertIds;
    }


    /**
     * 获取广告信息缓存:(这里用一句话描述这个方法的作用). <br/>
     *
     * @param advertId
     * @return
     * @throws TuiaException
     * @since JDK 1.6
     */

    public AdvertVO getAdvertByCache(Long advertId) {
        AdvertVO advertVO = null;
        try {
            if (null ==advertId || advertId.longValue() <= 0) {
                return advertVO;
            }
            advertVO = getAdvertVOInfo(advertId);
        } catch (Exception e) {
            logger.error(e.getMessage()+advertId.toString(), e);
        }
        return advertVO;
    }

    /**
     * 构建广告标签，广告主名称,投放时段，广告计划数据
     *
     * @param advertId 广告ID
     * @return 广告数据
     * @throws TuiaException the tuia exception
     */
    public AdvertVO getAdvertVOInfo(Long advertId) throws TuiaException {
        // 1.查询广告信息
        AdvertDO advertDO = advertDAO.getAdvertNoRegionIdsById(advertId);
        // 查询广告信息为空或者不是互动，构建缓存为空
       if (advertDO == null || advertDO.getAdvertType() != CommonConstant.HD_ADVERT_TYPE) {
           return null;
       }
        AdvertVO advertVO = new AdvertVO();
        // 2.查询广告标签数据
        AdvertTagDO advertTagDO = advertTagDAO.selectByAdvertId(advertId);
        if (null == advertTagDO) {
            logger.error("getAdvertVOInfo advertTagDO is null,advertId=[{}]", advertId);
            return null;
        }
        // 3.查询广告主名称
        AccountDO accountDO = accountDAO.selectAccountById(advertDO.getAccountId());
        if (null == accountDO) {
            logger.error("getAdvertVOInfo accountDO is null,advertId=[{}]，accountId={}", advertId, advertDO.getAccountId());
            return null;
        }

        //设置广告行业标签
        DataPermissionDO dataPermissionDO = dataPermissonDAO.selectTradeIdByAdvertId(new DataPermissionDO(advertId, DataPermissonSourceTypeEnum.SOURCE_TYPE_ADVERT.getCode()));
        Optional.ofNullable(dataPermissionDO).ifPresent(permission -> {
            advertTagDO.setAdvertTradeId(permission.getTradeId());
            advertTagDO.setNewAdvertTradeId(permission.getNewTradeId());
        });
        
        //设置广告资源标签
        ResoureTagsDO advertResourceTag = resourceTagsDAO.selectResoureTagsDOById(advertId, ResourceTagsTypeEnum.ADVERT_RESOURCE_TAG.getCode());
        if (null != advertResourceTag) {
            advertTagDO.setResourceTag(advertResourceTag.getTagNums());
        }
        
        //组装广告落地页标签数据
        ResoureTagsDO promoteUrlTagsDO = resourceTagsDAO.selectResoureTagsDOById(advertId, ResourceTagsTypeEnum.AD.getCode());
        List<String> promoteUrlTags = StringTool.getStringListByStr(Optional.ofNullable(promoteUrlTagsDO).map(ResoureTagsDO::getTagNums).orElse(null));
        advertTagDO.setPromoteUrlTags(promoteUrlTags);
        advertVO.setAdvertTagDO(advertTagDO);

        // 3.广告主名称
        advertVO.setAdvertiserName(accountDO.getCompanyName());
        advertVO.setDuibaAdvertiser(AccountSourceTypeEnum.DUIBA.getCode().equals(accountDO.getAccountSource()));
        advertVO.setCurrentMainStatus(accountDO.getCurrentMainStatus());
        advertVO.setEffectMainType(CurrentMainStatusEnum.ORIGINAL_MAIN_TYPE_CHANGING_STATUS.getStatus().equals(advertVO.getCurrentMainStatus()) ? accountDO.getPreCompanyOwner() : accountDO.getCompanyOwner());

        // 5.构建广告优惠券基本信息（仅用于校验的信息）
        advertVO.setCouponBase(buildCouponBaseData(advertDO.getId(), advertDO.getAdvertType()));

        //判断推广链接判断是否百奇积木域名，并设置标记
        if (advertVO.getCouponBase() != null) {
            setMakeTagSign(advertVO, advertVO.getCouponBase().getPromoteURL());
        }

        // 6.构建广告计划信息
        AdvertPlan advertPlan = new AdvertPlan();
        BeanCopierManager.getAdvertPlanBeanCopier().copy(advertDO, advertPlan, null);
        advertPlan.setAgentId(accountDO.getAgentId());

        // 7.查询
        String newAppTestUrl = advertPromoteTestDAO.selectNewAppTestUrl(advertId);
        advertPlan.setNewAppTestUrl(newAppTestUrl);
        
        advertVO.setAdvertPlan(advertPlan);
        //替换微信加粉
        replaceAdvertWeight(advertVO);

        //设置广告可以投放的媒体列表
        buildAdvertSpecialAppIds(advertDO.getAccountId(), advertVO);

        // 插入广告主回传链接
        advertVO.setPromoteBackUserUrl(advertDO.getPromoteBackUserUrl());

        advertVO.setPromoteDeepLink(advertDO.getPromoteDeepLink());
        advertVO.setAppDownloadUrl(advertDO.getAppDownloadUrl());

        // 查询广告扩展信息
        AdvertExpandDO advertExpandDO = advertExpandDAO.selectByAdvertId(advertId);
        if (null != advertExpandDO && null != advertExpandDO.getDepthSubtype() && null != advertExpandDO.getDepthTargetPrice()) {
            advertVO.setDepthSubtype(advertExpandDO.getDepthSubtype());
            advertVO.setDepthTargetPrice(advertExpandDO.getDepthTargetPrice());
        }

        return advertVO;
    }

    /**
     * 判断推广链接判断是否百奇积木域名，并设置标记
     * @param advertVO
     */
    public void setMakeTagSign(AdvertVO advertVO,String url) {
        try {
            MakeTagDO makeTag = makeTagCacheService.getMakeTagSign(url);
            if (makeTag != null) {
                advertVO.setSourceId(makeTag.getSourceId());
                advertVO.setSourceType(makeTag.getSourceType());
            }
        } catch (Exception e) {
            logger.error("推广链接设置百奇积木域名标识失败:{}", url);
        }
    }

    @SuppressWarnings("squid:S3776")
    private void buildAdvertSpecialAppIds(Long accountId, AdvertVO advertVO) {
        //查询该广告主是否是特殊广告主，如果是特殊广告主，则只能在特定的媒体上出券
        List<SpecialAccountAppDO> accountAppPackageList = specialAccountAppDAO.selectByAccountId(accountId);
        if (CollectionUtils.isEmpty(accountAppPackageList)) {
            return;
        }
        //查询广告主可以出券的特殊媒体(以流量包的形式存储)，为空则为不限 todo toMap的value判空
        Map<Long, List<Period>> packagePeriodMap = accountAppPackageList.stream().filter(Objects::nonNull)
                .collect(Collectors.toMap(
                SpecialAccountAppDO::getAppPackageId, accountAppPackage -> StringUtils.isBlank(accountAppPackage.getPeriodHours()) ?
                        NOT_LIMIT_PERIOD : JSON.parseArray(accountAppPackage.getPeriodHours(), Period.class), (oldVal, newVal) -> newVal));

        List<AppPackageDO> appPackageList = appPackageDAO.selectByIds(Lists.newArrayList(packagePeriodMap.keySet()));

        //配置的流量包对应的媒体列表 todo toMap的value判空
        Map<Long, List<Long>> packageAppMap = appPackageList.stream().filter(Objects::nonNull)
                .collect(Collectors.toMap(AppPackageDO::getId,
                appPackage -> StringTool.getLongListByStr(appPackage.getAppIds()), (oldVal, newVal) -> newVal));

        Map<Long, List<Period>> specialApps = Maps.newHashMap();

        /**
         * key->appPackageId，value->SpecialAppBo
         * sql中过滤了 AppPackageId == null的数据，此处 filter 可删除
         * todo toMap的value判空
         */
        Map<Long, SpecialAppBo> packageAppSpecialAppBoMap = accountAppPackageList.stream()
                .filter(specialAccountAppDO -> null != specialAccountAppDO && null != specialAccountAppDO.getAppPackageId()).
                collect(Collectors.toMap(SpecialAccountAppDO::getAppPackageId,specialAccountAppDO->{
                    SpecialAppBo specialAppBo = new SpecialAppBo();
                    specialAppBo.setLowestPrice(specialAccountAppDO.getLowestPrice());
                    return specialAppBo;
                }, (oldVal, newVal) -> newVal));
        /**
         * key->appId, value->SpecialAppBo
         */
        Map<Long,SpecialAppBo> specialAppsOtherData = new HashMap();

        packageAppMap.forEach((appPackageId, appList) -> {
            List<Period> packagePeriods = packagePeriodMap.get(appPackageId);
            SpecialAppBo specialAppBo = packageAppSpecialAppBoMap.get(appPackageId);

            if (packagePeriods == null||null == specialAppBo) {
                return;
            }
            //配置了可以出的流量包，但是是空包则不可以出券
            if (CollectionUtils.isEmpty(appList)) {
                specialApps.put(-1L, EMPTY_PERIOD);
                specialAppsOtherData.put(-1L, specialAppBo);
                return;
            }
            appList.forEach(appId -> {
                List<Period> appPeriods = specialApps.get(appId);
                if (appPeriods == null) {
                    appPeriods = Lists.newArrayList();
                    appPeriods.addAll(packagePeriods);
                    specialApps.put(appId, appPeriods);
                } else {
                    appPeriods.addAll(packagePeriods);
                }

                SpecialAppBo specialAppBoTmp = specialAppsOtherData.get(appId);
                if(null == specialAppBoTmp){
                    specialAppBoTmp = new SpecialAppBo();
                    specialAppBoTmp.setLowestPrice(specialAppBo.getLowestPrice());
                    specialAppsOtherData.put(appId,specialAppBoTmp);
                    //当发生冲突时，时间取并集，最低出价取低值。
                }else{
                    Long oldPrice = specialAppBoTmp.getLowestPrice();
                    Long newPrice = specialAppBo.getLowestPrice();
                    specialAppBoTmp.setLowestPrice(oldPrice<newPrice?oldPrice:newPrice);
                }
            });
        });
        advertVO.setSpecialApps(specialApps);
        advertVO.setSpecialAppsOtherData(specialAppsOtherData);
    }

    /**
     * 替换微信加粉类型广告的权重为全局命中的权重
     * @param advertVO
     */
    private void replaceAdvertWeight(AdvertVO advertVO){
        if(advertVO == null){
            return;
        }

        //全局流量配置只考虑微信加粉
        List<String> promoteUrlTags=advertVO.getAdvertTagDO().getPromoteUrlTags();
        if(CollectionUtils.isNotEmpty(promoteUrlTags) && promoteUrlTags.contains(WEI_XIN_JIA_FEN)){
            List<GlobalConfigFlowVO> globalConfigFlowVOS=globalConfigFlowService.queryAll();

            //获取微信加粉的投放时段配置
            List<GlobalConfigFlowVO> periodConfig=globalConfigFlowVOS.stream().filter(vo-> IS_ADVERT_WEIGHT.equals(vo.getConfigItem()) && vo.getConfigConditionValue().containsValue(WEI_XIN_JIA_FEN)).collect(Collectors.toList());

            if(CollectionUtils.isNotEmpty(periodConfig)){
                JSONArray periodArr =periodConfig.get(0).getConfigItemValue();
                String advertWeight=periodArr.getJSONObject(0).get("weight").toString();
                advertVO.getAdvertPlan().setAdvertWeight(new BigDecimal(advertWeight));
            }
        }

    }


    /**
     * 根据广告id查询优惠券信息,首先从缓存中查询，如果缓存中没有,则从DB查询
     * @param advertId
     * @return
     * @throws TuiaException
     */
    public AdvertCoupon getAdvertCouponByLocal(Long advertId) throws TuiaException{
        AdvertCoupon advertCoupon;
        AdvertNewCouponDto advertNewCouponDto = null;
        try {
            advertNewCouponDto = advertCouponBackCache.get(advertId).orElse(null);
        } catch (Exception e) {
            logger.info("ServiceManager.getAdvertCouponByLocal e",e);
        }

        if(advertNewCouponDto == null){
            advertCoupon = getAdvertCouponByRedis(advertId);
            advertCouponBackCache.refresh(advertId);
        }else{
            advertCoupon = setAdvertCoupon(advertNewCouponDto);
        }

        return advertCoupon;
    }

    /**
     * 广告优惠券缓存
     * @author wzj
     * @param advertId
     */
    public void updateAdvertCouponCache(Long advertId) {

        try {
            if (null != advertId) {
                advertCouponBackCache.refresh(advertId);

            }
        } catch (Exception e) {
            logger.error("ServiceManager.updateAdvertCouponCache errror", e);
        }
    }


    /**
     * 根据广告id查询优惠券信息,首先从缓存中查询，如果缓存中没有,则从DB查询，并更新到缓存中 <一句话功能描述>
     *
     * @param advertId 广告ID
     * @return 优惠券信息
     * @throws TuiaException the tuia exception
     */
    public AdvertCoupon getAdvertCouponByRedis(Long advertId) throws TuiaException {

        // 查询数据库，获取优惠券数据
        AdvertNewCouponDto advertNewCouponDto = getAdvertCoupon(advertId);

        AdvertCoupon advertCoupon = setAdvertCoupon(advertNewCouponDto);
        return advertCoupon;
    }

    @NotNull
    private AdvertCoupon setAdvertCoupon(AdvertNewCouponDto advertNewCouponDto) throws TuiaException{
        if(advertNewCouponDto == null){
            throw new TuiaException(ErrorCode.E0100005);
        }
        AdvertCoupon advertCoupon = new AdvertCoupon();
        advertCoupon.setBannerPng(advertNewCouponDto.getBannerPng());
        advertCoupon.setCouponType(advertNewCouponDto.getCouponType());
        advertCoupon.setCouponName(advertNewCouponDto.getCouponName());
        advertCoupon.setCouponRemark(advertNewCouponDto.getCouponRemark());
        advertCoupon.setDescription(advertNewCouponDto.getDescription());
        advertCoupon.setLimitReceive(advertNewCouponDto.getLimitReceive());
        advertCoupon.setThumbnailPng(advertNewCouponDto.getThumbnailPng());
        advertCoupon.setPromoteURL(advertNewCouponDto.getPromoteURL());
        advertCoupon.setIsWeixin(advertNewCouponDto.isWeixin());
        advertCoupon.setIsDisplayMenu(advertNewCouponDto.isDisplayMenu());
        String buttonText = advertNewCouponDto.getButtonText() != null ? advertNewCouponDto.getButtonText() : AdvertCoupon.DEFAULT_BTN_TEXT;
        advertCoupon.setButtonText(buttonText);
        return advertCoupon;
    }

    @NotNull
    private AdvertNewCouponDto getAdvertNewCoupon(Long advertId) throws TuiaException {
        AdvertNewCouponDto advertNewCouponDto = null;
        try {
            advertNewCouponDto = getAdvertCouponBackFromCache(advertId);
        }catch (Exception e){
            logger.error("getAdvertCoupon exception advertId:{}", advertId);
        }

        if(advertNewCouponDto == null){
            throw new TuiaException(ErrorCode.E0100005);
        }

        return advertNewCouponDto;
    }

    /**
     * 
     * getAdvertCouponBackFromCache:(从优惠券备份缓存获取). <br/>
     *
     * @author chencheng
     * @param advertId
     * @return
     * @since JDK 1.8
     */
    public AdvertNewCouponDto getAdvertCouponBackFromCache(Long advertId) {
        
        try {
            return advertCouponBackCache.get(advertId).orElse(null);
        } catch (ExecutionException e) {
            logger.error("get getAdvertCouponBackFromCache error", e);
            return null;
        }
    }
            
    
    /**
     * 直接查询优惠券备份表
     */
    public AdvertNewCouponDto getAdvertCoupon(Long advertId) throws TuiaException {
        try {
            AdvertCouponBackDO advertCouponBackDO = advertCouponBackDAO.getAdvertCouponBack(advertId);
            if (null == advertCouponBackDO) {
                return null;
            }
            AdvertNewCouponDto advertCouponGoodsDto = BeanUtils.copy(advertCouponBackDO, AdvertNewCouponDto.class);
            advertCouponGoodsDto.setWeixin(advertCouponBackDO.getIsWeixin() == 1);
            advertCouponGoodsDto.setDisplayMenu(advertCouponBackDO.getDisplayMenu() == 1);
            return advertCouponGoodsDto;
        } catch (Exception e) {
            logger.error("getAdvertCoupon error, advertId=[{}]", advertId, e);
            throw new TuiaException(ErrorCode.E0100005);
        }
    }


    /**
     * 优惠券备份缓存  有消息同步，不用设置过期时间来保证避免读取旧值
     */
    private final LoadingCache<Long, Optional<AdvertNewCouponDto>> advertCouponBackCache = CacheBuilder.newBuilder()
            .initialCapacity(1000).maximumSize(2000).refreshAfterWrite(20,TimeUnit.MINUTES)
            .build(new CacheLoader<Long, Optional<AdvertNewCouponDto>>() {
                @Override
                public Optional<AdvertNewCouponDto> load(Long key) throws Exception {
                    return Optional.ofNullable(getAdvertCoupon(key));
                }
            
                @Override
                public ListenableFuture<Optional<AdvertNewCouponDto>> reload(Long key, Optional<AdvertNewCouponDto> oldValue) throws Exception {
                    ListenableFutureTask<Optional<AdvertNewCouponDto>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });

    /**
     * 构建优惠券基本信息:(这里用一句话描述这个方法的作用). <br/>
     *
     * @return
     * @throws TuiaException
     * @author cdm
     * @param
     * @since JDK 1.6
     */
    public CouponBase buildCouponBaseData(Long advertId, Integer advertType) throws TuiaException {

        if (advertType != CommonConstant.HD_ADVERT_TYPE) {
            return null;
        }
        // 1.查询商品信息
        AdvertNewCouponDto advertNewCouponDto = getAdvertNewCoupon(advertId);

        // 2.构建优惠券基础信息
        CouponBase couponBase = new CouponBase();
        if (advertNewCouponDto != null) {
            couponBase.setPromoteURL(advertNewCouponDto.getPromoteURL());
            couponBase.setCouponType(advertNewCouponDto.getCouponType());
        }

        return couponBase;
    }

    /**
     * 根据广告id更新广告信息 <一句话功能描述>
     *
     * @param advertId
     * @throws TuiaException
     */
    public AdvertVO updateAdvert(Long advertId) throws TuiaException {
        AdvertVO advertVO = getAdvertVOInfo(advertId);
        return advertVO;
    }


    /**
     * 获取曾经请求过的广告订单 <一句话功能描述>
     *
     * @param consumerId
     * @param orderId
     * @return
     * @throws TuiaException
     */
    public AdvertOrderDO getAdvertOrderDO(final Long consumerId, final String orderId, String adxMediaType) throws TuiaException {

        String key = CacheKeyTool.getCacheKey(RedisKeys.K01, consumerId, orderId);
        return advancedCacheClient.getWithCacheLoader(key,60,TimeUnit.SECONDS, () -> {
            try {
                String hbaseTableName = ORDER_TABLE;
                if(AdxSceneEnum.ADX_MEITUAN.getCode().equals(adxMediaType)){
                    hbaseTableName = ORDER_RELOG_TABLE;
                }

                return advertOrderDAO.getAdvertOrder(consumerId, orderId, hbaseTableName);
            } catch (TuiaException e) {
                logger.error("getAdvertOrderDO error", e);
                return null;
            }
        });
    }

    /**
     * 将广告订单放到缓存中
     *
     * @param consumerId
     * @param orderId
     * @param advertOrderDO
     */
    public void setAdvertOrderDO(Long consumerId, String orderId, AdvertOrderDO advertOrderDO) {
        executorService.submit(new Runnable() {
            @Override
            public void run() {
                // 单个订单设置60s缓存
                String key = CacheKeyTool.getCacheKey(RedisKeys.K01, consumerId, orderId);
                String cacheKey = CacheKeyTool.getCacheKey(RedisKeys.K01, orderId);
                advancedCacheClient.set(key, advertOrderDO, 60, TimeUnit.SECONDS);
                advancedCacheClient.set(cacheKey, advertOrderDO, 60, TimeUnit.SECONDS);
            }
        });
    }

    /**
     * getKey:. <br/>
     *
     * @param tuiaKey
     * @return
     * @throws TuiaException
     * @author sunjiangrong
     * @since JDK 1.6
     */
    public Integer getIntValue(final String tuiaKey) throws TuiaException {
        try {
            return localCache.get(tuiaKey).map(val -> Integer.valueOf(val)).orElse(null);
        } catch (Exception e) {
            return getIntValueInner(tuiaKey);
        }
    }

    public String getStrValue(String tuiaKey) throws TuiaException {
        try {
            return localCache.get(tuiaKey).orElse(null);
        } catch (Exception e) {
            return getStrValueInner(tuiaKey);
        }
    }

    /**
     * 全局配置立即生效
     * @param key
     * @param
     */
    public void putStrValue(String key,String value){
        localCache.put(key,Optional.ofNullable(value));
    }
    private Integer getIntValueInner(String tuiaKey) throws TuiaException {
        String value = stringRedisTemplate.opsForValue().get(CacheKeyTool.getCacheKey(RedisKeys.K19, tuiaKey));
        if (StringUtils.isBlank(value)) {
            value = systemConfigDAO.getSystemConfig(tuiaKey);
            if (!StringUtils.isBlank(value)) {
                stringRedisTemplate.opsForValue().set(CacheKeyTool.getCacheKey(RedisKeys.K19, tuiaKey), value,
                        2 * 24 * 3600, TimeUnit.SECONDS);
                return Integer.valueOf(value.toString());
            } else {
                return null;
            }
        } else {
            return Integer.valueOf(value);
        }
    }

    public String getStrValueInner(String key) throws TuiaException {
        String value = stringRedisTemplate.opsForValue().get(CacheKeyTool.getCacheKey(RedisKeys.K19, key));
        if (StringUtils.isBlank(value)) {
            value = systemConfigDAO.getSystemConfig(key);
            if (!StringUtils.isBlank(value)) {
                stringRedisTemplate.opsForValue().set(CacheKeyTool.getCacheKey(RedisKeys.K19, key), value,
                        2 * 24 * 3600, TimeUnit.SECONDS);
                return value;
            } else {
                return null;
            }
        } else {
            return value;
        }
    }

    /**
     * clearValue:删除缓存里的某个系统配置项. <br/>
     *
     * @param tuiaKey
     * @author sunjiangrong
     * @since JDK 1.6
     */
    public void clearValue(String tuiaKey) {
        stringRedisTemplate.delete(CacheKeyTool.getCacheKey(RedisKeys.K19, tuiaKey));
        localCache.invalidate(tuiaKey);
    }

    /**
     * checkPeriods:检查此时刻是否可以发券. <br/>
     * 该方法包括两情况:1、默认情况下全天发放;2、在投放时段,且未达投放上限).<br/>
     *
     * @param periods the advert vo
     * @return true, if check periods
     * @throws TuiaException
     * @author sunjiangrong
     * @since JDK 1.6
     */
    public boolean checkPeriods(List<AdvertPlanPeriodDO> periods, Integer isChangePeriodList) {
        //全局流量命中但是交集为空
        if(isChangePeriodList !=null && isChangePeriodList ==-2){
            return false;
        }
        // 如果本身没有投放时段,则默认全天可以发放
        if (CollectionUtils.isEmpty(periods)) {
            return true;
        }

        AdvertPlanPeriodDO period = gainPeriods(periods);
        if (period == null) {// 不在该投放时段
            return false;
        }
        return advertPeriodService.getAdvertPeriodPut(period.getId(),period.getPeriodType());
    }
    /**
     * 
     * checkBudgetSmooth:(当前时段，消耗速度校验修改). <br/>
     *
     * @author chencheng
     * @param periods
     * @param adv
     * @since JDK 1.8
     */
    public void checkBudgetSmooth(List<AdvertPlanPeriodDO> periods, AdvOrientationItem adv) {
        AdvertPlanPeriodDO period = gainPeriods(periods);
        if (period == null) {// 不在该投放时段
            return;
        }
        // 默认加速消耗不修改
        if (adv.getBudgetSmooth() == null || adv.getBudgetSmooth().equals(AdvertOrientationPackageDO.BUDGET_SMOOTH_DEFULT)) {
            return;
        }
        // 开启匀速消耗，判断当前时段能否匀速消耗
        if (AdvertPkgPeriodTypeEnum.PERIOD_TYPE_COUNT_BUDGET.getCode().equals(period.getPeriodType())
                || AdvertPkgPeriodTypeEnum.PERIOD_TYPE_HOUR_BUDGET.getCode().equals(period.getPeriodType())){
            return;
        }
        // 按照小时/时段的发券量不允许匀速消耗
        adv.setBudgetSmooth(AdvertOrientationPackageDO.BUDGET_SMOOTH_DEFULT);
    }

    /**
     * checkIssueTime: 判断当前时间是否在投放时段内 . <br/>
     * 当返回null时存在两种情况，不在投放时段内和没有投放时段，需要与checkHavePeriod配合使用.<br/>
     *
     * @return
     * @author sunjiangrong
     * @since JDK 1.6
     */
    private AdvertPlanPeriodDO gainPeriods(List<AdvertPlanPeriodDO> periods) {

        DateTime dateTime = new DateTime(new Date());
        // 2.转为int类型比较大小
        int dt = Integer.parseInt(dateTime.toString("HHmm"));
        for (AdvertPlanPeriodDO period : periods) {
            if (StringUtils.isEmpty(period.getEndHour()) || StringUtils.isEmpty(period.getStartHour())) {
                continue;
            }

            String startHour = patternReplace.matcher(period.getStartHour()).replaceAll("");
            String endHour = patternReplace.matcher(period.getEndHour()).replaceAll("");
            int sh = Integer.parseInt(startHour);
            int eh = Integer.parseInt(endHour);
            if (dt >= sh && dt < eh) {
                return period;
            }
        }

        return null;
    }

    //有消息同步，不用设置过期时间来保证避免读取旧值
    private final LoadingCache<String, Optional<ActivityAdvert4MonitorDto>> BUSINESS_ACTIVITY_CACHE = CacheBuilder.newBuilder().initialCapacity(3500).maximumSize(4000)
            .refreshAfterWrite(10, TimeUnit.MINUTES).build(new CacheLoader<String, Optional<ActivityAdvert4MonitorDto>>() {
        @Override
        public Optional<ActivityAdvert4MonitorDto> load(String key) {
            return Optional.ofNullable(getBusinessActivity(key));
        }

        @Override
        public ListenableFuture<Optional<ActivityAdvert4MonitorDto>> reload(final String key, Optional<ActivityAdvert4MonitorDto> oldValue)  {
            ListenableFutureTask<Optional<ActivityAdvert4MonitorDto>> task = ListenableFutureTask.create(() -> load(key));
            executorService.submit(task);
            return task;
        }
    });

    /**
     * getBusinessActivityCache:(这里用一句话描述这个方法的作用). <br/>
     *
     * @param duibaActivityId  兑吧活动id
     * @param activityUserType 兑吧活动来源(0:普通兑吧活动,1:兑吧商业活动,2:推啊活动)
     *                         对于媒体那边而言:0代表兑吧商业活动,1:推啊活动
     * @return
     * @throws TuiaException
     * @author zp
     * @since JDK 1.6
     */
    public Optional<ActivityAdvert4MonitorDto> getBusinessActivityCache(Long duibaActivityId, Integer activityUserType){
        if(null == duibaActivityId || null == activityUserType){
            return Optional.empty();
        }
        String hashKey = duibaActivityId + "-" + (activityUserType - 1);
        return BUSINESS_ACTIVITY_CACHE.getUnchecked(hashKey);
    }

    public void updateBusinessActivityCache(String key){
        BUSINESS_ACTIVITY_CACHE.invalidate(key);
    }

    private ActivityAdvert4MonitorDto getBusinessActivity(String key){
        try{
            List<String> list = Splitter.on("-").splitToList(key);
            Long activityId = Long.parseLong(list.get(0));
            Integer source = Integer.parseInt(list.get(1));
            return remoteActivityBackendService.queryActivityAdvert4Monitor(activityId, source);
        }catch(Exception e){
            logger.error("getBusinessActivity error key:{}",key, e);
        }
        return null;
    }
    
    /**
     * checkClickVaild:(检查用户点击的有效性). <br/>
     *
     * @param consumerId
     * @param advertId
     * @return true-该次点击有效 false-该次点击无效
     * @author ZFZ
     * @since JDK 1.6
     */
    public boolean checkClickVaild(long appId, long advertId, long consumerId, String date) {
        try {
            // 判断5分钟间隔redis缓存是否存在
            boolean isInShieldTime = stringRedisTemplate.hasKey(CacheKeyTool.getCacheKey(RedisKeys.K23, date, consumerId, advertId));
            if (isInShieldTime) {
                return false;
            } else {
                int shieldTime = 60 * CLICK_LIMIT_TIME;
                stringRedisTemplate.opsForValue().set(CacheKeyTool.getCacheKey(RedisKeys.K23, date, consumerId, advertId),
                        "shield", shieldTime, TimeUnit.SECONDS);
                // 如果不在5分钟间隔之内，则判断当天点击数量
                return checkClickCount(appId, advertId, consumerId, date);
            }

        } catch (Exception e) {
            logger.error(" checkClickVaild is error the consumerId=[{}] the advertId=[{}] error is [{}]", consumerId,
                    advertId, e);
            return false;
        }
    }

    /**
     * checkClickCount:(判断用户当前点击是否有效-是否满足当天点击限制). <br/>
     *
     * @param consumerId
     * @param advertId
     * @param date
     * @return true-该点击有效，false-该点击无效
     * @throws TuiaException
     * @author ZFZ
     * @since JDK 1.6
     */
    private boolean checkClickCount(long appId, long advertId, long consumerId, String date) throws TuiaException {
        // 判断每天点击计数redis缓存是否存在
        String clickCount = stringRedisTemplate.opsForValue().get(CacheKeyTool.getCacheKey(RedisKeys.K24, date, consumerId, advertId));
        if (StringUtils.isEmpty(clickCount)) {

            stringRedisTemplate.opsForValue().increment(CacheKeyTool.getCacheKey(RedisKeys.K24, date, consumerId, advertId), 1);

            stringRedisTemplate.expire(CacheKeyTool.getCacheKey(RedisKeys.K24, date, consumerId, advertId), DateUtils.getToTomorrowSeconds() + new Random().nextInt(2 * 60 * 60), TimeUnit.SECONDS);

        } else {
            if (CLICK_LIMIT <= Integer.parseInt(clickCount)) {
                return false;
            }
            stringRedisTemplate.opsForValue().increment(CacheKeyTool.getCacheKey(RedisKeys.K24, date, consumerId, advertId), 1);
        }
        return true;
    }

    /**
     * getShuntConfig:(获取A/B Test 流量分配策略). <br/>
     *
     * @param shuntId
     * @param date
     * @return
     * @throws TuiaException
     * @author ZFZ
     * @since JDK 1.6
     */
    public ShuntConfigDO getShuntConfig(Long shuntId, String date) throws TuiaException {

        ShuntConfigDO shuntConfigDO;
        // 从缓存中获取策略流量分配
        String shuntConfig = stringRedisTemplate.opsForValue().get(CacheKeyTool.getCacheKey(RedisKeys.K25, date, shuntId));

        // 缓存中没有，则从数据库中获取策略流量分配
        if (StringUtils.isBlank(shuntConfig)) {
            shuntConfigDO = shuntConfigDAO.selectShuntConfigByShuntId(shuntId);
        } else {
            shuntConfigDO = JSONObject.parseObject(shuntConfig, ShuntConfigDO.class);
        }

        // 数据库获取策略配置不为null，则更新到缓存中
        if (StringUtils.isBlank(shuntConfig) && shuntConfigDO != null) {
            String shuntConfigJson = JSONObject.toJSONString(shuntConfigDO);
            stringRedisTemplate.opsForValue().set(CacheKeyTool.getCacheKey(RedisKeys.K25, date, shuntId), shuntConfigJson, 24 * 3600, TimeUnit.SECONDS);
        }

        return shuntConfigDO;
    }

    /**
     * refreshConfig:(更新系统配置参数). <br/>
     *
     * @param tuiaKey
     * @param tuiaValue
     * @throws TuiaException
     * @author ZFZ
     * @since JDK 1.6
     */
    public void refreshConfig(String tuiaKey, String tuiaValue) throws TuiaException {

        String value = systemConfigDAO.getSystemConfig(tuiaKey);
        if (value == null) {
            SystemConfigDO systemConfigDO = new SystemConfigDO();
            systemConfigDO.setTuiaKey(tuiaKey);
            systemConfigDO.setTuiaValue(tuiaValue);

            systemConfigDAO.insert(systemConfigDO);
        } else if (!value.equals(tuiaValue)) {
            systemConfigDAO.updateSystemConfig(tuiaKey, tuiaValue);
            clearValue(tuiaKey);
        }
    }

    /**
     * 查看该推广链接是否需要屏蔽该广告位的url. <br/>
     *
     * @param promoteURL
     * @param shieldUrls
     * @return false 不需要屏蔽，true 需要屏蔽
     * @author sunjiangrong
     * @since JDK 1.6
     */
    public boolean isNeedShield(String promoteURL, Set<String> shieldUrls) {
        if (StringUtils.isEmpty(promoteURL)) {
            return false;
        }

        if (shieldUrls == null || shieldUrls.size() == 0) {
            return false;
        }

        for (String url : shieldUrls) {
            if (promoteURL.contains(url)) {
                return true;
            }
        }

        return false;
    }

    /**
     * describe : 是否需要修改素材的状态. <br/>
     * 1.新素材，判断是否满足转换为老素材条件，更新数据库对应标签 2.老素材，不用管
     *
     * @param materialId 素材ID
     * @param appId      appID
     * @author : cdm
     * @date : 2016年12月26日下午3:09:20
     */
    public void updateMaterialStatus(Long advertId, Long materialId, Long appId) {
        executorService.submit(new Runnable(){
            @Override
            public void run() {
                Boolean isNew = advertMaterialRealtionService.isNewOfMaterial(advertId, materialId, appId);
                if (!isNew) {
                    return;
                }
                // 3.更新素材基于媒体的曝光数
                Long materiaPv = stringRedisTemplate.opsForValue().increment(CacheKeyTool.getCacheKey(RedisKeys.K28, appId, materialId), 1);
                // 4.判断是否大于状态转换临界值
                if (materiaPv >= TuiaServiceConfig.MATERIA_APP_STATUS_PV_CRITICAL_VALUE) {
                    // 4.1 修改数据库中的app素材状态
                    advertMaterialRealtionService.updateOldStatus(advertId, materialId, appId);
                    // 4.2 修改状态后删除素材基于媒体的曝光数
                    stringRedisTemplate.expire(CacheKeyTool.getCacheKey(RedisKeys.K28, appId, materialId), 120, TimeUnit.SECONDS);
                }
            }
        });
    }

    /**
     * 增加今日谢谢参与的次数
     */
    public void incrTodayMissedCoupon(){
        int today = DateUtils.getYYDayNumber(new Date());
        String key = CacheKeyTool.getCacheKey(RedisKeys.K31,today);
        Long result = stringRedisTemplate.opsForValue().increment(key,1L);
        if(result.equals(1L)){
            stringRedisTemplate.expire(key,DateUtils.getToTomorrowSeconds() + new Random().nextInt(2 * 60 * 60), TimeUnit.SECONDS);
        }
    }

    /**
     * 获取今日谢谢参与的次数
     * @return
     */
    public Long getTodayMissedCoupon(){
        int today = DateUtils.getYYDayNumber(new Date());
        String val = stringRedisTemplate.opsForValue().get(CacheKeyTool.getCacheKey(RedisKeys.K31,today));
        return StringUtils.isNotEmpty(val) ? Long.parseLong(val) : 0L;
    }

    /**
     * xuxx:20190530
     * //TODO 此处缓存使用不规范，后期整理缓存规范后再处理
     */
    private volatile Map<String,Boolean> URGENT_ADVERT_STATUS_CACHE;

    /**
     * 获取缓存
     * @param key
     * @return
     */
    public Boolean getUrgentAdvertStatusByKey(String key) {
        try {
            if(null == URGENT_ADVERT_STATUS_CACHE){
                initUrgentAdvertStatus();
            }
            return URGENT_ADVERT_STATUS_CACHE == null ? true : URGENT_ADVERT_STATUS_CACHE.get(key);
        } catch (Exception e) {
            logger.info("getUrgentAdvertStatusByKey获取配置消耗是否满足广告消耗占比失败，默认投放", e);
            return true;
        }
    }

    /**
     * 初始化缓存
     */
    @SuppressWarnings("squid:S3776")
    private synchronized void initUrgentAdvertStatus(){
        if(null == URGENT_ADVERT_STATUS_CACHE){
            try {
                Map resdisMap = stringRedisTemplate.opsForHash().entries(RedisCommonKeys.KC138.toString());
                if(null != resdisMap){
                    Map<String,Boolean> initMap = new HashMap<>();
                    resdisMap.forEach((key,value)->{
                        if(value instanceof String) {
                            Boolean b = "false".equals(value) ? false : ("true".equals(value) ? true : null);
                            Optional.ofNullable(b).ifPresent(bool->initMap.put((String) key,bool));
                        }else if(value instanceof  Boolean){
                            initMap.put((String) key, (Boolean) value);
                        }
                    });
                    URGENT_ADVERT_STATUS_CACHE = initMap;
                }else{
                    URGENT_ADVERT_STATUS_CACHE = new HashMap<>();
                }
            } catch (Exception e) {
                logger.error("initUrgentAdvertStatus获取redis失败", e);
                return;
            }
        }
    }

    /**
     * 清空缓存
     */
    public void cleanUpUrgentAdvertStatus(){
        URGENT_ADVERT_STATUS_CACHE = null;
    }

    /**
     * 刷新keys
     * @param inputMap
     */
    public void updateUrgentAdvertStatus(Map<String,Boolean> inputMap){
        if(inputMap instanceof HashMap){
            URGENT_ADVERT_STATUS_CACHE = inputMap;
        }else{
            Map<String,Boolean> hashMap = new HashMap<>(inputMap);
            URGENT_ADVERT_STATUS_CACHE = hashMap;
        }
    }

    /**
     * 获取 托管底价白名单开关状态
     * @注意：获取不到则认为 托管底价白名单 为关闭状态，（托管底价白名单 为测试用）
     * @return
     */
    public Boolean getOcpcBasePriceWhiteListStatus(){
        //todo获取 托管底价白名单 的开关状态
        try {
            String stringOptional = getStrValue(SystemConfigKeyConstant.OCPC_BASEPRICE_CONTROL_ENABLESTATE);
            //查询不到 或者 查到 不是 开的 状态 则返回 关闭
            return OCPC_BASEPRICE_CONTROL_OPEN.equals(stringOptional);
        } catch (Exception e) {
            logger.warn("查询托管底价白名单开关状态异常",e);
            return false;
        }
    }

    public Long getOcpcClickMinPrice(){
        try {
            String stringOptional = getStrValue(OCPC_CLICK_MIN_PRICE);
            return (null != stringOptional) ? Long.parseLong(stringOptional) : DEFAULT_BASE_PRICE;
        } catch (Exception e) {
            logger.warn("查询托管底价异常",e);
            return DEFAULT_BASE_PRICE;
        }
    }

    /**
     * advert_sdk缓存，用于获取 包名 和 应用名称
     */
    private final LoadingCache<Long, Optional<AdvertSdkDO>> ADVERT_SDK_CACHE = CacheBuilder.newBuilder().initialCapacity(1500).maximumSize(2000)
            .refreshAfterWrite(10, TimeUnit.MINUTES).expireAfterWrite(2,TimeUnit.HOURS).build(new CacheLoader<Long, Optional<AdvertSdkDO>>() {

                @Override
                public Optional<AdvertSdkDO> load(Long key) {
                    return Optional.ofNullable(findAdvertSdkDtoByAdvertId(key));
                }

                @Override
                public ListenableFuture<Optional<AdvertSdkDO>> reload(final Long key, Optional<AdvertSdkDO> oldValue)  {
                    ListenableFutureTask<Optional<AdvertSdkDO>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });


    /**
     * 按advertId查询数据库获取缓存
     */
    private AdvertSdkDO findAdvertSdkDtoByAdvertId(Long advertId){
        return advertSdkDao.getSdkMsg(advertId);
    }


    /**
     * 按advertId获取缓存
     */
    public AdvertSdkDO getAdvertSdkDtoByAdvertId(Long advertId){
        if(null == advertId){
            return null;
        }

        try {
            return ADVERT_SDK_CACHE.get(advertId).orElse(null);
        } catch (ExecutionException e) {
            logger.error("get getAdvertSdkDtoByAdvertId error", e);
            return null;
        }
    }

    /**
     * advert_expand缓存，用于获取 包名 和 应用名称 有消息同步，不用设置过期时间来保证避免读取旧值
     */
    private final LoadingCache<Long, Optional<Integer>> ADVERT_EXPAND_CACHE = CacheBuilder.newBuilder().initialCapacity(1500).maximumSize(2000)
            .refreshAfterWrite(10, TimeUnit.MINUTES).build(new CacheLoader<Long, Optional<Integer>>() {

                @Override
                public Optional<Integer> load(Long key) {
                    return Optional.ofNullable(findAdvertExpandByAdvertId(key));
                }

                @Override
                public ListenableFuture<Optional<Integer>> reload(final Long key, Optional<Integer> oldValue)  {
                    ListenableFutureTask<Optional<Integer>> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });


    public Boolean getPromoteJsFromCache(String key){

        return promoteUrlJsCache.getUnchecked(key);
    }

    /**
     * 落地页地址是否接入js缓存
     */
    private final LoadingCache<String, Boolean> promoteUrlJsCache = CacheBuilder.newBuilder()
            .initialCapacity(1000).refreshAfterWrite(5, TimeUnit.MINUTES).expireAfterWrite(24,TimeUnit.HOURS)
            .build(new CacheLoader<String, Boolean>() {
                @Override
                public Boolean load(String key) throws Exception {
                    return getPromoteJsValue(key);
                }

                @Override
                public ListenableFuture<Boolean> reload(String key,Boolean oldValue) throws Exception {
                    ListenableFutureTask<Boolean> task = ListenableFutureTask.create(() -> load(key));
                    executorService.submit(task);
                    return task;
                }
            });


    public Boolean getPromoteJsValue(String key) throws TuiaException {
        String value = stringRedisTemplate.opsForValue().get(CacheKeyTool.getCacheKey(RedisKeys.K32, key));
        if (StringUtils.isBlank(value)) {
            value = calculateValue(key);
            //接入js 今日数据存储一天
            if (HAS_JS.equals(value)) {
                stringRedisTemplate.opsForValue().set(CacheKeyTool.getCacheKey(RedisKeys.K32, key), value,
                        DateUtils.getToTomorrowSeconds()+300, TimeUnit.SECONDS);
            }
        }
        return Boolean.valueOf(value);
    }

    private String calculateValue(String key){

        try {
            DwsPromoteJsDO dwsPromoteJsDO = promoteUrlJsDAO.getPromteJs(key);

            if (dwsPromoteJsDO == null) {
                return NO_JS;
            }

            //当前日期，落地页访问pv/落地页转换pv>=20% 代表接入js
            Long visitPv = dwsPromoteJsDO.getVisitPv();
            Long clickPv = dwsPromoteJsDO.getClickPv();

            if (clickPv == 0) {
                return NO_JS;
            }

            double value = BigDecimal.valueOf(visitPv).divide(BigDecimal.valueOf(clickPv), 2, RoundingMode.HALF_UP).doubleValue();

            return value >= 0.2 ? HAS_JS : NO_JS;
        }catch (Exception e){
            logger.warn("calculateValue exception",e);
            return NO_JS;
        }
    }

    /**
     * 从数据库中查询 数据
     * @param key
     * @return
     */
    private Integer findAdvertExpandByAdvertId(Long key) {
        return advertExpandDAO.selectWhiteListById(key);
    }


    public Integer getAdvertExpandById(Long advertId) {
        if(null == advertId){
            return null;
        }

        try {
            return ADVERT_EXPAND_CACHE.get(advertId).orElse(null);
        } catch (ExecutionException e) {
            logger.error("get getAdvertExpandDtoByAdvertId error", e);
            return findAdvertExpandByAdvertId(advertId);
        }
    }

    public void refreshAdvertExpandCache(List<Long> advertIds) {
        for (Long advertId : advertIds) {
            ADVERT_EXPAND_CACHE.refresh(advertId);
        }
    }
}
