package cn.com.duiba.tuia.filter.service.impl;

import java.util.*;
import java.util.stream.Collectors;

import cn.com.duiba.tuia.domain.dataobject.AdvertFilterKeywordDO;
import cn.com.duiba.tuia.domain.dataobject.GeneralFilterKeywordDO;
import cn.com.duiba.tuia.domain.dataobject.NewTradeFilterKeywordDO;
import cn.com.duiba.tuia.enums.CatGroupEnum;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import cn.com.duiba.boot.utils.WarningUtils;
import cn.com.duiba.tuia.cache.MediaCacheService;
import cn.com.duiba.tuia.cache.ServiceManager;
import cn.com.duiba.tuia.constants.ErrorCode;
import cn.com.duiba.tuia.constants.PackagePlanConstants;
import cn.com.duiba.tuia.domain.model.AdvOrientationItem;
import cn.com.duiba.tuia.domain.model.AdvQueryParam;
import cn.com.duiba.tuia.domain.model.AdvertFilter;
import cn.com.duiba.tuia.domain.model.AppDetail;
import cn.com.duiba.tuia.domain.model.CatMonitorWarnThreshold;
import cn.com.duiba.tuia.domain.model.FilterResult;
import cn.com.duiba.tuia.domain.model.ShieldStrategyVO;
import cn.com.duiba.tuia.domain.vo.AdvertFilterVO;
import cn.com.duiba.tuia.domain.vo.AdvertOrientationPackageVO;
import cn.com.duiba.tuia.domain.vo.AdvertPriceVO;
import cn.com.duiba.tuia.exception.TuiaException;
import cn.com.duiba.tuia.filter.service.DirectAdvertFilterService;
import cn.com.duiba.tuia.service.AdvertOrientationService;
import cn.com.duiba.tuia.service.AdvertPreFilterService;
import cn.com.duiba.tuia.service.CommonService;
import cn.com.duiba.tuia.service.ShieldingStrategyService;
import cn.com.duiba.tuia.strategy.StrategyBeans;
import cn.com.duiba.tuia.tool.CatUtil;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.tuia.advert.constants.CommonConstant;
import cn.com.tuia.advert.enums.ChargeTypeEnum;
import cn.com.tuia.advert.model.DirectObtainAdvertReq;
import cn.com.tuia.advert.model.ObtainAdvertRsp;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2017年08月22日 11:17
 * @descript:
 * @version: 1.0
 */
@Service
public class DirectAdvertFilterServiceImpl implements DirectAdvertFilterService {

    private final static Logger logger = LoggerFactory.getLogger(DirectAdvertFilterServiceImpl.class);

    @Autowired
    private CommonService commonService;

    @Autowired
    private MediaCacheService mediaCacheService;

    @Autowired
    private AdvertOrientationService advertOrientationService;

    @Autowired
    private AdvertPreFilterService advertPreFilterService;

    @Autowired
    private ServiceManager serviceManager;
    
    @Autowired
    private  CatMonitorWarnThreshold catMonitorWarnThreshold;

    @Override
    public List<AdvOrientationItem> filter(DirectObtainAdvertReq req) throws Exception {

        try {

            DBTimeProfile.enter("DirectAdvertFilterServiceImpl.filter");
            //1. 校验参数
            this.checkAdvertReq(req);

            // for fix sonar
            if (req == null) {
                return null;
            }

            AdvertFilter advertFilter = new AdvertFilter();
            FilterResult filterResult = new FilterResult();

            advertFilter.setAppId(req.getAppId());
            advertFilter.setSlotId(req.getSlotId());
            advertFilter.setTime(DateUtils.getSecondStr(new Date()));

            //2.获取ip对应的城市id
            AdvQueryParam advQueryParam = commonService.ipGeoAnalysis(req.getIp());
            String cityId = advQueryParam.getRegionId();
            
            //3.获取当前请求的应用信息和屏蔽策略
            ObtainAdvertRsp obtainAdvertRsp = new ObtainAdvertRsp();
            AppDetail appDetail = mediaCacheService.getAppDetailCache(req.getAppId(), req.getSlotId(), obtainAdvertRsp, req.getProxy(), cityId, req);
            ShieldStrategyVO shieldStrategyVO = appDetail.getShieldStrategyVO();
            filterResult.setAdvBannedTag(shieldStrategyVO.getAdvBannedTag());

            //4.封装过滤参数
            advQueryParam = advertPreFilterService.buildDirectAdvQueryParam(advQueryParam, req, shieldStrategyVO);
            
            //5.定向配置包过滤
            advQueryParam.setActivityType(PackagePlanConstants.DIRECT_PAGE_ACTIVITY_TYPE);
            List<AdvOrientationItem> validAdvertResults = advertPreFilterService.directAdvertOrientationWithEsHystrix(advQueryParam, req, filterResult, advertFilter);
            //关键词过滤落地页维度
            validAdvertResults = filterKeyword(validAdvertResults, advQueryParam);

            if(CollectionUtils.isEmpty(validAdvertResults)){
                return Collections.emptyList();
            }
            logger.info("directAdvert filter=[{}]", JSON.toJSONString(advertFilter));

            //7.媒体白名单过滤
            ShieldingStrategyService shieldingStrategy = StrategyBeans.shieldingStrategyMap.get(obtainAdvertRsp.getStrategyType());

            
            //8.查询该应用的白名单列表
            List<Long> appWhiteList = mediaCacheService.getWhiteList(appDetail, shieldingStrategy);
            if (CollectionUtils.isEmpty(appWhiteList)) {
                return validAdvertResults;
            }
            
            List<AdvOrientationItem> advertWhitelist = Lists.newArrayList();
            
            Map<Long, Integer> validAdvertOrderLevelMap = serviceManager.queryValidAdvertOrderLevel();
            Integer validAdvertIdsSize = validAdvertOrderLevelMap.size();
            // 有效广告排序
            validAdvertResults = validAdvertResults.stream().map(dto -> {
                        Integer order = Optional.ofNullable(validAdvertOrderLevelMap.get(dto.getAdvertId())).orElse(validAdvertIdsSize);
                        dto.setOrder(order);
                        return dto;
                    })
                    .sorted(Comparator.comparing(AdvOrientationItem::getOrder))
                    .collect(toList());
            
            Map<Long,List<AdvOrientationItem>> resultMap = Maps.newLinkedHashMap();
            for(AdvOrientationItem result : validAdvertResults){
                List<AdvOrientationItem> results = resultMap.get(result.getAdvertId());
                if(null == results){
                    results = Lists.newArrayList();
                }
                results.add(result);
                resultMap.put(result.getAdvertId(), results);
            }
            
            for (Long advertId : appWhiteList) {
                if (resultMap.containsKey(advertId)) {
                    advertWhitelist.addAll(resultMap.get(advertId));
                }
            }
            
            //白名单不可投且福袋开关开启，则发默认广告列表
            if(advertWhitelist.isEmpty() && shieldingStrategy.isSendLuckybag(appDetail.getObjParam())){
                return validAdvertResults;
            }
            
            return advertWhitelist;

        } catch (TuiaException e) {
            if (e instanceof TuiaException) {
                TuiaException ex = (TuiaException) e;
                CatUtil.log("directAdvertFilterError"+ ex.getResultCode());
                WarningUtils.markThresholdWarning("directAdvertFilterError", catMonitorWarnThreshold.getDirectAdvertError());
            } else {
                CatUtil.log(CatGroupEnum.CAT_102017.getCode());
                WarningUtils.markThresholdWarning("directAdvertFilterException", catMonitorWarnThreshold.getDirectAdvertExc());
                logger.error("直投广告过滤异常", e);
            }
        } finally {
            DBTimeProfile.release();
        }

        return Collections.emptyList();
    }
    
    @Override
    @SuppressWarnings("squid:S3776")
    public Map<Long, AdvertFilterVO> getValidAdvertPrice(List<AdvOrientationItem> validAdvertIds,List<Long> paramsAdvertIds){
        //保证有序性,group要使用linkdedhashmap
        Map<Long, AdvertFilterVO> advertFilterVOMap = Maps.newLinkedHashMap();
        validAdvertIds.stream().collect(groupingBy(AdvOrientationItem::getAdvertId, LinkedHashMap::new, toList())).forEach((Long advertId, List<AdvOrientationItem> list) -> {
            SortedSet<AdvertPriceVO> advertPriceVOSet = Sets.newTreeSet();
            SortedSet<AdvertPriceVO> manualAdvertPriceVOSet = Sets.newTreeSet();

            List<AdvertOrientationPackageVO> advertOrientationPackageVOS = advertOrientationService.getOrientationList(advertId);
            Map<Long,AdvertOrientationPackageVO> advertOrientationPackageVOMap = advertOrientationPackageVOS.stream().filter(Objects::nonNull)
                    .collect(Collectors.toMap(AdvertOrientationPackageVO::getId, x -> x, (oldVal, newVal) -> newVal));

            for (AdvOrientationItem advOrientationItem : list) {
                //传递过来的广告列表不包含该广告，则过滤掉
                if(!paramsAdvertIds.contains(advOrientationItem.getAdvertId())){
                    continue;
                }
                //如果该配置为CPA，则过滤掉
                if(advOrientationItem.getChargeType().equals(ChargeTypeEnum.TYPE_CPA.getCode())){
                    continue;
                }
                //直投过滤掉扶持的广告
                if (advOrientationItem.getSupportStatus() == CommonConstant.YES) {
                    continue;
                }
//                //如果包中设置了活动类型，则直投只取直投页活动
//                List<String> activityType = advOrientationItem.getActivityType();
//                if (CollectionUtils.isNotEmpty(activityType) && !activityType.contains(PackagePlanConstants.PACKAGE_PLAN_DEFAULT_TYPE)
//                        && !activityType.contains(PackagePlanConstants.DIRECT_PAGE_ACTIVITY_TYPE)) {
//                    continue;
//                }

                AdvertOrientationPackageVO orientation = advertOrientationPackageVOMap.get(advOrientationItem.getOrientationId());
                if (orientation == null) {
                    continue;
                }
                //直投过滤掉自动托管配置,直投只投放人工包，非人工的都过滤掉
                if (advOrientationItem.isTrusteeshipOrientPackage()) {
                    continue;
                }
                AdvertPriceVO priceVO = new AdvertPriceVO(Collections.emptyList(), advOrientationItem.getCpcPrice(),advOrientationItem.getChargeType());
                priceVO.setAdvertId(advertId);
                priceVO.setAdvertOrientationPackageId(advOrientationItem.getOrientationId());
                priceVO.setOriginalOrientationId(advOrientationItem.getInitialOrientationId());
                priceVO.setPackageType(advOrientationItem.getPackageType());
                priceVO.setActivityType(advOrientationItem.getActivityType());
                advertPriceVOSet.add(priceVO);
                manualAdvertPriceVOSet.add(priceVO);
            }

            if (advertPriceVOSet.size() > 0) {
                AdvertFilterVO advertFilterVO = new AdvertFilterVO();
                advertFilterVO.setManualAdvertSet(manualAdvertPriceVOSet);
                advertFilterVO.setAdvertId(advertId);
                advertFilterVO.setAccountId(list.get(0).getAccountId());
                advertFilterVO.setAdvertPriceVOSortedSet(advertPriceVOSet);
                advertFilterVOMap.put(advertId, advertFilterVO);
            }
        });
        return advertFilterVOMap;
    }

    /**
     * 校验兑吧请求时的输入参数
     *
     * @param req the req
     * @throws TuiaException the tuia exception
     */
    private void checkAdvertReq(DirectObtainAdvertReq req) throws TuiaException {

        try {
            DBTimeProfile.enter("DirectAdvertFilterServiceImpl.checkAdvertReq");
            if (req == null) {
                logger.warn("checkAdvertReq error, req = [{}], please check the ，req is null");
                throw new TuiaException(ErrorCode.E0100002);
            }

            if (CollectionUtils.isEmpty(req.getAdvertIds())) {
                logger.warn("checkAdvertReq error, req = [{}], please check the", req);
                throw new TuiaException(ErrorCode.E0100002);
            }

            // 关键请求参数判断非空
            if (req.getConsumerId() == null || req.getAppId() == null) {
                logger.warn("checkAdvertReq error, req = [{}], please check the", req);
                throw new TuiaException(ErrorCode.E0100002);
            }

            if (StringUtils.isEmpty(req.getUa()) || StringUtils.isEmpty(req.getIp())) {
                logger.warn("checkAdvertReq error, req = [{}], please check the", req);
                throw new TuiaException(ErrorCode.E0100002);
            }

            if ("unknown".equals(req.getUa())) {
                req.setUa("unknow");
            }

        } finally {
            DBTimeProfile.release();
        }
    }

    /**
     * 关键词屏蔽
     * @param list
     * @param advQueryParam
     * @return
     */
    private List<AdvOrientationItem> filterKeyword(List<AdvOrientationItem> list, AdvQueryParam advQueryParam) {
        AdvertFilterKeywordDO advertFilterKeywordDO = advQueryParam.getAdvertFilterKeywordDO();

        if (CollectionUtils.isNotEmpty(list) && Objects.nonNull(advertFilterKeywordDO)) {
            List<AdvOrientationItem> copyList = new ArrayList<>(list);

            NewTradeFilterKeywordDO newTradeFilterKeywordDO = advertFilterKeywordDO.getNewTradeFilterKeywordDO();
            if (Objects.nonNull(newTradeFilterKeywordDO) && CollectionUtils.isNotEmpty(newTradeFilterKeywordDO.getShieldNormalLandingList())) {
                //被屏蔽的素材
                List<AdvOrientationItem> shieldList = Lists.newArrayList();
                for (AdvOrientationItem item : copyList) {
                    if (newTradeFilterKeywordDO.getShieldNormalLandingList().contains(item.getAdvertId())) {
                        shieldList.add(item);
                    }
                }

                if (CollectionUtils.isNotEmpty(shieldList)) {
                    copyList.removeAll(shieldList);
                }
            }

            GeneralFilterKeywordDO generalFilterKeywordDO = advertFilterKeywordDO.getGeneralFilterKeywordDO();
            if (Objects.nonNull(generalFilterKeywordDO) && CollectionUtils.isNotEmpty(generalFilterKeywordDO.getShieldNormalLandingList())) {
                //被屏蔽的素材
                List<AdvOrientationItem> shieldList = Lists.newArrayList();
                for (AdvOrientationItem item : copyList) {
                    if (generalFilterKeywordDO.getShieldNormalLandingList().contains(item.getAdvertId())) {
                        shieldList.add(item);
                    }
                }

                if (CollectionUtils.isNotEmpty(shieldList)) {
                    copyList.removeAll(shieldList);
                }
            }

            return copyList;
        }

        return list;
    }
}
