package cn.com.duiba.tuia.filter.service.impl;

import cn.com.duiba.bigdata.dmp.service.api.remoteservice.dto.DeviceTagDto;
import cn.com.duiba.bigdata.dmp.service.api.remoteservice.dto.BusinessTagDto;
import cn.com.duiba.bigdata.dmp.service.api.remoteservice.dto.OneIdFeatureDto;
import cn.com.duiba.bigdata.dmp.service.api.remoteservice.dto.OneIdFeatureOfflineDto;
import cn.com.duiba.bigdata.dmp.service.api.remoteservice.form.OneIdFeatureForm;
import cn.com.duiba.bigdata.dmp.service.api.remoteservice.remoteservice.RemoteOneIdFeatureService;
import cn.com.duiba.nezha.engine.api.dto.ConsumerDto;
import cn.com.duiba.nezha.engine.api.dto.OneIdDataDto;
import cn.com.duiba.nezha.engine.api.dto.TagStat;
import cn.com.duiba.tuia.cache.ServiceManager;
import cn.com.duiba.tuia.constants.AdvertDataTypeConstant;
import cn.com.duiba.tuia.constants.AdvertReqLogExtKeyConstant;
import cn.com.duiba.tuia.domain.dataobject.ConsumerInteractiveRecordDO;
import cn.com.duiba.tuia.domain.dataobject.TradeTagRuleScopeDO;
import cn.com.duiba.tuia.domain.model.CatMonitorWarnThreshold;
import cn.com.duiba.tuia.domain.model.FilterResult;
import cn.com.duiba.tuia.domain.model.TagRuleScope;
import cn.com.duiba.tuia.domain.vo.TagDataVO;
import cn.com.duiba.tuia.domain.vo.TradeRuleRadixVO;
import cn.com.duiba.tuia.enums.AdvertTradeAcceptLevelEnum;
import cn.com.duiba.tuia.enums.CatGroupEnum;
import cn.com.duiba.tuia.exception.TuiaException;
import cn.com.duiba.tuia.filter.service.InterestAdvertTagFilter;
import cn.com.duiba.tuia.filter.service.TradeTagRuleFilterService;
import cn.com.duiba.tuia.service.DmpTagDataService;
import cn.com.duiba.tuia.service.TradeTagRuleService;
import cn.com.duiba.tuia.tool.CatUtil;
import cn.com.duiba.tuia.utils.TuiaStringUtils;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.tuia.advert.constants.SystemConfigKeyConstant;
import cn.com.tuia.advert.model.ObtainAdvertReq;
import com.alibaba.fastjson.JSON;
import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;

import static java.util.stream.Collectors.toList;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2018年02月08日 17:40
 * @descript:
 * @version: 1.0
 */
@Service
public class TradeTagRuleFilterServiceImpl implements TradeTagRuleFilterService {

    private final Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private TradeTagRuleService tradeTagRuleService;
    
    @Autowired
    private InterestAdvertTagFilter interestAdvertTagFilter;
    
    @Autowired
    private ServiceManager serviceManager;

    @Autowired
    private DmpTagDataService dmpTagDataService;

    private static final String CONSUME_TAG_JOINER = ",";

    private static final String TAG_TRADE_ACCEPT_JOINER = "-";

    @Autowired
    private  CatMonitorWarnThreshold catMonitorWarnThreshold;

    @Resource
    private RemoteOneIdFeatureService remoteOneIdFeatureService;

    /**
     * 获取标签的数据
     *
     * @param tagRuleDataMap
     * @param ruleMap 
     * @return
     */

    private Map<String, TagDataVO> buildTagData(Map<String, Long> tagRuleDataMap, Map<String, TradeTagRuleScopeDO> ruleMap) {

        Map<String, TagDataVO> advertDataVOMap = Maps.newHashMap();
        for (Map.Entry<String, Long> entry : tagRuleDataMap.entrySet()) {

            String key = entry.getKey();
            String tagId = Splitter.on("-").splitToList(key).get(1);
            if (null == ruleMap.get(tagId)) {
                continue;
            }
            Long value = entry.getValue();
            TagDataVO tagDataVO = advertDataVOMap.get(tagId);
            if (tagDataVO == null) {
                tagDataVO = new TagDataVO();
                tagDataVO.setTagId(tagId);
                advertDataVOMap.put(tagId, tagDataVO);
            }

            if (key.startsWith(AdvertDataTypeConstant.LAUNCH)) {
                tagDataVO.setLaunch(value);
            } else if (key.startsWith(AdvertDataTypeConstant.EFCLICK)) {
                tagDataVO.setEfclick(value);
            } else if (key.startsWith(AdvertDataTypeConstant.EFFECT)) {
                tagDataVO.setEffect(value);
            }
        }

        return advertDataVOMap;
    }

    /**
     * 计算行为规则
     *
     * @param consumerDto
     * @param advertDataVOMap
     * @param tagRuleDOMap
     * @param userAllRoleTagNum 
     * @return
     * @throws TuiaException
     */
    @SuppressWarnings("squid:S3776")
    private List<String> evelRule(ConsumerDto consumerDto, Map<String, TagDataVO> advertDataVOMap,
                          Map<String, TradeTagRuleScopeDO> tagRuleDOMap, List<String> userAllRoleTagNum) throws TuiaException {

        Set<String> clickInterestedTags = Sets.newHashSetWithExpectedSize(advertDataVOMap.size());
        Set<String> clickNotInterestedTags = Sets.newHashSetWithExpectedSize(advertDataVOMap.size());
        Set<String> effectInterestedTags = Sets.newHashSetWithExpectedSize(advertDataVOMap.size());
        Set<String> effectNotInterestedTags = Sets.newHashSetWithExpectedSize(advertDataVOMap.size());

        List<TagStat> tagStatList = Lists.newArrayListWithCapacity(advertDataVOMap.size());
        List<String> userTradeAccept = Lists.newArrayList();
        if (advertDataVOMap.isEmpty()) {
            return userTradeAccept;
        }
        
        for (String tagNum : userAllRoleTagNum) {
            TagDataVO tagDataVO = advertDataVOMap.get(tagNum);
            StringBuilder tagNumAccept = new StringBuilder();
            tagNumAccept.append(tagNum);
            tagNumAccept.append(TAG_TRADE_ACCEPT_JOINER);
            // 没有打分规则或没有分数，则不能判断出用户对行业的兴趣接受度，归为行业标签-其他
            if (null == tagDataVO || null == tagDataVO.getTagId() || null == tagRuleDOMap.get(tagDataVO.getTagId())) {
                tagNumAccept.append(AdvertTradeAcceptLevelEnum.OTHER.getCode());
                userTradeAccept.add(tagNumAccept.toString());
                continue;
            }
            
            TradeTagRuleScopeDO tagRuleDO = tagRuleDOMap.get(tagDataVO.getTagId());
            

            int score = this.eval(tagDataVO);
            Boolean ifHitRule = false;
            
            // 对点击感兴趣 - 接受型
            if (isClickInterest(score, tagRuleDO)) {
                clickInterestedTags.add(tagNum);
                ifHitRule = true;
                tagNumAccept.append(AdvertTradeAcceptLevelEnum.ACCEPT.getCode());
            }

            // 对点击不感兴趣 - 无感型
            if (!ifHitRule && isClickNoInterest(score, tagRuleDO)) {
                clickNotInterestedTags.add(tagNum);
                ifHitRule = true;
                tagNumAccept.append(AdvertTradeAcceptLevelEnum.NOSENSE.getCode());
            }

            // 对转化感兴趣 - 喜欢型
            if (!ifHitRule && isEffectInterest(score, tagRuleDO)) {
                effectInterestedTags.add(tagNum);
                ifHitRule = true;
                tagNumAccept.append(AdvertTradeAcceptLevelEnum.LIKE.getCode());
            }
            
            // 对转化不感兴趣 - 排斥型
            if (!ifHitRule && isEffectNoInterest(score, tagRuleDO)) {
                effectNotInterestedTags.add(tagNum);
                ifHitRule = true;
                tagNumAccept.append(AdvertTradeAcceptLevelEnum.REJECT.getCode());
            }

            // 分数不在得分规则内
            if (!ifHitRule) {
                tagNumAccept.append(AdvertTradeAcceptLevelEnum.OTHER.getCode());
            }
            userTradeAccept.add(tagNumAccept.toString());
            
            if (tagDataVO.getLaunch() != 0) {
                //组装传给nezha的数据
                TagStat tagStat = new TagStat();
                tagStat.setTagId(tagDataVO.getTagId());
                tagStat.setClick(tagDataVO.getEfclick());
                tagStat.setLaunch(tagDataVO.getLaunch());
                tagStat.setConvert(tagDataVO.getEffect());
                tagStat.setScore((double) score);
                tagStatList.add(tagStat);
            }
        }


        //设置这些列表会传给nezha
        consumerDto.setTagStats(tagStatList);
        consumerDto.setClickIntersredTags(Joiner.on(CONSUME_TAG_JOINER).join(clickInterestedTags));
        consumerDto.setClickUnintersredTags(Joiner.on(CONSUME_TAG_JOINER).join(clickNotInterestedTags));
        consumerDto.setConverIntersredTags(Joiner.on(CONSUME_TAG_JOINER).join(effectInterestedTags));
        consumerDto.setConverUnintersredTags(Joiner.on(CONSUME_TAG_JOINER).join(effectNotInterestedTags));
        
        return userTradeAccept;
    }

    /**
     * 
     * isClickInterest:(对点击感兴趣,肯定有最小阈值). <br/>
     *
     * @author chencheng
     * @param score
     * @param tagRuleDO
     * @return
     * @since JDK 1.8
     */
    private boolean isClickInterest(int score, TradeTagRuleScopeDO tagRuleDO) {

        if (tagRuleDO.getClickInterestedMin() != null && score >= tagRuleDO.getClickInterestedMin()) {

            if (tagRuleDO.getClickInterestedMax() != null && score <= tagRuleDO.getClickInterestedMax()) {
                return Boolean.TRUE;
            }

            if (tagRuleDO.getClickInterestedMax() == null) {
                return Boolean.TRUE;
            }

            return Boolean.FALSE;
        }

        return Boolean.FALSE;
    }

    /**
     * 
     * isClickNoInterest:(对点击不感兴趣,肯定有最大阈值). <br/>
     *
     * @author chencheng
     * @param score
     * @param tagRuleDO
     * @return
     * @since JDK 1.8
     */
    private boolean isClickNoInterest(int score, TradeTagRuleScopeDO tagRuleDO) {

        if (tagRuleDO.getClickNoInterestedMax() != null && score <= tagRuleDO.getClickNoInterestedMax()) {
            
            if (tagRuleDO.getClickNoInterestedMin() != null && score >= tagRuleDO.getClickNoInterestedMin()) {
                return Boolean.TRUE;
            }

            if (tagRuleDO.getClickNoInterestedMin() == null) {
                return Boolean.TRUE;
            }

            return Boolean.FALSE;
        }

        return Boolean.FALSE;
    }

    /**
     * 
     * isEffectInterest:(对转化感兴趣,只比较最小值即可). <br/>
     *
     * @author chencheng
     * @param score
     * @param tagRuleDO
     * @return
     * @since JDK 1.8
     */
    private boolean isEffectInterest(int score, TradeTagRuleScopeDO tagRuleDO) {
        return tagRuleDO.getEffectInterestedMin() != null && score >= tagRuleDO.getEffectInterestedMin();
    }

    /**
     * 
     * isEffectNoInterest:(对转化感兴趣,只比较最大值即可). <br/>
     *
     * @author chencheng
     * @param score
     * @param tagRuleDO
     * @return
     * @since JDK 1.8
     */
    private boolean isEffectNoInterest(int score, TradeTagRuleScopeDO tagRuleDO) {
        return tagRuleDO.getEffectNoInterestedMax() != null && score <= tagRuleDO.getEffectNoInterestedMax();
    }

    /**
     * 计算规则最后得分
     *
     * @param tagDataVO
     * @return
     */
    private int eval(TagDataVO tagDataVO) throws TuiaException {

        TradeRuleRadixVO tradeRuleRadixVO = this.getRuleRadix();
        return (int) (tagDataVO.getLaunch() * tradeRuleRadixVO.getLaunchRadix()
                + tagDataVO.getEfclick() * tradeRuleRadixVO.getEfclickRadix()
                + tagDataVO.getEffect() * tradeRuleRadixVO.getEffectRadix());
    }

    /**
     * 获取规则计算的基数
     *
     * @return Set<Long> 测试广告列表
     * @throws TuiaException 业务异常
     */
    private TradeRuleRadixVO getRuleRadix() throws TuiaException {

        String str = serviceManager.getStrValue(SystemConfigKeyConstant.TUIA_TRADE_TAG_RULE_RADIX);
        List<Integer> radixs = TuiaStringUtils.getIntegerListByStr(str, ",");
        TradeRuleRadixVO radixVO = new TradeRuleRadixVO();
        radixVO.setLaunchRadix(radixs.get(0));
        radixVO.setEfclickRadix(radixs.get(1));
        radixVO.setEffectRadix(radixs.get(2));
        return radixVO;
    }

    /**
     * 
     * getUserTradeAccept:(用户行业接受度). <br/>
     * @author chencheng
     * @param consumerDto 用户信息
     * @param consumerVOList 用户领取记录
     * @return
     * @since JDK 1.8
     */
    private List<String> getUserTradeAccept(ConsumerDto consumerDto, List<ConsumerInteractiveRecordDO> consumerVOList, DeviceTagDto deviceTagDto, ObtainAdvertReq req) {
        try {

            //设置oneId的用户行为特征信息
            setOneIdTagStats(consumerDto, req);

            //获取所有规则标签
            TagRuleScope tagRuleScope = tradeTagRuleService.getTagScopeAll();
            
            //新用户
            if (CollectionUtils.isEmpty(consumerVOList)) {
                return getAllTagNumOther(tagRuleScope.getUserAllRoleTagNum());
            }
            
            // 查询用户数据
            Map<String, Long> tagRuleDataMap = convertHbaseDate(deviceTagDto);

            // 用户无数据.无法判断用户对于行业的兴趣爱好。会中定向用户兴趣度为行业-其他或不限的广告
            if (tagRuleDataMap == null || tagRuleDataMap.size() == 0) {
                CatUtil.catLog(CatGroupEnum.CAT_107003.getCode());
                return getAllTagNumOther(tagRuleScope.getUserAllRoleTagNum());
            }

            //获取标签tagnum对应的发券转化数据
            Map<String, TagDataVO> advertDataVOMap = this.buildTagData(tagRuleDataMap, tagRuleScope.getTagRuleDOMap());

            //获取感兴趣和不感兴趣的标签
            return this.evelRule(consumerDto, advertDataVOMap, tagRuleScope.getTagRuleDOMap(), tagRuleScope.getUserAllRoleTagNum());

        } catch (Exception e) {
            logger.error("获取用户行业标签兴趣点异常", e);
            return Lists.newArrayList();
        } 
    }

    /**
     * 设置oneId的用户行为特征信息
     * @param consumerDto
     */
    private void setOneIdTagStats(ConsumerDto consumerDto, ObtainAdvertReq req) {

        try {
            Map<String, String> logExtMap = Optional.ofNullable(req.getLogExtMap()).orElse(new HashMap<>());
            String imeiMd5 = logExtMap.get(AdvertReqLogExtKeyConstant.IMEI_MD5);
            String idfaMd5 = logExtMap.get(AdvertReqLogExtKeyConstant.IDFA_MD5);
            String oaidMd5 = logExtMap.get(AdvertReqLogExtKeyConstant.OAID_MD5);

            OneIdFeatureForm form = new OneIdFeatureForm();
            if (StringUtils.isAllBlank(consumerDto.getOneId(), consumerDto.getOneIdType(), imeiMd5, idfaMd5, oaidMd5)) {
                // 如果oneId与设备信息不存在，取deviceid
                form.setOneId(req.getDeviceId());
                form.setOneIdType(2);
            } else {
                //如果oneId不存在，不再查询oneId的用户行为标签
                if (StringUtils.isNotBlank(consumerDto.getOneId()) && StringUtils.isNotBlank(consumerDto.getOneIdType())) {
                    form.setOneId(consumerDto.getOneId());
                    form.setOneIdType(Integer.valueOf(consumerDto.getOneIdType()));
                }
                form.setImeiMd5(imeiMd5);
                form.setIdfaMd5(idfaMd5);
                form.setOaidMd5(oaidMd5);
            }
            form.setActivityId(req.getActivityId());
            OneIdFeatureDto oneIdFeatureDto = remoteOneIdFeatureService.getFeatureByOneIdAndImei(form, 20L);
            if (oneIdFeatureDto == null) {
                return;
            }
            //如果没有oneId 信息 oneId信息设置到全链路
            if(StringUtils.isBlank(consumerDto.getOneId())){
                Map<String, String> logExtExpMap = Optional.ofNullable(req.getLogExtExpMap()).orElse(new HashMap<>());
                logExtExpMap.put(AdvertReqLogExtKeyConstant.ONE_ID,oneIdFeatureDto.getOneId());
                logExtExpMap.put(AdvertReqLogExtKeyConstant.ONE_ID_TYPE,String.valueOf(oneIdFeatureDto.getOneIdType()));
                req.setLogExtExpMap(logExtExpMap);
            }
            //组装nezha 特征数据
            OneIdDataDto oneIdDataDto=new OneIdDataDto();
            Optional.ofNullable(oneIdFeatureDto.getOnlineDto()).ifPresent(val->BeanUtils.copy(val,oneIdDataDto));
            Optional.ofNullable(oneIdFeatureDto.getOfflineDto()).ifPresent(val->{
                BeanUtils.copy(val,oneIdDataDto);
                List<String> tagIds = new ArrayList<>();
                List<Long> launchs = new ArrayList<>();
                List<Long> clicks = new ArrayList<>();
                List<Long> converts = new ArrayList<>();
                //设置标签对应的发券数据，以发券数据的标签为准
                Optional.ofNullable(val.getAdvertMatchTagLaunchPv()).ifPresent(vals -> {
                    for (Map.Entry<String, Long> entry : vals.entrySet()) {
                        tagIds.add(entry.getKey());
                        launchs.add(entry.getValue() != null ? entry.getValue() : 0L);
                        //设置标签对应的点击数据
                        if (val.getAdvertMatchTagClickPv() == null) {
                            clicks.add(0L);
                        } else {
                            clicks.add(Optional.ofNullable(val.getAdvertMatchTagClickPv().get(entry.getKey())).orElse(0L));
                        }

                        //设置标签对应的转化数据
                        if (val.getAdvertMatchTagEffectPv() == null) {
                            converts.add(0L);
                        } else {
                            converts.add(Optional.ofNullable(val.getAdvertMatchTagEffectPv().get(entry.getKey())).orElse(0L));
                        }
                    }
                    Joiner joiner = Joiner.on(",");
                    if(CollectionUtils.isNotEmpty(tagIds)){
                        oneIdDataDto.setOUIIds(joiner.join(tagIds));
                        oneIdDataDto.setOUILaunchPV(joiner.join(launchs));
                        oneIdDataDto.setOUIClickPv(joiner.join(clicks));
                        oneIdDataDto.setOUIEffectPv(joiner.join(converts));
                    }

                });
            });
            consumerDto.setOneIdDataDto(oneIdDataDto);

        } catch (Exception e) {
            logger.error("获取oneId用户行业标签兴趣点异常", e);
        }
    }


    /**
     * 将hbase返回的用户行业接受度 转为map
     * @param deviceTagDto
     * @return
     */
    private Map<String,Long> convertHbaseDate(DeviceTagDto deviceTagDto){

        Map<String,Long> map = new HashMap<>();


        if(deviceTagDto == null){
            return map;
        }

        List<BusinessTagDto> businessTagDtos = deviceTagDto.getBusinessTag();

        if(CollectionUtils.isEmpty(businessTagDtos)){
            return map;
        }

        businessTagDtos.forEach(dto->{

            map.putAll(buildTagMap(dto));
        });

        return map;
    }


    /**
     * 构建map
     * @param businessTagDto
     * @return
     */
    private Map<String,Long> buildTagMap(BusinessTagDto businessTagDto){

        String tagId = businessTagDto.getBusinessTagId();
        Map<String,Long> map = new HashMap<>();
        AdvertDataTypeConstant.TAG_LISTS.forEach(tagName->{


            switch (tagName){
                case AdvertDataTypeConstant.EFCLICK:
                    map.put(AdvertDataTypeConstant.EFCLICK+"-"+tagId,businessTagDto.getEFClickPV());
                    break;
                case AdvertDataTypeConstant.EFFECT:
                    map.put(AdvertDataTypeConstant.EFFECT+"-"+tagId,businessTagDto.getLandpageClick());
                    break;
                case AdvertDataTypeConstant.LAUNCH:
                    map.put(AdvertDataTypeConstant.LAUNCH+"-"+tagId,businessTagDto.getLaunch());
                    break;
                default:
                    break;

            }

        });

        return map;
    }

    
    /**
     * 
     * getAllTagNumOther:(所有用户兴趣打分的其他兴趣点). <br/>
     *
     * @author chencheng
     * @return
     * @since JDK 1.8
     */
    private List<String> getAllTagNumOther(List<String> allLevelTowTag) {
        return allLevelTowTag.stream().map(tagNum -> {
            return new StringBuilder(tagNum).append(TAG_TRADE_ACCEPT_JOINER).append(AdvertTradeAcceptLevelEnum.OTHER.getCode()).toString();
        }).collect(toList()); 
    }


    @Override
    public List<String> getUserInterest(ConsumerDto consumerDto, List<ConsumerInteractiveRecordDO> consumerVOList, String deviceId, FilterResult filterResult, ObtainAdvertReq req, DeviceTagDto deviceTagDto, Map<String,String> newUserDmpTagsMap) {

        List<String> userTradeAccept = Lists.newArrayList();
        try {
            DBTimeProfile.enter("TradeTagRuleFilterServiceImpl.getUserInterest");
            // 用户行业接受度(放到后面拿出配置时过滤)
            userTradeAccept = getUserTradeAccept(consumerDto, consumerVOList,deviceTagDto,req);
            // 社会属性兴趣点
            userTradeAccept.addAll(interestAdvertTagFilter.getUserCrowdInterest(deviceId, consumerDto,filterResult,req,deviceTagDto,newUserDmpTagsMap));

            //获取基础属性标签
            userTradeAccept.addAll(dmpTagDataService.getDeviceBaseTag(deviceId));

        } catch (Exception e) {
            logger.error("获取用户行业标签兴趣点异常，deviceId:{}",deviceId, e);
        } finally {
            DBTimeProfile.release();
        }
        return userTradeAccept;

    }
}
