/**
 * Project Name:engine-service
 * File Name:UpdateAdvertOrientPkgMsgChannel.java
 * Package Name:cn.com.duiba.tuia.message.channel
 * Date:2017年8月21日下午5:43:43
 * Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 *
*/

package cn.com.duiba.tuia.message.channel;

import cn.com.duiba.tuia.task.MsgDelayTask;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.tuia.advert.constants.CommonConstant;
import cn.com.tuia.advert.enums.PkgPutTypeEnum;

import com.alibaba.fastjson.JSON;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import cn.com.duiba.tuia.cache.AdvertMapCacheManager;
import cn.com.duiba.tuia.constants.AdvertConstants;
import cn.com.duiba.tuia.dao.engine.AdvertOrientationPackageDAO;
import cn.com.duiba.tuia.domain.dataobject.AdvertOrientationPackageDO;
import cn.com.duiba.tuia.service.AdvertOrientationService;
import cn.com.tuia.advert.message.RedisMessageChannel;
import cn.com.tuia.advert.message.consumer.AbstractRedisMessageHandle;
import cn.com.tuia.advert.message.consumer.RedisMessageHandle;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * ClassName:UpdateAdvertOrientPkgMsgChannel <br/>
 * Function: TODO ADD FUNCTION. <br/>
 * Reason:	 TODO ADD REASON. <br/>
 * Date:     2017年8月21日 下午5:43:43 <br/>
 * @author   zp
 * @version  
 * @since    JDK 1.6
 * @see 	 
 */
@Component
@RedisMessageHandle(RedisMessageChannel.UPDATE_ADVERT_ORIENT_PACKAGE_MSG)
public class UpdateAdvertOrientPkgMsgChannel extends AbstractRedisMessageHandle implements InitializingBean {
    
    @Autowired
    private AdvertOrientationService orientationService;

    @Autowired
    private AdvertOrientationPackageDAO orientationPackageDAO;
    @Autowired
    private AdvertMapCacheManager advertMapCacheManager;
    
    private static final String ORIENTATION_ID = "orientationId";

    private static final String ADVERT_TYPE = "advertType";

    /** 延迟执行的毫秒数 */
    private static final long DELAY_MS = 1000;

    /** 重复消息列表哈希 */
    private static final ConcurrentHashMap<String, List<String>> map = new ConcurrentHashMap<>();

    /** 延时队列 */
    private static final DelayQueue<MsgDelayTask> queue = new DelayQueue<>();

    /** 限制队列长度 */
    private static final Integer QUEUE_SIZE_LIMIT = 1000;

    @Override
    @SuppressWarnings("squid:S3776")
    public Runnable createHandle(RedisMessageChannel channel, String message) {
        
        return new Runnable(){

            @Override
            public void run() {
                logger.info(channel + ";msg=" + message);

                Long orientationId;

                //兼容以前的处理
                if (StringUtils.isNotBlank(message) && StringUtils.isNumeric(message)) {
                    orientationId = Long.parseLong(message);
                } else {
                    Map msgMap = JSON.parseObject(message, Map.class);

                    Object advertTypeObj = msgMap.get(ADVERT_TYPE);
                    Object orientationIdObj = msgMap.get(ORIENTATION_ID);
                    if (orientationIdObj == null) {
                        logger.error("handle orient package massage orientationId is null error");
                        return;
                    }

                    orientationId = Long.valueOf(orientationIdObj.toString());
                    Integer advertType = advertTypeObj == null ? null : (Integer) advertTypeObj;

                    //互动广告也能接收到非互动广告的消息，如果非互动广告的则不处理
                    if (advertType != null && advertType != CommonConstant.HD_ADVERT_TYPE) {
                        return;
                    }
                }

//                AdvertOrientationPackageDO orientationPackageDO = orientationPackageDAO.selectById(orientationId);
//                // 配置投放类型
//                if (orientationPackageDO == null || !orientationPackageDO.getPkgPutType().equals(PkgPutTypeEnum.INTERACTIVE_TYPE.getCode())) {
//                    return;
//                }
//
//                orientationService.updateOrientation(orientationPackageDO);
//                Long advertId = orientationPackageDO.getAdvertId();
//                if (orientationPackageDO.getIsDefault().equals(AdvertConstants.DEFAULT_ORIENTATION)) {
//                    orientationId = 0L;
//                }
//                advertMapCacheManager.updateValidPkgFilterCache(advertId, orientationId);

                preExecute(String.valueOf(orientationId));
            }
        };
    }

    /**
     * 前置处理，延时队列和哈希表添加任务
     *
     * @param orientId 配置ID
     */
    private void preExecute(String orientId) {
        String key = orientId + DateUtils.getSecondOnlyStr(new Date());
        boolean isExist = map.containsKey(key);
        putToMap(orientId, key);

        if (!isExist){
            execute(orientId, false);
        } else {
            if (queue.size() >= QUEUE_SIZE_LIMIT) {
                logger.info("updateAdvertOrientPackageMsg消息, {}, 超出队列限制长度直接执行, orientId={}", DateUtils.getMillisecond(), orientId);
                execute(orientId, false);
                return;
            }

            logger.info("updateAdvertOrientPackageMsg消息, {}, 加入延时队列, orientId={}", DateUtils.getMillisecond(), orientId);
        }

        queue.add(new MsgDelayTask(orientId, key, System.currentTimeMillis() + DELAY_MS));
    }

    private void putToMap(String orientId, String key) {
        map.compute(key, (s, strings) -> {
            if (null == strings) {
                strings = new ArrayList<>();
            }
            strings.add(orientId);
            return strings;
        });
    }

    /**
     * 执行业务逻辑
     *
     * @param orientId 配置ID
     * @param isDelay 是否延迟执行
     */
    private void execute(String orientId, boolean isDelay) {
        Long orientationId = Long.valueOf(orientId);
        AdvertOrientationPackageDO orientationPackageDO = orientationPackageDAO.selectById(orientationId);
        // 配置投放类型
        if (orientationPackageDO == null || !orientationPackageDO.getPkgPutType().equals(PkgPutTypeEnum.INTERACTIVE_TYPE.getCode())) {
            return;
        }

        orientationService.updateOrientation(orientationPackageDO);
        Long advertId = orientationPackageDO.getAdvertId();
        if (orientationPackageDO.getIsDefault().equals(AdvertConstants.DEFAULT_ORIENTATION)) {
            orientationId = 0L;
        }
        advertMapCacheManager.updateValidPkgFilterCache(advertId, orientationId);

        logger.info("updateAdvertOrientPackageMsg消息, {}, {}, orientId={}",
                DateUtils.getMillisecond(), isDelay ? "延迟消费" : "直接消费", orientId);
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        // 用于延迟队列消费的单线程线程池
        ExecutorService executorService = Executors.newSingleThreadExecutor();
        executorService.submit((Runnable) () -> {
            while (true) {
                try {
                    MsgDelayTask task = queue.take();
                    List<String> tasks = map.get(task.getKey());
                    if (null != tasks && tasks.size() > 1) {
                        execute(tasks.get(0), true);
                    }
                    map.remove(task.getKey());
                } catch (Exception e) {
                    logger.error("updateAdvertOrientPackageMsg消息, 延时线程执行异常", e);
                }
            }
        });
    }
}

