package cn.com.duiba.tuia.message.rocketmq.listener;

import cn.com.duiba.tuia.cache.AdvertABTestCacheService;
import cn.com.duiba.tuia.cache.AdvertMapCacheManager;
import cn.com.duiba.tuia.cache.AdvertPkgCacheService;
import cn.com.duiba.tuia.cache.ServiceManager;
import cn.com.duiba.tuia.service.AdvertOrientationService;
import cn.com.duiba.tuia.task.MsgDelayTask;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.tuia.advert.constants.CommonConstant;
import cn.com.tuia.advert.model.messageDto.AdvertMsg;
import com.alibaba.fastjson.JSON;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import cn.com.tuia.advert.cache.RedisCommonKeys;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * 刷新广告缓存消息
 */
@Component
public class RefreshAdvertCacheHandler extends AbstractMessageResultHandler {

    @Autowired
    private AdvertMapCacheManager advertMapCacheManager;
    @Autowired
    private AdvertOrientationService orientationService;
    @Autowired
    private ServiceManager serviceManager;
    @Autowired
    private AdvertABTestCacheService advertABTestCacheService;
    @Autowired
    private AdvertPkgCacheService advertPkgCacheService;

    /** 延迟执行的毫秒数 */
    private static final long DELAY_MS = 1000;

    /** 重复消息列表哈希 */
    private static final ConcurrentHashMap<String, List<String>> map = new ConcurrentHashMap<>();

    /** 延时队列 */
    private static final DelayQueue<MsgDelayTask> queue = new DelayQueue<>();

    /** 限制队列长度 */
    private static final Integer QUEUE_SIZE_LIMIT = 1000;

    @Override
    public String getListenTag() {
        return RedisCommonKeys.KC104.toString();
    }

    @Override
    public void consumer(String message) {
        logger.info("Topic:{},Msg:{}",getListenTag(),message);
        if (StringUtils.isBlank(message)) {
            return;
        }
        AdvertMsg uAdvertMsg = JSON.parseObject(message, AdvertMsg.class);
        int type = uAdvertMsg.getType();
        List<Long> ids = uAdvertMsg.getIds();

        Integer advertType = uAdvertMsg.getAdvertType();
        if (advertType != null && advertType != CommonConstant.HD_ADVERT_TYPE) {
            return;
        }

//        if (type == 0) {
//            advertMapCacheManager.initAllAdvertCache();
//            advertMapCacheManager.initAllValidAdvertFilterCache();
//        } else if (type == 1) {
//            for (Long advertId : ids) {
//                advertMapCacheManager.updateAdvertCache(advertId);
//                //广告优惠券缓存
//                serviceManager.updateAdvertCouponCache(advertId);
//
//                advertMapCacheManager.updateValidAdvertFilterCache(advertId);
//                orientationService.updateDefaultOrientation(advertId);
//            }
//        }

        if (type == 0) {
            preExecute(String.valueOf(type));
        } else if (type == 1) {
            for (Long advertId : ids) {
                preExecute(String.valueOf(advertId));
            }
        }
    }

    /**
     * 前置处理，延时队列和哈希表添加任务
     *
     * @param advertId 广告ID
     */
    private void preExecute(String advertId) {
        String key = advertId + DateUtils.getSecondOnlyStr(new Date());
        boolean isExist = map.containsKey(key);
        putToMap(advertId, key);

        if (!isExist){
            execute(advertId, false);
        } else {
            if (queue.size() >= QUEUE_SIZE_LIMIT) {
                logger.info("KC104消息, {}, 超出队列限制长度直接执行, advertId={}", DateUtils.getMillisecond(), advertId);
                execute(advertId, false);
                return;
            }

            logger.info("KC104消息, {}, 加入延时队列, advertId={}", DateUtils.getMillisecond(), advertId);
        }

        queue.add(new MsgDelayTask(advertId, key, System.currentTimeMillis() + DELAY_MS));
    }

    private void putToMap(String advertId, String key) {
        map.compute(key, (s, strings) -> {
            if (null == strings) {
                strings = new ArrayList<>();
            }
            strings.add(advertId);
            return strings;
        });
    }

    /**
     * 执行业务逻辑
     *
     * @param advertId 广告Id，为0则刷新全部广告
     * @param isDelay 是否延迟执行
     */
    private void execute(String advertId, boolean isDelay) {
        Long id = Long.valueOf(advertId);
        if (id == 0L) {
            advertMapCacheManager.initAllAdvertCache();
            advertMapCacheManager.initAllValidAdvertFilterCache();
        } else {
            advertMapCacheManager.updateAdvertCache(id);
            //广告优惠券缓存
            serviceManager.updateAdvertCouponCache(id);

            advertMapCacheManager.updateValidAdvertFilterCache(id);
            orientationService.updateDefaultOrientation(id);

            //更新爱奇艺特定缓存
            advertPkgCacheService.updateAdvertOrientPkgCache(id);
            advertABTestCacheService.updateAdvertABTestCache(id);
        }

        logger.info("KC104消息, {}, {}, advertId={}", DateUtils.getMillisecond(), isDelay ? "延迟消费" : "直接消费", advertId);
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        //在消息处理器中注册
        RocketMqMessageListener.registerCallback(this);

        // 用于延迟队列消费的单线程线程池
        ExecutorService executorService = Executors.newSingleThreadExecutor();
        executorService.submit((Runnable) () -> {
            while (true) {
                try {
                    MsgDelayTask task = queue.take();
                    List<String> tasks = map.get(task.getKey());
                    if (null != tasks && tasks.size() > 1) {
                        execute(tasks.get(0), true);
                    }
                    map.remove(task.getKey());
                } catch (Exception e) {
                    logger.error("KC104消息, 延时线程执行异常", e);
                }
            }
        });
    }
}
