package cn.com.duiba.tuia.message.rocketmq.listener;

import cn.com.duiba.tuia.task.MsgDelayTask;
import cn.com.duiba.wolf.utils.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import cn.com.duiba.tuia.cache.ServiceManager;
import cn.com.tuia.advert.cache.RedisCommonKeys;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * 
 * ClassName: RefreshValidAdvertSortCacheHandler <br/>
 * Function: 刷新有效广告排序缓存. <br/>
 * date: 2018年11月27日 下午6:28:48 <br/>
 *
 * @author chencheng
 * @version 
 * @since JDK 1.8
 */
@Component
public class RefreshValidAdvertSortCacheHandler extends AbstractMessageResultHandler {
    
    @Autowired
    private ServiceManager serviceManager;

    /** 延迟执行的毫秒数 */
    private static final long DELAY_MS = 1000;

    /** 重复消息列表哈希 */
    private static final ConcurrentHashMap<String, List<String>> map = new ConcurrentHashMap<>();

    /** 延时队列 */
    private static final DelayQueue<MsgDelayTask> queue = new DelayQueue<>();

    /** 限制队列长度 */
    private static final Integer QUEUE_SIZE_LIMIT = 1000;

    @Override
    public String getListenTag() {
        return RedisCommonKeys.KC101.toString();
    }

    @Override
    public void consumer(String message) {
        logger.info("Topic:{},Msg:{}",getListenTag(),message);
//        serviceManager.deleteValidAdvertIdsSortCache();
        preExecute();
    }

    /**
     * 前置处理，延时队列和哈希表添加任务
     */
    private void preExecute() {
        String key = DateUtils.getSecondOnlyStr(new Date());
        boolean isExist = map.containsKey(key);
        putToMap("", key);

        if (!isExist){
            execute(false);
        } else {
            if (queue.size() >= QUEUE_SIZE_LIMIT) {
                logger.info("KC101消息, {}, 超出队列限制长度直接执行", DateUtils.getMillisecond());
                execute(false);
                return;
            }

            logger.info("KC101消息, {}, 加入延时队列", DateUtils.getMillisecond());
        }

        queue.add(new MsgDelayTask("", key, System.currentTimeMillis() + DELAY_MS));
    }

    private void putToMap(String value, String key) {
        map.compute(key, (s, strings) -> {
            if (null == strings) {
                strings = new ArrayList<>();
            }
            strings.add(value);
            return strings;
        });
    }

    /**
     * 执行业务逻辑
     *
     * @param isDelay 是否延迟执行
     */
    private void execute(boolean isDelay) {
        serviceManager.deleteValidAdvertIdsSortCache();
        logger.info("KC101消息, {}, {}", DateUtils.getMillisecond(), isDelay ? "延迟消费" : "直接消费");
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        //在消息处理器中注册
        RocketMqMessageListener.registerCallback(this);

        // 用于延迟队列消费的单线程线程池
        ExecutorService executorService = Executors.newSingleThreadExecutor();
        executorService.submit((Runnable) () -> {
            while (true) {
                try {
                    MsgDelayTask task = queue.take();
                    List<String> tasks = map.get(task.getKey());
                    if (null != tasks && tasks.size() > 1) {
                        execute(true);
                    }
                    map.remove(task.getKey());
                } catch (Exception e) {
                    logger.error("KC101消息, 延时线程执行异常", e);
                }
            }
        });
    }
}
